namespace Mbrcpval.Core;

/// <summary>
/// Represents the result of validating a JSON message against a schema.
/// </summary>
public sealed class SchemaValidationResult
{
    private readonly List<ValidationError> _errors;
    private readonly List<ValidationError> _warnings;

    /// <summary>
    /// Gets a value indicating whether the validation was successful.
    /// </summary>
    public bool IsValid => _errors.Count == 0;

    /// <summary>
    /// Gets the collection of validation errors.
    /// </summary>
    public IReadOnlyList<ValidationError> Errors => _errors.AsReadOnly();

    /// <summary>
    /// Gets the collection of validation warnings (non-fatal issues).
    /// </summary>
    public IReadOnlyList<ValidationError> Warnings => _warnings.AsReadOnly();

    /// <summary>
    /// Gets the name of the schema used for validation.
    /// </summary>
    public string SchemaName { get; }

    /// <summary>
    /// Initializes a new instance of the <see cref="SchemaValidationResult"/> class.
    /// </summary>
    /// <param name="schemaName">The name of the schema used for validation.</param>
    /// <param name="errors">The collection of validation errors.</param>
    /// <param name="warnings">The collection of validation warnings.</param>
    private SchemaValidationResult(string schemaName, IEnumerable<ValidationError>? errors = null, IEnumerable<ValidationError>? warnings = null)
    {
        SchemaName = schemaName ?? string.Empty;
        _errors = errors?.ToList() ?? new List<ValidationError>();
        _warnings = warnings?.ToList() ?? new List<ValidationError>();
    }

    /// <summary>
    /// Creates a successful validation result with no errors.
    /// </summary>
    /// <param name="schemaName">The name of the schema used for validation.</param>
    /// <returns>A successful validation result.</returns>
    public static SchemaValidationResult Success(string schemaName = "")
        => new(schemaName);

    /// <summary>
    /// Creates a failed validation result with the specified errors.
    /// </summary>
    /// <param name="errors">The validation errors.</param>
    /// <param name="schemaName">The name of the schema used for validation.</param>
    /// <returns>A failed validation result.</returns>
    public static SchemaValidationResult Failure(IEnumerable<ValidationError> errors, string schemaName = "")
        => new(schemaName, errors);

    /// <summary>
    /// Creates a failed validation result with a single error.
    /// </summary>
    /// <param name="error">The validation error.</param>
    /// <param name="schemaName">The name of the schema used for validation.</param>
    /// <returns>A failed validation result.</returns>
    public static SchemaValidationResult Failure(ValidationError error, string schemaName = "")
        => new(schemaName, new[] { error });

    /// <summary>
    /// Creates a validation result with both errors and warnings.
    /// </summary>
    /// <param name="errors">The validation errors.</param>
    /// <param name="warnings">The validation warnings.</param>
    /// <param name="schemaName">The name of the schema used for validation.</param>
    /// <returns>A validation result with errors and warnings.</returns>
    public static SchemaValidationResult WithWarnings(IEnumerable<ValidationError> errors, IEnumerable<ValidationError> warnings, string schemaName = "")
        => new(schemaName, errors, warnings);

    /// <summary>
    /// Returns a summary string of the validation result.
    /// </summary>
    public override string ToString()
    {
        if (IsValid)
        {
            return _warnings.Count > 0
                ? $"Valid (schema: {SchemaName}) with {_warnings.Count} warning(s)"
                : $"Valid (schema: {SchemaName})";
        }

        return $"Invalid (schema: {SchemaName}): {_errors.Count} error(s), {_warnings.Count} warning(s)";
    }
}
