namespace Mbrcpval.Core;

/// <summary>
/// Represents a single validation error with location and description.
/// </summary>
public sealed class ValidationError
{
    /// <summary>
    /// Gets the JSONPath to the location of the error in the document.
    /// </summary>
    public string Path { get; }

    /// <summary>
    /// Gets the human-readable error description.
    /// </summary>
    public string Message { get; }

    /// <summary>
    /// Gets the type of validation error.
    /// </summary>
    public ValidationErrorType ErrorType { get; }

    /// <summary>
    /// Gets the line number where the error occurred, if available.
    /// </summary>
    public int? LineNumber { get; }

    /// <summary>
    /// Initializes a new instance of the <see cref="ValidationError"/> class.
    /// </summary>
    /// <param name="path">The JSONPath to the error location.</param>
    /// <param name="message">The error description.</param>
    /// <param name="errorType">The type of validation error.</param>
    /// <param name="lineNumber">The optional line number where the error occurred.</param>
    public ValidationError(string path, string message, ValidationErrorType errorType, int? lineNumber = null)
    {
        Path = path ?? string.Empty;
        Message = message ?? string.Empty;
        ErrorType = errorType;
        LineNumber = lineNumber;
    }

    /// <summary>
    /// Creates a validation error for a missing required field.
    /// </summary>
    public static ValidationError MissingRequired(string path, string fieldName)
        => new(path, $"Missing required field: '{fieldName}'", ValidationErrorType.MissingRequiredField);

    /// <summary>
    /// Creates a validation error for an invalid type.
    /// </summary>
    public static ValidationError TypeMismatch(string path, string expectedType, string actualType)
        => new(path, $"Expected type '{expectedType}' but got '{actualType}'", ValidationErrorType.InvalidType);

    /// <summary>
    /// Creates a validation error for an invalid value.
    /// </summary>
    public static ValidationError InvalidValue(string path, string message)
        => new(path, message, ValidationErrorType.InvalidValue);

    /// <summary>
    /// Creates a validation error for an invalid format.
    /// </summary>
    public static ValidationError FormatMismatch(string path, string expectedFormat)
        => new(path, $"Value does not match expected format: {expectedFormat}", ValidationErrorType.InvalidFormat);

    /// <summary>
    /// Creates a validation error for an additional property not allowed by the schema.
    /// </summary>
    public static ValidationError AdditionalProperty(string path, string propertyName)
        => new(path, $"Additional property '{propertyName}' is not allowed", ValidationErrorType.AdditionalProperty);

    /// <summary>
    /// Creates a validation error for a schema not found.
    /// </summary>
    public static ValidationError SchemaNotFound(string schemaName)
        => new(string.Empty, $"Schema not found: '{schemaName}'", ValidationErrorType.SchemaNotFound);

    /// <summary>
    /// Creates a validation error for a JSON parse error.
    /// </summary>
    public static ValidationError ParseError(string message, int? lineNumber = null)
        => new(string.Empty, $"JSON parse error: {message}", ValidationErrorType.ParseError, lineNumber);

    /// <summary>
    /// Returns a string representation of the validation error.
    /// </summary>
    public override string ToString()
    {
        var location = string.IsNullOrEmpty(Path) ? "root" : Path;
        var lineInfo = LineNumber.HasValue ? $" (line {LineNumber.Value})" : string.Empty;
        return $"[{ErrorType}] {location}{lineInfo}: {Message}";
    }
}
