using System.Reflection;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using NJsonSchema;

namespace Mbrcpval.Core;

/// <summary>
/// Validates JSON messages against MBRC protocol schemas using NJsonSchema.
/// </summary>
public sealed class SchemaValidator
{
    private readonly SchemaRegistry _registry;
    private readonly Dictionary<string, JsonSchema> _schemaCache = new(StringComparer.OrdinalIgnoreCase);
    private readonly object _cacheLock = new();

    /// <summary>
    /// Initializes a new instance of the <see cref="SchemaValidator"/> class.
    /// </summary>
    /// <param name="registry">The schema registry to use for lookups.</param>
    public SchemaValidator(SchemaRegistry registry)
    {
        _registry = registry ?? throw new ArgumentNullException(nameof(registry));
    }

    /// <summary>
    /// Loads a JSON schema from an embedded resource or file path.
    /// </summary>
    /// <param name="schemaPath">The path to the schema (embedded resource name or file path).</param>
    /// <param name="assembly">The assembly containing the embedded resource. If null, uses the executing assembly.</param>
    /// <returns>The loaded schema, or null if not found.</returns>
    public async Task<JsonSchema?> LoadSchemaAsync(string schemaPath, Assembly? assembly = null)
    {
        if (string.IsNullOrWhiteSpace(schemaPath))
            return null;

        // Check cache first (inside lock to avoid race)
        lock (_cacheLock)
        {
            if (_schemaCache.TryGetValue(schemaPath, out var cached))
                return cached;
        }

        JsonSchema? schema = null;

        // Try loading as file first
        if (File.Exists(schemaPath))
        {
            var json = await File.ReadAllTextAsync(schemaPath).ConfigureAwait(false);
            schema = await JsonSchema.FromJsonAsync(json).ConfigureAwait(false);
        }
        else
        {
            // Try loading as embedded resource
            assembly ??= Assembly.GetExecutingAssembly();
            var resourceName = assembly.GetManifestResourceNames()
                .FirstOrDefault(n => n.EndsWith(schemaPath, StringComparison.OrdinalIgnoreCase));

            if (resourceName != null)
            {
                using var stream = assembly.GetManifestResourceStream(resourceName);
                if (stream != null)
                {
                    using var reader = new StreamReader(stream);
                    var json = await reader.ReadToEndAsync().ConfigureAwait(false);
                    schema = await JsonSchema.FromJsonAsync(json).ConfigureAwait(false);
                }
            }
        }

        if (schema != null)
        {
            // Use TryAdd pattern to handle concurrent loads
            lock (_cacheLock)
            {
                // Another thread may have cached it while we were loading
                if (!_schemaCache.TryGetValue(schemaPath, out _))
                {
                    _schemaCache[schemaPath] = schema;
                }
                else
                {
                    // Use the cached version for consistency
                    schema = _schemaCache[schemaPath];
                }
            }
        }

        return schema;
    }

    /// <summary>
    /// Loads all schemas from embedded resources into a dictionary keyed by context.
    /// </summary>
    /// <param name="assembly">The assembly containing embedded schema resources.</param>
    /// <returns>A dictionary mapping context names to schemas.</returns>
    public async Task<Dictionary<string, JsonSchema>> LoadAllSchemasAsync(Assembly? assembly = null)
    {
        assembly ??= Assembly.GetExecutingAssembly();
        var result = new Dictionary<string, JsonSchema>(StringComparer.OrdinalIgnoreCase);

        var resourceNames = assembly.GetManifestResourceNames()
            .Where(name => name.EndsWith(".schema.json", StringComparison.OrdinalIgnoreCase));

        foreach (var resourceName in resourceNames)
        {
            using var stream = assembly.GetManifestResourceStream(resourceName);
            if (stream == null) continue;

            using var reader = new StreamReader(stream);
            var json = await reader.ReadToEndAsync();
            var schema = await JsonSchema.FromJsonAsync(json);

            // Extract the context name from the resource name
            var contextName = ExtractContextFromResourceName(resourceName);
            result[contextName] = schema;

            lock (_cacheLock)
            {
                _schemaCache[resourceName] = schema;
            }
        }

        return result;
    }

    /// <summary>
    /// Validates an MBRC message object against the appropriate schema.
    /// </summary>
    /// <param name="message">The message to validate (should have 'context' and 'data' properties).</param>
    /// <param name="direction">The message direction to determine which schema to use.</param>
    /// <returns>The validation result.</returns>
    public SchemaValidationResult ValidateMessage(object message, MessageDirection direction)
    {
        if (message == null)
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError("Message cannot be null"),
                "unknown");
        }

        try
        {
            var json = JsonConvert.SerializeObject(message);
            return ValidateMessageJson(json, direction);
        }
        catch (JsonException ex)
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError(ex.Message),
                "unknown");
        }
    }

    /// <summary>
    /// Validates a JSON string message against the appropriate schema based on context.
    /// </summary>
    /// <param name="json">The JSON message string.</param>
    /// <param name="direction">The message direction.</param>
    /// <returns>The validation result.</returns>
    public SchemaValidationResult ValidateMessageJson(string json, MessageDirection direction)
    {
        if (string.IsNullOrWhiteSpace(json))
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError("JSON cannot be empty"),
                "unknown");
        }

        JObject parsed;
        try
        {
            parsed = JObject.Parse(json);
        }
        catch (JsonReaderException ex)
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError(ex.Message, ex.LineNumber),
                "unknown");
        }

        var context = parsed["context"]?.Value<string>();
        if (string.IsNullOrEmpty(context))
        {
            return SchemaValidationResult.Failure(
                ValidationError.MissingRequired("$", "context"),
                "unknown");
        }

        var schema = GetSchemaForContext(context, direction);
        if (schema == null)
        {
            return SchemaValidationResult.Failure(
                ValidationError.SchemaNotFound($"{context} ({direction})"),
                $"{context}.{direction}");
        }

        return ValidateJsonAgainstSchema(json, schema, $"{context}.{direction}");
    }

    /// <summary>
    /// Validates raw JSON against a named schema.
    /// </summary>
    /// <param name="json">The JSON string to validate.</param>
    /// <param name="schemaName">The name of the schema to validate against.</param>
    /// <returns>The validation result.</returns>
    public async Task<SchemaValidationResult> ValidateJsonAsync(string json, string schemaName)
    {
        if (string.IsNullOrWhiteSpace(json))
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError("JSON cannot be empty"),
                schemaName);
        }

        var schema = await LoadSchemaAsync(schemaName);
        if (schema == null)
        {
            return SchemaValidationResult.Failure(
                ValidationError.SchemaNotFound(schemaName),
                schemaName);
        }

        return ValidateJsonAgainstSchema(json, schema, schemaName);
    }

    /// <summary>
    /// Validates raw JSON against a named schema (synchronous version).
    /// Uses cached schema if available to avoid async operations.
    /// </summary>
    public SchemaValidationResult ValidateJson(string json, string schemaName)
    {
        if (string.IsNullOrWhiteSpace(json))
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError("JSON cannot be empty"),
                schemaName);
        }

        // Try to get from cache first to avoid async call
        JsonSchema? schema;
        lock (_cacheLock)
        {
            _schemaCache.TryGetValue(schemaName, out schema);
        }

        if (schema == null)
        {
            // Fallback to async load - use ConfigureAwait(false) to avoid deadlock
            schema = LoadSchemaAsync(schemaName).ConfigureAwait(false).GetAwaiter().GetResult();
        }

        if (schema == null)
        {
            return SchemaValidationResult.Failure(
                ValidationError.SchemaNotFound(schemaName),
                schemaName);
        }

        return ValidateJsonAgainstSchema(json, schema, schemaName);
    }

    /// <summary>
    /// Gets the appropriate schema based on context and direction.
    /// </summary>
    /// <param name="context">The message context (e.g., "player", "playerstatus").</param>
    /// <param name="direction">The message direction.</param>
    /// <returns>The schema if found, null otherwise.</returns>
    public JsonSchema? GetSchemaForContext(string context, MessageDirection direction)
    {
        return _registry.GetSchema(context, direction);
    }

    /// <summary>
    /// Validates JSON against a pre-loaded schema.
    /// </summary>
    private SchemaValidationResult ValidateJsonAgainstSchema(string json, JsonSchema schema, string schemaName)
    {
        try
        {
            var errors = schema.Validate(json);

            if (errors.Count == 0)
            {
                return SchemaValidationResult.Success(schemaName);
            }

            var validationErrors = errors.Select(e => ConvertToValidationError(e)).ToList();
            return SchemaValidationResult.Failure(validationErrors, schemaName);
        }
        catch (JsonReaderException ex)
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError(ex.Message, ex.LineNumber),
                schemaName);
        }
        catch (Exception ex)
        {
            return SchemaValidationResult.Failure(
                ValidationError.ParseError($"Unexpected error during validation: {ex.Message}"),
                schemaName);
        }
    }

    /// <summary>
    /// Converts an NJsonSchema validation error to our ValidationError type.
    /// </summary>
    private static ValidationError ConvertToValidationError(NJsonSchema.Validation.ValidationError error)
    {
        var path = error.Path ?? "#";
        var message = error.ToString();
        var errorType = DetermineErrorType(error);
        int? lineNumber = error.LineNumber > 0 ? error.LineNumber : null;

        return new ValidationError(path, message, errorType, lineNumber);
    }

    /// <summary>
    /// Determines the ValidationErrorType based on the NJsonSchema error kind.
    /// </summary>
    private static ValidationErrorType DetermineErrorType(NJsonSchema.Validation.ValidationError error)
    {
        var kind = error.Kind;

        return kind switch
        {
            NJsonSchema.Validation.ValidationErrorKind.PropertyRequired => ValidationErrorType.MissingRequiredField,
            NJsonSchema.Validation.ValidationErrorKind.StringExpected or
            NJsonSchema.Validation.ValidationErrorKind.NumberExpected or
            NJsonSchema.Validation.ValidationErrorKind.BooleanExpected or
            NJsonSchema.Validation.ValidationErrorKind.ArrayExpected or
            NJsonSchema.Validation.ValidationErrorKind.ObjectExpected or
            NJsonSchema.Validation.ValidationErrorKind.NullExpected or
            NJsonSchema.Validation.ValidationErrorKind.IntegerExpected => ValidationErrorType.InvalidType,
            NJsonSchema.Validation.ValidationErrorKind.NotInEnumeration => ValidationErrorType.InvalidValue,
            NJsonSchema.Validation.ValidationErrorKind.PatternMismatch or
            NJsonSchema.Validation.ValidationErrorKind.DateExpected or
            NJsonSchema.Validation.ValidationErrorKind.DateTimeExpected or
            NJsonSchema.Validation.ValidationErrorKind.TimeExpected or
            NJsonSchema.Validation.ValidationErrorKind.TimeSpanExpected or
            NJsonSchema.Validation.ValidationErrorKind.UriExpected or
            NJsonSchema.Validation.ValidationErrorKind.EmailExpected or
            NJsonSchema.Validation.ValidationErrorKind.IpV4Expected or
            NJsonSchema.Validation.ValidationErrorKind.IpV6Expected or
            NJsonSchema.Validation.ValidationErrorKind.GuidExpected or
            NJsonSchema.Validation.ValidationErrorKind.HostnameExpected => ValidationErrorType.InvalidFormat,
            NJsonSchema.Validation.ValidationErrorKind.NoAdditionalPropertiesAllowed => ValidationErrorType.AdditionalProperty,
            _ => ValidationErrorType.InvalidValue
        };
    }

    /// <summary>
    /// Extracts the context name from an embedded resource name.
    /// </summary>
    private static string ExtractContextFromResourceName(string resourceName)
    {
        // Example: Mbrcpval.schemas.requests.player.schema.json -> player
        var parts = resourceName.Split('.');
        if (parts.Length >= 3)
        {
            var nameIndex = Array.FindIndex(parts, p =>
                p.Equals("schema", StringComparison.OrdinalIgnoreCase));

            if (nameIndex > 0)
            {
                return parts[nameIndex - 1];
            }
        }

        return resourceName;
    }

    /// <summary>
    /// Creates a SchemaValidator with a fully loaded registry.
    /// </summary>
    /// <param name="assembly">The assembly containing embedded schema resources.</param>
    /// <returns>A ready-to-use SchemaValidator.</returns>
    public static async Task<SchemaValidator> CreateAsync(Assembly? assembly = null)
    {
        var registry = await SchemaRegistry.CreateAndLoadAsync(assembly);
        return new SchemaValidator(registry);
    }
}
