using System.Reflection;
using NJsonSchema;

namespace Mbrcpval.Core;

/// <summary>
/// Registry for managing and retrieving JSON schemas by context and direction.
/// </summary>
public sealed class SchemaRegistry
{
    private readonly Dictionary<(string Context, MessageDirection Direction), JsonSchema> _schemas = new();
    private readonly Dictionary<string, JsonSchema> _commonSchemas = new(StringComparer.OrdinalIgnoreCase);
    private readonly object _lock = new();
    private bool _isLoaded;

    /// <summary>
    /// Gets the number of registered schemas.
    /// </summary>
    public int Count
    {
        get
        {
            lock (_lock)
            {
                return _schemas.Count;
            }
        }
    }

    /// <summary>
    /// Gets the number of common schemas.
    /// </summary>
    public int CommonSchemaCount
    {
        get
        {
            lock (_lock)
            {
                return _commonSchemas.Count;
            }
        }
    }

    /// <summary>
    /// Registers a schema for a specific context and direction.
    /// </summary>
    /// <param name="context">The message context (e.g., "player", "playerstatus").</param>
    /// <param name="direction">The message direction.</param>
    /// <param name="schema">The JSON schema.</param>
    public void RegisterSchema(string context, MessageDirection direction, JsonSchema schema)
    {
        ArgumentNullException.ThrowIfNull(context);
        ArgumentNullException.ThrowIfNull(schema);

        lock (_lock)
        {
            var key = (context.ToLowerInvariant(), direction);
            _schemas[key] = schema;
        }
    }

    /// <summary>
    /// Registers a common schema that can be referenced by other schemas.
    /// </summary>
    /// <param name="name">The schema name (e.g., "message", "track").</param>
    /// <param name="schema">The JSON schema.</param>
    public void RegisterCommonSchema(string name, JsonSchema schema)
    {
        ArgumentNullException.ThrowIfNull(name);
        ArgumentNullException.ThrowIfNull(schema);

        lock (_lock)
        {
            _commonSchemas[name.ToLowerInvariant()] = schema;
        }
    }

    /// <summary>
    /// Gets a schema for the specified context and direction.
    /// </summary>
    /// <param name="context">The message context.</param>
    /// <param name="direction">The message direction.</param>
    /// <returns>The schema if found, null otherwise.</returns>
    public JsonSchema? GetSchema(string context, MessageDirection direction)
    {
        if (string.IsNullOrEmpty(context))
            return null;

        lock (_lock)
        {
            var key = (context.ToLowerInvariant(), direction);
            return _schemas.TryGetValue(key, out var schema) ? schema : null;
        }
    }

    /// <summary>
    /// Gets a common schema by name.
    /// </summary>
    /// <param name="name">The schema name.</param>
    /// <returns>The schema if found, null otherwise.</returns>
    public JsonSchema? GetCommonSchema(string name)
    {
        if (string.IsNullOrEmpty(name))
            return null;

        lock (_lock)
        {
            return _commonSchemas.TryGetValue(name.ToLowerInvariant(), out var schema) ? schema : null;
        }
    }

    /// <summary>
    /// Checks if a schema exists for the specified context and direction.
    /// </summary>
    public bool HasSchema(string context, MessageDirection direction)
    {
        if (string.IsNullOrEmpty(context))
            return false;

        lock (_lock)
        {
            var key = (context.ToLowerInvariant(), direction);
            return _schemas.ContainsKey(key);
        }
    }

    /// <summary>
    /// Gets all registered context-direction pairs.
    /// </summary>
    public IReadOnlyList<(string Context, MessageDirection Direction)> GetAllSchemaKeys()
    {
        lock (_lock)
        {
            return _schemas.Keys.ToList().AsReadOnly();
        }
    }

    /// <summary>
    /// Loads all schemas from embedded resources in the specified assembly.
    /// </summary>
    /// <param name="assembly">The assembly containing embedded schema resources. If null, uses the calling assembly.</param>
    public async Task LoadFromEmbeddedResourcesAsync(Assembly? assembly = null)
    {
        assembly ??= Assembly.GetExecutingAssembly();

        lock (_lock)
        {
            if (_isLoaded)
                return;
        }

        var resourceNames = assembly.GetManifestResourceNames()
            .Where(name => name.EndsWith(".schema.json", StringComparison.OrdinalIgnoreCase))
            .ToList();

        // First pass: load common schemas
        foreach (var resourceName in resourceNames.Where(n => n.Contains(".schemas.common.", StringComparison.OrdinalIgnoreCase)))
        {
            await LoadSchemaFromResourceAsync(assembly, resourceName);
        }

        // Second pass: load all other schemas
        foreach (var resourceName in resourceNames.Where(n => !n.Contains(".schemas.common.", StringComparison.OrdinalIgnoreCase)))
        {
            await LoadSchemaFromResourceAsync(assembly, resourceName);
        }

        lock (_lock)
        {
            _isLoaded = true;
        }
    }

    private async Task LoadSchemaFromResourceAsync(Assembly assembly, string resourceName)
    {
        try
        {
            using var stream = assembly.GetManifestResourceStream(resourceName);
            if (stream == null)
            {
                Console.Error.WriteLine($"Warning: Could not load resource '{resourceName}'");
                return;
            }

            using var reader = new StreamReader(stream);
            var schemaJson = await reader.ReadToEndAsync();
            var schema = await JsonSchema.FromJsonAsync(schemaJson);

            var (context, direction) = ParseResourceName(resourceName);

            if (direction == null)
            {
                // It's a common schema
                RegisterCommonSchema(context, schema);
            }
            else
            {
                RegisterSchema(context, direction.Value, schema);
            }
        }
        catch (Exception ex)
        {
            Console.Error.WriteLine($"Warning: Failed to load schema from '{resourceName}': {ex.Message}");
        }
    }

    /// <summary>
    /// Parses a resource name to extract context and direction.
    /// </summary>
    /// <remarks>
    /// Expected resource name patterns:
    /// - Mbrcpval.schemas.common.message.schema.json -> ("message", null)
    /// - Mbrcpval.schemas.requests.player.schema.json -> ("player", Request)
    /// - Mbrcpval.schemas.responses.playerstatus.schema.json -> ("playerstatus", Response)
    /// - Mbrcpval.schemas.events.playerstate.schema.json -> ("playerstate", Event)
    /// - Mbrcpval.schemas.discovery.discovery-request.schema.json -> ("discovery-request", Discovery)
    /// </remarks>
    private static (string Context, MessageDirection? Direction) ParseResourceName(string resourceName)
    {
        // Remove common prefixes and suffixes
        var name = resourceName
            .Replace(".schema.json", string.Empty, StringComparison.OrdinalIgnoreCase);

        // Split by dots to find the folder and schema name
        var parts = name.Split('.');

        // Find the schemas folder index
        var schemasIndex = Array.FindIndex(parts, p => p.Equals("schemas", StringComparison.OrdinalIgnoreCase));
        if (schemasIndex < 0 || schemasIndex + 2 >= parts.Length)
        {
            // Fallback: use the last part as context
            return (parts[^1], null);
        }

        var folder = parts[schemasIndex + 1].ToLowerInvariant();
        var schemaName = parts[schemasIndex + 2].ToLowerInvariant();

        // Handle response schemas that have "-response" suffix
        if (schemaName.EndsWith("-response"))
        {
            schemaName = schemaName[..^9]; // Remove "-response" suffix
        }

        var direction = folder switch
        {
            "requests" => MessageDirection.Request,
            "responses" => MessageDirection.Response,
            "events" => MessageDirection.Event,
            "discovery" => MessageDirection.Discovery,
            "common" => (MessageDirection?)null,
            _ => null
        };

        return (schemaName, direction);
    }

    /// <summary>
    /// Creates a new registry and loads all schemas from embedded resources.
    /// </summary>
    public static async Task<SchemaRegistry> CreateAndLoadAsync(Assembly? assembly = null)
    {
        var registry = new SchemaRegistry();
        await registry.LoadFromEmbeddedResourcesAsync(assembly);
        return registry;
    }
}
