using System.Collections.Concurrent;
using System.Diagnostics;
using System.Net.Sockets;

namespace Mbrcpval.Core;

/// <summary>
/// Circuit breaker states for connection resilience.
/// </summary>
public enum CircuitState
{
    /// <summary>Circuit is closed (normal operation).</summary>
    Closed,
    /// <summary>Circuit is open (failing fast).</summary>
    Open,
    /// <summary>Circuit is half-open (testing recovery).</summary>
    HalfOpen
}

/// <summary>
/// Circuit breaker pattern implementation for connection resilience.
/// Prevents cascade failures by failing fast when a threshold is exceeded.
/// </summary>
public class CircuitBreaker
{
    private readonly object _lock = new();
    private readonly int _failureThreshold;
    private readonly TimeSpan _openDuration;
    private readonly TimeSpan _halfOpenTestInterval;

    private CircuitState _state = CircuitState.Closed;
    private int _failureCount;
    private DateTime _lastFailure;
    private DateTime _openedAt;
    private int _successCount;
    private int _halfOpenSuccessThreshold;

    /// <summary>
    /// Gets the current circuit state.
    /// </summary>
    public CircuitState State
    {
        get { lock (_lock) return _state; }
    }

    /// <summary>
    /// Gets the current failure count.
    /// </summary>
    public int FailureCount
    {
        get { lock (_lock) return _failureCount; }
    }

    /// <summary>
    /// Event raised when circuit state changes.
    /// </summary>
    public event EventHandler<CircuitStateChangedEventArgs>? StateChanged;

    /// <summary>
    /// Creates a new circuit breaker.
    /// </summary>
    /// <param name="failureThreshold">Number of failures before opening circuit.</param>
    /// <param name="openDuration">Duration to keep circuit open before testing.</param>
    /// <param name="halfOpenSuccessThreshold">Successes needed to close circuit from half-open.</param>
    public CircuitBreaker(
        int failureThreshold = 5,
        TimeSpan? openDuration = null,
        int halfOpenSuccessThreshold = 2)
    {
        _failureThreshold = failureThreshold;
        _openDuration = openDuration ?? TimeSpan.FromSeconds(30);
        _halfOpenTestInterval = TimeSpan.FromSeconds(5);
        _halfOpenSuccessThreshold = halfOpenSuccessThreshold;
    }

    /// <summary>
    /// Executes an operation through the circuit breaker.
    /// </summary>
    public async Task<T> ExecuteAsync<T>(Func<CancellationToken, Task<T>> operation, CancellationToken cancellationToken = default)
    {
        EnsureCircuitAllowsExecution();

        try
        {
            var result = await operation(cancellationToken);
            RecordSuccess();
            return result;
        }
        catch (Exception ex) when (IsTransientFailure(ex))
        {
            RecordFailure();
            throw;
        }
    }

    /// <summary>
    /// Executes an operation through the circuit breaker.
    /// </summary>
    public async Task ExecuteAsync(Func<CancellationToken, Task> operation, CancellationToken cancellationToken = default)
    {
        EnsureCircuitAllowsExecution();

        try
        {
            await operation(cancellationToken);
            RecordSuccess();
        }
        catch (Exception ex) when (IsTransientFailure(ex))
        {
            RecordFailure();
            throw;
        }
    }

    private void EnsureCircuitAllowsExecution()
    {
        lock (_lock)
        {
            switch (_state)
            {
                case CircuitState.Open:
                    if (DateTime.UtcNow - _openedAt >= _openDuration)
                    {
                        TransitionTo(CircuitState.HalfOpen);
                    }
                    else
                    {
                        throw new CircuitBreakerOpenException(
                            $"Circuit breaker is open. Retry after {_openDuration - (DateTime.UtcNow - _openedAt)}");
                    }
                    break;

                case CircuitState.HalfOpen:
                    // Allow limited testing
                    break;

                case CircuitState.Closed:
                    // Normal operation
                    break;
            }
        }
    }

    private void RecordSuccess()
    {
        lock (_lock)
        {
            _failureCount = 0;

            if (_state == CircuitState.HalfOpen)
            {
                _successCount++;
                if (_successCount >= _halfOpenSuccessThreshold)
                {
                    TransitionTo(CircuitState.Closed);
                    _successCount = 0;
                }
            }
        }
    }

    private void RecordFailure()
    {
        lock (_lock)
        {
            _failureCount++;
            _lastFailure = DateTime.UtcNow;

            if (_state == CircuitState.HalfOpen)
            {
                TransitionTo(CircuitState.Open);
                _successCount = 0;
            }
            else if (_state == CircuitState.Closed && _failureCount >= _failureThreshold)
            {
                TransitionTo(CircuitState.Open);
            }
        }
    }

    private void TransitionTo(CircuitState newState)
    {
        var oldState = _state;
        _state = newState;

        if (newState == CircuitState.Open)
        {
            _openedAt = DateTime.UtcNow;
        }

        // Fire event outside lock to prevent deadlocks
        Task.Run(() =>
        {
            try
            {
                StateChanged?.Invoke(this, new CircuitStateChangedEventArgs(oldState, newState));
            }
            catch
            {
                // Ignore event handler exceptions
            }
        });
    }

    private static bool IsTransientFailure(Exception ex)
    {
        return ex is IOException or SocketException or TimeoutException or OperationCanceledException;
    }

    /// <summary>
    /// Manually resets the circuit breaker to closed state.
    /// </summary>
    public void Reset()
    {
        lock (_lock)
        {
            _failureCount = 0;
            _successCount = 0;
            TransitionTo(CircuitState.Closed);
        }
    }
}

/// <summary>
/// Event args for circuit state changes.
/// </summary>
public class CircuitStateChangedEventArgs : EventArgs
{
    public CircuitState OldState { get; }
    public CircuitState NewState { get; }

    public CircuitStateChangedEventArgs(CircuitState oldState, CircuitState newState)
    {
        OldState = oldState;
        NewState = newState;
    }
}

/// <summary>
/// Exception thrown when circuit breaker is open.
/// </summary>
public class CircuitBreakerOpenException : Exception
{
    public CircuitBreakerOpenException(string message) : base(message) { }
}

/// <summary>
/// Retry policy with exponential backoff and jitter.
/// </summary>
public class RetryPolicy
{
    private readonly int _maxRetries;
    private readonly TimeSpan _initialDelay;
    private readonly TimeSpan _maxDelay;
    private readonly double _backoffMultiplier;
    private readonly Random _jitter = new();

    /// <summary>
    /// Creates a new retry policy.
    /// </summary>
    /// <param name="maxRetries">Maximum number of retry attempts.</param>
    /// <param name="initialDelay">Initial delay between retries.</param>
    /// <param name="maxDelay">Maximum delay between retries.</param>
    /// <param name="backoffMultiplier">Multiplier for exponential backoff.</param>
    public RetryPolicy(
        int maxRetries = 3,
        TimeSpan? initialDelay = null,
        TimeSpan? maxDelay = null,
        double backoffMultiplier = 2.0)
    {
        _maxRetries = maxRetries;
        _initialDelay = initialDelay ?? TimeSpan.FromMilliseconds(100);
        _maxDelay = maxDelay ?? TimeSpan.FromSeconds(30);
        _backoffMultiplier = backoffMultiplier;
    }

    /// <summary>
    /// Executes an operation with retry logic.
    /// </summary>
    public async Task<T> ExecuteAsync<T>(
        Func<CancellationToken, Task<T>> operation,
        Func<Exception, bool>? shouldRetry = null,
        CancellationToken cancellationToken = default)
    {
        shouldRetry ??= IsTransientException;
        var delay = _initialDelay;

        for (int attempt = 0; attempt <= _maxRetries; attempt++)
        {
            try
            {
                return await operation(cancellationToken);
            }
            catch (Exception ex) when (attempt < _maxRetries && shouldRetry(ex))
            {
                // Add jitter to prevent thundering herd
                var jitteredDelay = AddJitter(delay);
                await Task.Delay(jitteredDelay, cancellationToken);

                // Exponential backoff
                delay = TimeSpan.FromTicks(Math.Min(
                    (long)(delay.Ticks * _backoffMultiplier),
                    _maxDelay.Ticks));
            }
        }

        // This shouldn't be reached, but compiler needs it
        throw new InvalidOperationException("Retry loop exited unexpectedly");
    }

    /// <summary>
    /// Executes an operation with retry logic.
    /// </summary>
    public async Task ExecuteAsync(
        Func<CancellationToken, Task> operation,
        Func<Exception, bool>? shouldRetry = null,
        CancellationToken cancellationToken = default)
    {
        await ExecuteAsync(async ct =>
        {
            await operation(ct);
            return true;
        }, shouldRetry, cancellationToken);
    }

    private TimeSpan AddJitter(TimeSpan delay)
    {
        // Add up to 25% jitter
        var jitterFactor = 0.75 + (_jitter.NextDouble() * 0.5);
        return TimeSpan.FromTicks((long)(delay.Ticks * jitterFactor));
    }

    private static bool IsTransientException(Exception ex)
    {
        return ex is IOException or SocketException or TimeoutException;
    }
}

/// <summary>
/// Message recorder for replay and debugging.
/// Implements time-travel debugging by recording all messages.
/// </summary>
public class MessageRecorder
{
    private readonly ConcurrentQueue<RecordedMessage> _messages = new();
    private readonly int _maxMessages;
    private long _sequenceNumber;

    public MessageRecorder(int maxMessages = 10000)
    {
        _maxMessages = maxMessages;
    }

    /// <summary>
    /// Records a sent message.
    /// </summary>
    public void RecordSent(string message, string? context = null)
    {
        Record(RecordedMessageDirection.Sent, message, context);
    }

    /// <summary>
    /// Records a received message.
    /// </summary>
    public void RecordReceived(string message, string? context = null)
    {
        Record(RecordedMessageDirection.Received, message, context);
    }

    private void Record(RecordedMessageDirection direction, string message, string? context)
    {
        var recorded = new RecordedMessage
        {
            SequenceNumber = Interlocked.Increment(ref _sequenceNumber),
            Timestamp = DateTime.UtcNow,
            Direction = direction,
            Message = message,
            Context = context
        };

        _messages.Enqueue(recorded);

        // Trim if over limit
        while (_messages.Count > _maxMessages && _messages.TryDequeue(out _))
        {
            // Discard oldest
        }
    }

    /// <summary>
    /// Gets all recorded messages.
    /// </summary>
    public IReadOnlyList<RecordedMessage> GetMessages()
    {
        return _messages.ToList();
    }

    /// <summary>
    /// Gets messages in a time range.
    /// </summary>
    public IReadOnlyList<RecordedMessage> GetMessages(DateTime from, DateTime to)
    {
        return _messages.Where(m => m.Timestamp >= from && m.Timestamp <= to).ToList();
    }

    /// <summary>
    /// Exports messages to JSON for analysis.
    /// </summary>
    public string ExportToJson()
    {
        return System.Text.Json.JsonSerializer.Serialize(_messages.ToList(),
            new System.Text.Json.JsonSerializerOptions { WriteIndented = true });
    }

    /// <summary>
    /// Clears all recorded messages.
    /// </summary>
    public void Clear()
    {
        while (_messages.TryDequeue(out _)) { }
        Interlocked.Exchange(ref _sequenceNumber, 0);
    }
}

/// <summary>
/// Recorded message for replay.
/// </summary>
public class RecordedMessage
{
    public long SequenceNumber { get; init; }
    public DateTime Timestamp { get; init; }
    public RecordedMessageDirection Direction { get; init; }
    public string Message { get; init; } = string.Empty;
    public string? Context { get; init; }
}

/// <summary>
/// Direction for recorded messages.
/// </summary>
public enum RecordedMessageDirection
{
    Sent,
    Received
}

/// <summary>
/// Health monitor for connection liveness.
/// Implements heartbeat-based connection health monitoring.
/// </summary>
public class ConnectionHealthMonitor : IDisposable
{
    private readonly Func<CancellationToken, Task<bool>> _healthCheck;
    private readonly TimeSpan _checkInterval;
    private readonly TimeSpan _timeout;
    private readonly int _unhealthyThreshold;

    private CancellationTokenSource? _cts;
    private Task? _monitorTask;
    private volatile bool _isHealthy = true;
    private volatile int _consecutiveFailures;
    private volatile bool _disposed;

    /// <summary>
    /// Gets whether the connection is currently healthy.
    /// </summary>
    public bool IsHealthy => _isHealthy;

    /// <summary>
    /// Gets the number of consecutive health check failures.
    /// </summary>
    public int ConsecutiveFailures => _consecutiveFailures;

    /// <summary>
    /// Event raised when health status changes.
    /// </summary>
    public event EventHandler<HealthChangedEventArgs>? HealthChanged;

    /// <summary>
    /// Creates a new connection health monitor.
    /// </summary>
    /// <param name="healthCheck">Function to check connection health (e.g., send ping).</param>
    /// <param name="checkInterval">Interval between health checks.</param>
    /// <param name="timeout">Timeout for each health check.</param>
    /// <param name="unhealthyThreshold">Consecutive failures before marking unhealthy.</param>
    public ConnectionHealthMonitor(
        Func<CancellationToken, Task<bool>> healthCheck,
        TimeSpan? checkInterval = null,
        TimeSpan? timeout = null,
        int unhealthyThreshold = 3)
    {
        _healthCheck = healthCheck ?? throw new ArgumentNullException(nameof(healthCheck));
        _checkInterval = checkInterval ?? TimeSpan.FromSeconds(10);
        _timeout = timeout ?? TimeSpan.FromSeconds(5);
        _unhealthyThreshold = unhealthyThreshold;
    }

    /// <summary>
    /// Starts the health monitoring.
    /// </summary>
    public void Start()
    {
        if (_disposed) throw new ObjectDisposedException(nameof(ConnectionHealthMonitor));
        if (_monitorTask != null) return;

        _cts = new CancellationTokenSource();
        _monitorTask = MonitorLoopAsync(_cts.Token);
    }

    /// <summary>
    /// Stops the health monitoring.
    /// </summary>
    public async Task StopAsync()
    {
        if (_cts == null) return;

        await _cts.CancelAsync();

        if (_monitorTask != null)
        {
            try
            {
                await _monitorTask.WaitAsync(TimeSpan.FromSeconds(5));
            }
            catch (OperationCanceledException)
            {
                // Expected
            }
            catch (TimeoutException)
            {
                // Monitor didn't stop in time
            }
        }

        _cts.Dispose();
        _cts = null;
        _monitorTask = null;
    }

    private async Task MonitorLoopAsync(CancellationToken cancellationToken)
    {
        while (!cancellationToken.IsCancellationRequested)
        {
            try
            {
                await Task.Delay(_checkInterval, cancellationToken);

                using var timeoutCts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                timeoutCts.CancelAfter(_timeout);

                bool healthy;
                try
                {
                    healthy = await _healthCheck(timeoutCts.Token);
                }
                catch (OperationCanceledException) when (!cancellationToken.IsCancellationRequested)
                {
                    // Timeout
                    healthy = false;
                }
                catch
                {
                    healthy = false;
                }

                if (healthy)
                {
                    var wasUnhealthy = !_isHealthy;
                    Interlocked.Exchange(ref _consecutiveFailures, 0);
                    _isHealthy = true;

                    if (wasUnhealthy)
                    {
                        OnHealthChanged(true);
                    }
                }
                else
                {
                    var failures = Interlocked.Increment(ref _consecutiveFailures);
                    if (failures >= _unhealthyThreshold && _isHealthy)
                    {
                        _isHealthy = false;
                        OnHealthChanged(false);
                    }
                }
            }
            catch (OperationCanceledException)
            {
                break;
            }
        }
    }

    private void OnHealthChanged(bool isHealthy)
    {
        Task.Run(() =>
        {
            try
            {
                HealthChanged?.Invoke(this, new HealthChangedEventArgs(isHealthy, _consecutiveFailures));
            }
            catch
            {
                // Ignore event handler exceptions
            }
        });
    }

    public void Dispose()
    {
        if (_disposed) return;
        _disposed = true;

        _cts?.Cancel();
        _cts?.Dispose();
        GC.SuppressFinalize(this);
    }
}

/// <summary>
/// Event args for health status changes.
/// </summary>
public class HealthChangedEventArgs : EventArgs
{
    public bool IsHealthy { get; }
    public int ConsecutiveFailures { get; }

    public HealthChangedEventArgs(bool isHealthy, int consecutiveFailures)
    {
        IsHealthy = isHealthy;
        ConsecutiveFailures = consecutiveFailures;
    }
}

/// <summary>
/// Provides structured operation metrics for observability.
/// </summary>
public class OperationMetrics
{
    private long _totalOperations;
    private long _successfulOperations;
    private long _failedOperations;
    private long _totalDurationTicks;
    private readonly ConcurrentDictionary<string, long> _operationCounts = new();
    private readonly ConcurrentDictionary<string, long> _errorCounts = new();

    /// <summary>
    /// Records a successful operation.
    /// </summary>
    public void RecordSuccess(string operationType, TimeSpan duration)
    {
        Interlocked.Increment(ref _totalOperations);
        Interlocked.Increment(ref _successfulOperations);
        Interlocked.Add(ref _totalDurationTicks, duration.Ticks);
        _operationCounts.AddOrUpdate(operationType, 1, (_, count) => count + 1);
    }

    /// <summary>
    /// Records a failed operation.
    /// </summary>
    public void RecordFailure(string operationType, string errorType, TimeSpan duration)
    {
        Interlocked.Increment(ref _totalOperations);
        Interlocked.Increment(ref _failedOperations);
        Interlocked.Add(ref _totalDurationTicks, duration.Ticks);
        _operationCounts.AddOrUpdate(operationType, 1, (_, count) => count + 1);
        _errorCounts.AddOrUpdate(errorType, 1, (_, count) => count + 1);
    }

    /// <summary>
    /// Gets a snapshot of current metrics.
    /// </summary>
    public MetricsSnapshot GetSnapshot()
    {
        var total = Interlocked.Read(ref _totalOperations);
        var successful = Interlocked.Read(ref _successfulOperations);
        var failed = Interlocked.Read(ref _failedOperations);
        var durationTicks = Interlocked.Read(ref _totalDurationTicks);

        return new MetricsSnapshot
        {
            TotalOperations = total,
            SuccessfulOperations = successful,
            FailedOperations = failed,
            SuccessRate = total > 0 ? (double)successful / total : 1.0,
            AverageDuration = total > 0 ? TimeSpan.FromTicks(durationTicks / total) : TimeSpan.Zero,
            OperationCounts = new Dictionary<string, long>(_operationCounts),
            ErrorCounts = new Dictionary<string, long>(_errorCounts)
        };
    }

    /// <summary>
    /// Resets all metrics.
    /// </summary>
    public void Reset()
    {
        Interlocked.Exchange(ref _totalOperations, 0);
        Interlocked.Exchange(ref _successfulOperations, 0);
        Interlocked.Exchange(ref _failedOperations, 0);
        Interlocked.Exchange(ref _totalDurationTicks, 0);
        _operationCounts.Clear();
        _errorCounts.Clear();
    }
}

/// <summary>
/// Snapshot of operation metrics.
/// </summary>
public class MetricsSnapshot
{
    public long TotalOperations { get; init; }
    public long SuccessfulOperations { get; init; }
    public long FailedOperations { get; init; }
    public double SuccessRate { get; init; }
    public TimeSpan AverageDuration { get; init; }
    public Dictionary<string, long> OperationCounts { get; init; } = new();
    public Dictionary<string, long> ErrorCounts { get; init; } = new();
}

/// <summary>
/// Rate limiter to prevent overwhelming the server.
/// Implements token bucket algorithm.
/// </summary>
public class RateLimiter
{
    private readonly object _lock = new();
    private readonly int _maxTokens;
    private readonly double _tokensPerSecond;
    private double _tokens;
    private DateTime _lastRefill;

    /// <summary>
    /// Creates a new rate limiter.
    /// </summary>
    /// <param name="maxTokens">Maximum tokens in the bucket.</param>
    /// <param name="tokensPerSecond">Token refill rate per second.</param>
    public RateLimiter(int maxTokens = 10, double tokensPerSecond = 10)
    {
        _maxTokens = maxTokens;
        _tokensPerSecond = tokensPerSecond;
        _tokens = maxTokens;
        _lastRefill = DateTime.UtcNow;
    }

    /// <summary>
    /// Acquires a token, waiting if necessary.
    /// </summary>
    public async Task AcquireAsync(CancellationToken cancellationToken = default)
    {
        while (!cancellationToken.IsCancellationRequested)
        {
            lock (_lock)
            {
                RefillTokens();

                if (_tokens >= 1)
                {
                    _tokens -= 1;
                    return;
                }
            }

            // Wait for token availability
            var waitTime = TimeSpan.FromSeconds(1.0 / _tokensPerSecond);
            await Task.Delay(waitTime, cancellationToken);
        }

        cancellationToken.ThrowIfCancellationRequested();
    }

    /// <summary>
    /// Tries to acquire a token without waiting.
    /// </summary>
    public bool TryAcquire()
    {
        lock (_lock)
        {
            RefillTokens();

            if (_tokens >= 1)
            {
                _tokens -= 1;
                return true;
            }

            return false;
        }
    }

    private void RefillTokens()
    {
        var now = DateTime.UtcNow;
        var elapsed = (now - _lastRefill).TotalSeconds;
        _tokens = Math.Min(_maxTokens, _tokens + (elapsed * _tokensPerSecond));
        _lastRefill = now;
    }
}

/// <summary>
/// Input sanitizer for protocol messages.
/// Prevents injection attacks and malformed data.
/// </summary>
public static class InputSanitizer
{
    private const int MaxMessageLength = 1024 * 1024; // 1 MB
    private const int MaxContextLength = 256;
    private const int MaxNestingDepth = 32;

    /// <summary>
    /// Validates and sanitizes a JSON message.
    /// </summary>
    /// <param name="json">The JSON to validate.</param>
    /// <param name="error">Error message if validation fails.</param>
    /// <returns>True if valid, false otherwise.</returns>
    public static bool ValidateJson(string? json, out string? error)
    {
        error = null;

        if (string.IsNullOrEmpty(json))
        {
            error = "Message cannot be empty";
            return false;
        }

        if (json.Length > MaxMessageLength)
        {
            error = $"Message exceeds maximum length of {MaxMessageLength} bytes";
            return false;
        }

        // Check for valid JSON structure
        try
        {
            using var doc = System.Text.Json.JsonDocument.Parse(json, new System.Text.Json.JsonDocumentOptions
            {
                MaxDepth = MaxNestingDepth
            });

            // Verify it has required 'context' field
            if (!doc.RootElement.TryGetProperty("context", out var context))
            {
                error = "Message must have 'context' property";
                return false;
            }

            var contextValue = context.GetString();
            if (string.IsNullOrEmpty(contextValue))
            {
                error = "Context cannot be empty";
                return false;
            }

            if (contextValue.Length > MaxContextLength)
            {
                error = $"Context exceeds maximum length of {MaxContextLength}";
                return false;
            }

            // Check for control characters in context (potential injection)
            if (contextValue.Any(c => char.IsControl(c)))
            {
                error = "Context contains invalid control characters";
                return false;
            }

            return true;
        }
        catch (System.Text.Json.JsonException ex)
        {
            error = $"Invalid JSON: {ex.Message}";
            return false;
        }
    }

    /// <summary>
    /// Sanitizes a string for safe logging/display.
    /// </summary>
    public static string SanitizeForDisplay(string input, int maxLength = 1000)
    {
        if (string.IsNullOrEmpty(input))
            return string.Empty;

        // Truncate
        var result = input.Length > maxLength
            ? input[..maxLength] + "...[truncated]"
            : input;

        // Remove/escape control characters
        result = new string(result.Select(c => char.IsControl(c) && c != '\n' && c != '\r' ? '?' : c).ToArray());

        return result;
    }
}
