using Newtonsoft.Json;

namespace Mbrcpval.Core;

/// <summary>
/// Parses MBRC protocol messages from JSON and NDJSON streams.
/// </summary>
public static class MessageParser
{
    /// <summary>
    /// Parses a single MBRC message from a JSON string.
    /// </summary>
    /// <param name="json">The JSON string to parse.</param>
    /// <returns>The parsed MBRC message.</returns>
    /// <exception cref="MessageParseException">Thrown when parsing fails.</exception>
    public static MbrcMessage ParseMessage(string json)
    {
        if (string.IsNullOrWhiteSpace(json))
        {
            throw new MessageParseException("Cannot parse empty or null JSON string.", 0, json);
        }

        try
        {
            return MbrcMessage.Parse(json.Trim());
        }
        catch (JsonException ex)
        {
            throw new MessageParseException($"Failed to parse JSON: {ex.Message}", 0, json, ex);
        }
        catch (Exception ex) when (ex is not MessageParseException)
        {
            throw new MessageParseException($"Unexpected error parsing message: {ex.Message}", 0, json, ex);
        }
    }

    /// <summary>
    /// Attempts to parse a single MBRC message from a JSON string.
    /// </summary>
    /// <param name="json">The JSON string to parse.</param>
    /// <param name="message">The parsed message if successful; otherwise, null.</param>
    /// <param name="error">The error message if parsing failed; otherwise, null.</param>
    /// <returns>True if parsing was successful; otherwise, false.</returns>
    public static bool TryParseMessage(string? json, out MbrcMessage? message, out string? error)
    {
        message = null;
        error = null;

        if (string.IsNullOrWhiteSpace(json))
        {
            error = "Cannot parse empty or null JSON string.";
            return false;
        }

        try
        {
            message = ParseMessage(json);
            return true;
        }
        catch (MessageParseException ex)
        {
            error = ex.Message;
            return false;
        }
    }

    /// <summary>
    /// Parses multiple MBRC messages from an NDJSON stream (CRLF-separated JSON lines).
    /// </summary>
    /// <param name="data">The NDJSON data to parse.</param>
    /// <returns>An enumerable of successfully parsed messages.</returns>
    /// <exception cref="MessageParseException">Thrown when parsing of any message fails.</exception>
    public static IEnumerable<MbrcMessage> ParseMessages(string data)
    {
        if (string.IsNullOrEmpty(data))
        {
            yield break;
        }

        var lines = SplitIntoLines(data);
        var lineNumber = 0;

        foreach (var line in lines)
        {
            lineNumber++;

            if (string.IsNullOrWhiteSpace(line))
                continue;

            MbrcMessage message;
            try
            {
                message = MbrcMessage.Parse(line.Trim());
            }
            catch (JsonException ex)
            {
                throw new MessageParseException(
                    $"Failed to parse JSON at line {lineNumber}: {ex.Message}",
                    lineNumber,
                    line,
                    ex);
            }
            catch (Exception ex) when (ex is not MessageParseException)
            {
                throw new MessageParseException(
                    $"Unexpected error at line {lineNumber}: {ex.Message}",
                    lineNumber,
                    line,
                    ex);
            }

            yield return message;
        }
    }

    /// <summary>
    /// Parses multiple MBRC messages, returning results for each line including errors.
    /// </summary>
    /// <param name="data">The NDJSON data to parse.</param>
    /// <returns>An enumerable of parse results containing either a message or an error.</returns>
    public static IEnumerable<MessageParseResult> ParseMessagesWithErrors(string data)
    {
        if (string.IsNullOrEmpty(data))
        {
            yield break;
        }

        var lines = SplitIntoLines(data);
        var lineNumber = 0;

        foreach (var line in lines)
        {
            lineNumber++;

            if (string.IsNullOrWhiteSpace(line))
                continue;

            MessageParseResult result;
            try
            {
                var message = MbrcMessage.Parse(line.Trim());
                result = new MessageParseResult(message, lineNumber, line);
            }
            catch (Exception ex)
            {
                result = new MessageParseResult(ex.Message, lineNumber, line);
            }

            yield return result;
        }
    }

    /// <summary>
    /// Counts the number of complete messages in an NDJSON stream.
    /// </summary>
    /// <param name="data">The NDJSON data to count.</param>
    /// <returns>The number of complete JSON lines (terminated by CRLF).</returns>
    public static int CountCompleteMessages(string data)
    {
        if (string.IsNullOrEmpty(data))
            return 0;

        var count = 0;
        var lines = SplitIntoLines(data);

        foreach (var line in lines)
        {
            if (!string.IsNullOrWhiteSpace(line))
            {
                // Only count if it looks like valid JSON (starts with {)
                var trimmed = line.Trim();
                if (trimmed.StartsWith('{'))
                    count++;
            }
        }

        return count;
    }

    /// <summary>
    /// Extracts complete messages from a buffer, returning the remaining incomplete data.
    /// </summary>
    /// <param name="buffer">The buffer containing message data.</param>
    /// <param name="messages">The list to add parsed messages to.</param>
    /// <returns>Any remaining incomplete data that should be kept in the buffer.</returns>
    public static string ExtractCompleteMessages(string buffer, List<MbrcMessage> messages)
    {
        if (string.IsNullOrEmpty(buffer))
            return string.Empty;

        var lastTerminator = buffer.LastIndexOf(Constants.MessageTerminator, StringComparison.Ordinal);

        if (lastTerminator < 0)
        {
            // No complete messages yet
            return buffer;
        }

        // Extract complete portion (including the terminator)
        var completePortion = buffer[..(lastTerminator + Constants.MessageTerminator.Length)];
        var remainder = buffer[(lastTerminator + Constants.MessageTerminator.Length)..];

        // Parse all complete messages
        foreach (var msg in ParseMessages(completePortion))
        {
            messages.Add(msg);
        }

        return remainder;
    }

    /// <summary>
    /// Splits NDJSON data into individual lines based on CRLF terminators.
    /// </summary>
    /// <param name="data">The data to split.</param>
    /// <returns>An enumerable of individual lines.</returns>
    private static IEnumerable<string> SplitIntoLines(string data)
    {
        // Split on CRLF, handling both \r\n and just \n for flexibility
        return data.Split(new[] { Constants.MessageTerminator, "\n" }, StringSplitOptions.None);
    }
}

/// <summary>
/// Exception thrown when message parsing fails.
/// </summary>
public class MessageParseException : Exception
{
    /// <summary>
    /// Gets the line number where the error occurred (1-based, or 0 for single message).
    /// </summary>
    public int LineNumber { get; }

    /// <summary>
    /// Gets the raw input that failed to parse.
    /// </summary>
    public string? RawInput { get; }

    /// <summary>
    /// Creates a new MessageParseException.
    /// </summary>
    /// <param name="message">The error message.</param>
    /// <param name="lineNumber">The line number where the error occurred.</param>
    /// <param name="rawInput">The raw input that failed to parse.</param>
    /// <param name="innerException">The inner exception, if any.</param>
    public MessageParseException(string message, int lineNumber, string? rawInput = null, Exception? innerException = null)
        : base(message, innerException)
    {
        LineNumber = lineNumber;
        RawInput = rawInput;
    }
}

/// <summary>
/// Represents the result of parsing a single message line.
/// </summary>
public class MessageParseResult
{
    /// <summary>
    /// Gets the parsed message, if successful.
    /// </summary>
    public MbrcMessage? Message { get; }

    /// <summary>
    /// Gets the error message, if parsing failed.
    /// </summary>
    public string? Error { get; }

    /// <summary>
    /// Gets the line number (1-based).
    /// </summary>
    public int LineNumber { get; }

    /// <summary>
    /// Gets the raw line content.
    /// </summary>
    public string RawLine { get; }

    /// <summary>
    /// Gets whether parsing was successful.
    /// </summary>
    public bool IsSuccess => Message is not null;

    /// <summary>
    /// Creates a successful parse result.
    /// </summary>
    /// <param name="message">The parsed message.</param>
    /// <param name="lineNumber">The line number.</param>
    /// <param name="rawLine">The raw line content.</param>
    public MessageParseResult(MbrcMessage message, int lineNumber, string rawLine)
    {
        Message = message ?? throw new ArgumentNullException(nameof(message));
        LineNumber = lineNumber;
        RawLine = rawLine;
    }

    /// <summary>
    /// Creates a failed parse result.
    /// </summary>
    /// <param name="error">The error message.</param>
    /// <param name="lineNumber">The line number.</param>
    /// <param name="rawLine">The raw line content.</param>
    public MessageParseResult(string error, int lineNumber, string rawLine)
    {
        Error = error ?? throw new ArgumentNullException(nameof(error));
        LineNumber = lineNumber;
        RawLine = rawLine;
    }
}
