using System.Net.Sockets;
using System.Text;

namespace Mbrcpval.Core;

/// <summary>
/// TCP client for MBRC protocol with NDJSON framing support.
/// Handles connection lifecycle, message sending/receiving, and push notifications.
/// </summary>
public class MbrcSocketClient : IDisposable
{
    /// <summary>
    /// Maximum size of the message buffer to prevent memory exhaustion attacks.
    /// </summary>
    private const int MaxBufferSize = 1024 * 1024; // 1 MB

    private TcpClient? _tcpClient;
    private NetworkStream? _networkStream;
    private StreamReader? _reader;
    private StreamWriter? _writer;
    private CancellationTokenSource? _receiveCts;
    private Task? _receiveTask;
    private readonly StringBuilder _messageBuffer = new();
    private readonly object _stateLock = new();
    private bool _disposed;

    /// <summary>
    /// Gets the current connection state.
    /// </summary>
    public ConnectionState State { get; private set; } = ConnectionState.Disconnected;

    /// <summary>
    /// Gets whether the client is connected to a server.
    /// </summary>
    public bool IsConnected => State is ConnectionState.Connected or ConnectionState.HandshakingPlayer
        or ConnectionState.HandshakingProtocol or ConnectionState.HandshakingInit or ConnectionState.Ready;

    /// <summary>
    /// Gets whether the client has completed the handshake and is ready for commands.
    /// </summary>
    public bool IsReady => State == ConnectionState.Ready;

    /// <summary>
    /// Gets the remote host address.
    /// </summary>
    public string? Host { get; private set; }

    /// <summary>
    /// Gets the remote port number.
    /// </summary>
    public int Port { get; private set; }

    /// <summary>
    /// Occurs when a message is received from the server.
    /// </summary>
    public event EventHandler<MessageReceivedEventArgs>? MessageReceived;

    /// <summary>
    /// Occurs when the connection state changes.
    /// </summary>
    public event EventHandler<ConnectionStateChangedEventArgs>? StateChanged;

    /// <summary>
    /// Occurs when an error occurs.
    /// </summary>
    public event EventHandler<ErrorEventArgs>? ErrorOccurred;

    /// <summary>
    /// Creates a new MBRC socket client.
    /// </summary>
    public MbrcSocketClient()
    {
    }

    /// <summary>
    /// Connects to an MBRC server.
    /// </summary>
    /// <param name="host">The server hostname or IP address.</param>
    /// <param name="port">The server port (default: 3000).</param>
    /// <param name="timeout">Connection timeout in milliseconds (default: 5000).</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    /// <returns>True if connection was successful; otherwise, false.</returns>
    public async Task<bool> ConnectAsync(string host, int port = Constants.DefaultPort, int timeout = 5000,
        CancellationToken cancellationToken = default)
    {
        ObjectDisposedException.ThrowIf(_disposed, this);

        if (IsConnected)
        {
            throw new InvalidOperationException("Already connected. Call Disconnect first.");
        }

        Host = host;
        Port = port;

        SetState(ConnectionState.Connecting);

        try
        {
            _tcpClient = new TcpClient();

            using var timeoutCts = new CancellationTokenSource(timeout);
            using var linkedCts = CancellationTokenSource.CreateLinkedTokenSource(timeoutCts.Token, cancellationToken);

            await _tcpClient.ConnectAsync(host, port, linkedCts.Token).ConfigureAwait(false);

            _networkStream = _tcpClient.GetStream();
            _reader = new StreamReader(_networkStream, Encoding.UTF8, leaveOpen: true);
            _writer = new StreamWriter(_networkStream, new UTF8Encoding(false), leaveOpen: true)
            {
                AutoFlush = true,
                NewLine = Constants.MessageTerminator
            };

            SetState(ConnectionState.Connected);

            // Start the receive loop
            _receiveCts = new CancellationTokenSource();
            _receiveTask = ReceiveLoopAsync(_receiveCts.Token);

            return true;
        }
        catch (OperationCanceledException)
        {
            SetState(ConnectionState.Error);
            OnError("Connection timed out or was cancelled.");
            await CleanupAsync().ConfigureAwait(false);
            return false;
        }
        catch (SocketException ex)
        {
            SetState(ConnectionState.Error);
            OnError($"Socket error: {ex.Message}");
            await CleanupAsync().ConfigureAwait(false);
            return false;
        }
        catch (Exception ex)
        {
            SetState(ConnectionState.Error);
            OnError($"Connection failed: {ex.Message}");
            await CleanupAsync().ConfigureAwait(false);
            return false;
        }
    }

    /// <summary>
    /// Disconnects from the server.
    /// </summary>
    public async Task DisconnectAsync()
    {
        if (State == ConnectionState.Disconnected)
            return;

        await CleanupAsync().ConfigureAwait(false);
        SetState(ConnectionState.Disconnected);
    }

    /// <summary>
    /// Sends an MBRC message to the server.
    /// </summary>
    /// <param name="message">The message to send.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    public async Task SendMessageAsync(MbrcMessage message, CancellationToken cancellationToken = default)
    {
        ArgumentNullException.ThrowIfNull(message);
        await SendRawAsync(message.ToJson(), cancellationToken).ConfigureAwait(false);
    }

    /// <summary>
    /// Sends a raw JSON string to the server with CRLF terminator.
    /// </summary>
    /// <param name="json">The JSON string to send.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    public async Task SendRawAsync(string json, CancellationToken cancellationToken = default)
    {
        ObjectDisposedException.ThrowIf(_disposed, this);

        if (!IsConnected || _writer is null)
        {
            throw new InvalidOperationException("Not connected to server.");
        }

        try
        {
            await _writer.WriteLineAsync(json.AsMemory(), cancellationToken).ConfigureAwait(false);
        }
        catch (Exception ex)
        {
            SetState(ConnectionState.Error);
            OnError($"Send failed: {ex.Message}");
            throw;
        }
    }

    /// <summary>
    /// Receives a single message from the server.
    /// </summary>
    /// <param name="timeout">Receive timeout in milliseconds.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    /// <returns>The received message, or null if no message available within timeout.</returns>
    public async Task<MbrcMessage?> ReceiveMessageAsync(int timeout = 5000,
        CancellationToken cancellationToken = default)
    {
        ObjectDisposedException.ThrowIf(_disposed, this);

        if (!IsConnected || _reader is null)
        {
            throw new InvalidOperationException("Not connected to server.");
        }

        using var timeoutCts = new CancellationTokenSource(timeout);
        using var linkedCts = CancellationTokenSource.CreateLinkedTokenSource(timeoutCts.Token, cancellationToken);

        try
        {
            var line = await _reader.ReadLineAsync(linkedCts.Token).ConfigureAwait(false);

            if (string.IsNullOrWhiteSpace(line))
                return null;

            return MessageParser.ParseMessage(line);
        }
        catch (OperationCanceledException)
        {
            return null;
        }
        catch (MessageParseException)
        {
            throw;
        }
        catch (Exception ex)
        {
            SetState(ConnectionState.Error);
            OnError($"Receive failed: {ex.Message}");
            throw;
        }
    }

    /// <summary>
    /// Receives all available messages within the timeout period.
    /// </summary>
    /// <param name="timeout">Timeout in milliseconds to wait for messages.</param>
    /// <param name="cancellationToken">Cancellation token.</param>
    /// <returns>List of received messages.</returns>
    public async Task<List<MbrcMessage>> ReceiveMessagesAsync(int timeout = 1000,
        CancellationToken cancellationToken = default)
    {
        var messages = new List<MbrcMessage>();

        using var timeoutCts = new CancellationTokenSource(timeout);
        using var linkedCts = CancellationTokenSource.CreateLinkedTokenSource(timeoutCts.Token, cancellationToken);

        while (!linkedCts.Token.IsCancellationRequested)
        {
            try
            {
                var message = await ReceiveMessageAsync(100, linkedCts.Token).ConfigureAwait(false);
                if (message is not null)
                {
                    messages.Add(message);
                }
            }
            catch (OperationCanceledException)
            {
                break;
            }
            catch
            {
                break;
            }
        }

        return messages;
    }

    /// <summary>
    /// Sets the connection state and updates handshake progress.
    /// </summary>
    /// <param name="newState">The new connection state.</param>
    public void SetState(ConnectionState newState)
    {
        ConnectionState oldState;
        lock (_stateLock)
        {
            if (State == newState)
                return;

            oldState = State;
            State = newState;
        }

        StateChanged?.Invoke(this, new ConnectionStateChangedEventArgs(oldState, newState));
    }

    /// <summary>
    /// Advances to the next handshake state.
    /// </summary>
    public void AdvanceHandshake()
    {
        var newState = State switch
        {
            ConnectionState.Connected => ConnectionState.HandshakingPlayer,
            ConnectionState.HandshakingPlayer => ConnectionState.HandshakingProtocol,
            ConnectionState.HandshakingProtocol => ConnectionState.HandshakingInit,
            ConnectionState.HandshakingInit => ConnectionState.Ready,
            _ => State
        };

        if (newState != State)
        {
            SetState(newState);
        }
    }

    /// <summary>
    /// Background loop that receives messages and raises events.
    /// </summary>
    private async Task ReceiveLoopAsync(CancellationToken cancellationToken)
    {
        var buffer = new char[4096];

        while (!cancellationToken.IsCancellationRequested && IsConnected && _reader is not null)
        {
            try
            {
                var bytesRead = await _reader.ReadAsync(buffer.AsMemory(), cancellationToken).ConfigureAwait(false);

                if (bytesRead == 0)
                {
                    // Connection closed by server
                    SetState(ConnectionState.Disconnected);
                    break;
                }

                // Check buffer size limit to prevent memory exhaustion
                if (_messageBuffer.Length + bytesRead > MaxBufferSize)
                {
                    OnError("Message buffer exceeded maximum size limit. Possible malformed data.");
                    _messageBuffer.Clear();
                    continue;
                }

                _messageBuffer.Append(buffer, 0, bytesRead);

                // Extract and process complete messages
                var messages = new List<MbrcMessage>();
                var remainder = MessageParser.ExtractCompleteMessages(_messageBuffer.ToString(), messages);

                _messageBuffer.Clear();
                _messageBuffer.Append(remainder);

                foreach (var message in messages)
                {
                    OnMessageReceived(message);
                }
            }
            catch (OperationCanceledException)
            {
                break;
            }
            catch (IOException)
            {
                // Connection lost
                SetState(ConnectionState.Disconnected);
                break;
            }
            catch (ObjectDisposedException)
            {
                // Reader was disposed - exit gracefully
                break;
            }
            catch (Exception ex)
            {
                OnError($"Receive loop error: {ex.Message}");
            }
        }
    }

    /// <summary>
    /// Raises the MessageReceived event.
    /// </summary>
    private void OnMessageReceived(MbrcMessage message)
    {
        MessageReceived?.Invoke(this, new MessageReceivedEventArgs(message));
    }

    /// <summary>
    /// Raises the ErrorOccurred event.
    /// </summary>
    private void OnError(string error)
    {
        ErrorOccurred?.Invoke(this, new ErrorEventArgs(error));
    }

    /// <summary>
    /// Cleans up resources.
    /// </summary>
    private async Task CleanupAsync()
    {
        _receiveCts?.Cancel();

        if (_receiveTask is not null)
        {
            try
            {
                await _receiveTask.WaitAsync(TimeSpan.FromSeconds(1)).ConfigureAwait(false);
            }
            catch
            {
                // Ignore timeout/cancellation
            }
        }

        _receiveCts?.Dispose();
        _receiveCts = null;
        _receiveTask = null;

        _writer?.Dispose();
        _writer = null;

        _reader?.Dispose();
        _reader = null;

        _networkStream?.Dispose();
        _networkStream = null;

        _tcpClient?.Dispose();
        _tcpClient = null;

        _messageBuffer.Clear();
    }

    /// <inheritdoc />
    public void Dispose()
    {
        if (_disposed)
            return;

        _disposed = true;

        // Synchronous cleanup - avoid deadlock by not calling async methods directly
        _receiveCts?.Cancel();

        // Give receive task a moment to complete, but don't block indefinitely
        if (_receiveTask != null)
        {
            try
            {
                _receiveTask.Wait(TimeSpan.FromMilliseconds(500));
            }
            catch (AggregateException)
            {
                // Ignore cancellation/timeout exceptions
            }
        }

        _receiveCts?.Dispose();
        _receiveCts = null;
        _receiveTask = null;

        _writer?.Dispose();
        _writer = null;

        _reader?.Dispose();
        _reader = null;

        _networkStream?.Dispose();
        _networkStream = null;

        _tcpClient?.Dispose();
        _tcpClient = null;

        _messageBuffer.Clear();

        GC.SuppressFinalize(this);
    }
}

/// <summary>
/// Event arguments for the MessageReceived event.
/// </summary>
public class MessageReceivedEventArgs : EventArgs
{
    /// <summary>
    /// Gets the received message.
    /// </summary>
    public MbrcMessage Message { get; }

    /// <summary>
    /// Gets the timestamp when the message was received.
    /// </summary>
    public DateTime ReceivedAt { get; }

    /// <summary>
    /// Creates new MessageReceivedEventArgs.
    /// </summary>
    /// <param name="message">The received message.</param>
    public MessageReceivedEventArgs(MbrcMessage message)
    {
        Message = message ?? throw new ArgumentNullException(nameof(message));
        ReceivedAt = DateTime.UtcNow;
    }
}

/// <summary>
/// Event arguments for the StateChanged event.
/// </summary>
public class ConnectionStateChangedEventArgs : EventArgs
{
    /// <summary>
    /// Gets the previous connection state.
    /// </summary>
    public ConnectionState OldState { get; }

    /// <summary>
    /// Gets the new connection state.
    /// </summary>
    public ConnectionState NewState { get; }

    /// <summary>
    /// Creates new ConnectionStateChangedEventArgs.
    /// </summary>
    /// <param name="oldState">The previous state.</param>
    /// <param name="newState">The new state.</param>
    public ConnectionStateChangedEventArgs(ConnectionState oldState, ConnectionState newState)
    {
        OldState = oldState;
        NewState = newState;
    }
}

/// <summary>
/// Event arguments for the ErrorOccurred event.
/// </summary>
public class ErrorEventArgs : EventArgs
{
    /// <summary>
    /// Gets the error message.
    /// </summary>
    public string Error { get; }

    /// <summary>
    /// Gets the timestamp when the error occurred.
    /// </summary>
    public DateTime OccurredAt { get; }

    /// <summary>
    /// Creates new ErrorEventArgs.
    /// </summary>
    /// <param name="error">The error message.</param>
    public ErrorEventArgs(string error)
    {
        Error = error ?? throw new ArgumentNullException(nameof(error));
        OccurredAt = DateTime.UtcNow;
    }
}
