using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Mbrcpval.Core;

/// <summary>
/// Represents an MBRC protocol message with context and data payload.
/// </summary>
public class MbrcMessage
{
    /// <summary>
    /// Gets or sets the message context (command type).
    /// </summary>
    [JsonProperty("context")]
    public string Context { get; set; } = string.Empty;

    /// <summary>
    /// Gets or sets the message data payload.
    /// Can be a primitive, object, or array depending on the context.
    /// </summary>
    [JsonProperty("data")]
    public JToken? Data { get; set; }

    /// <summary>
    /// Creates a new empty MBRC message.
    /// </summary>
    public MbrcMessage()
    {
    }

    /// <summary>
    /// Creates a new MBRC message with the specified context and data.
    /// </summary>
    /// <param name="context">The message context (command type).</param>
    /// <param name="data">The message data payload.</param>
    public MbrcMessage(string context, JToken? data = null)
    {
        Context = context ?? throw new ArgumentNullException(nameof(context));
        Data = data;
    }

    /// <summary>
    /// Creates a new MBRC message with the specified context and string data.
    /// </summary>
    /// <param name="context">The message context (command type).</param>
    /// <param name="data">The string data payload.</param>
    public MbrcMessage(string context, string? data)
    {
        Context = context ?? throw new ArgumentNullException(nameof(context));
        Data = data is not null ? JToken.FromObject(data) : null;
    }

    /// <summary>
    /// Creates a new MBRC message with the specified context and object data.
    /// </summary>
    /// <param name="context">The message context (command type).</param>
    /// <param name="data">The object data payload (will be serialized to JToken).</param>
    public MbrcMessage(string context, object? data)
    {
        Context = context ?? throw new ArgumentNullException(nameof(context));
        Data = data is not null ? JToken.FromObject(data) : null;
    }

    /// <summary>
    /// Parses an MBRC message from a JSON string.
    /// </summary>
    /// <param name="json">The JSON string to parse.</param>
    /// <returns>The parsed MBRC message.</returns>
    /// <exception cref="ArgumentNullException">Thrown when json is null or empty.</exception>
    /// <exception cref="JsonException">Thrown when the JSON is malformed or invalid.</exception>
    public static MbrcMessage Parse(string json)
    {
        if (string.IsNullOrWhiteSpace(json))
            throw new ArgumentNullException(nameof(json), "JSON string cannot be null or empty.");

        var message = JsonConvert.DeserializeObject<MbrcMessage>(json);

        if (message is null)
            throw new JsonException("Failed to deserialize JSON to MbrcMessage.");

        if (string.IsNullOrEmpty(message.Context))
            throw new JsonException("Message must have a 'context' property.");

        return message;
    }

    /// <summary>
    /// Attempts to parse an MBRC message from a JSON string.
    /// </summary>
    /// <param name="json">The JSON string to parse.</param>
    /// <param name="message">The parsed message if successful; otherwise, null.</param>
    /// <returns>True if parsing was successful; otherwise, false.</returns>
    public static bool TryParse(string? json, out MbrcMessage? message)
    {
        message = null;

        if (string.IsNullOrWhiteSpace(json))
            return false;

        try
        {
            message = Parse(json);
            return true;
        }
        catch
        {
            return false;
        }
    }

    /// <summary>
    /// Serializes the message to a JSON string.
    /// </summary>
    /// <returns>The JSON representation of the message.</returns>
    public string ToJson()
    {
        return JsonConvert.SerializeObject(this, Formatting.None);
    }

    /// <summary>
    /// Serializes the message to a JSON string with the specified formatting.
    /// </summary>
    /// <param name="formatting">The JSON formatting to use.</param>
    /// <returns>The JSON representation of the message.</returns>
    public string ToJson(Formatting formatting)
    {
        return JsonConvert.SerializeObject(this, formatting);
    }

    /// <summary>
    /// Gets the data as a strongly-typed object.
    /// </summary>
    /// <typeparam name="T">The type to deserialize to.</typeparam>
    /// <returns>The data as the specified type, or default if data is null.</returns>
    public T? GetDataAs<T>()
    {
        return Data is null ? default : Data.ToObject<T>();
    }

    /// <summary>
    /// Gets the data as a string.
    /// </summary>
    /// <returns>The data as a string, or null if data is null.</returns>
    public string? GetDataAsString()
    {
        return Data?.ToString();
    }

    /// <summary>
    /// Checks if this message has a valid, recognized context.
    /// </summary>
    /// <returns>True if the context is a known MBRC context; otherwise, false.</returns>
    public bool HasValidContext()
    {
        return Constants.IsValidContext(Context);
    }

    /// <summary>
    /// Creates a protocol handshake message.
    /// </summary>
    /// <param name="protocolVersion">The protocol version to use.</param>
    /// <returns>A protocol handshake message.</returns>
    public static MbrcMessage CreateProtocolMessage(string protocolVersion = Constants.CurrentProtocolVersion)
    {
        return new MbrcMessage(Constants.ContextProtocol, new JObject
        {
            ["protocol_version"] = protocolVersion
        });
    }

    /// <summary>
    /// Creates an init message for client identification.
    /// </summary>
    /// <param name="clientId">The unique client identifier.</param>
    /// <returns>An init message.</returns>
    public static MbrcMessage CreateInitMessage(string clientId)
    {
        return new MbrcMessage(Constants.ContextInit, new JObject
        {
            ["client_id"] = clientId
        });
    }

    /// <summary>
    /// Creates a player status request message.
    /// </summary>
    /// <returns>A player status request message.</returns>
    public static MbrcMessage CreatePlayerMessage()
    {
        return new MbrcMessage(Constants.ContextPlayer, string.Empty);
    }

    /// <summary>
    /// Creates a ping message.
    /// </summary>
    /// <returns>A ping message.</returns>
    public static MbrcMessage CreatePingMessage()
    {
        return new MbrcMessage(Constants.ContextPing, string.Empty);
    }

    /// <summary>
    /// Creates a pong response message.
    /// </summary>
    /// <returns>A pong message.</returns>
    public static MbrcMessage CreatePongMessage()
    {
        return new MbrcMessage(Constants.ContextPong, string.Empty);
    }

    /// <inheritdoc />
    public override string ToString()
    {
        return $"MbrcMessage[{Context}]: {Data?.ToString(Formatting.None) ?? "null"}";
    }
}
