namespace Mbrcpval.Core;

/// <summary>
/// Protocol constants and context strings for the MBRC protocol.
/// </summary>
public static class Constants
{
    #region Protocol Contexts

    /// <summary>
    /// Player state context - get/set player state.
    /// </summary>
    public const string ContextPlayer = "player";

    /// <summary>
    /// Protocol handshake context.
    /// </summary>
    public const string ContextProtocol = "protocol";

    /// <summary>
    /// Initialization context for client identification.
    /// </summary>
    public const string ContextInit = "init";

    /// <summary>
    /// Start playback command.
    /// </summary>
    public const string ContextPlayerPlay = "playerplay";

    /// <summary>
    /// Pause playback command.
    /// </summary>
    public const string ContextPlayerPause = "playerpause";

    /// <summary>
    /// Stop playback command.
    /// </summary>
    public const string ContextPlayerStop = "playerstop";

    /// <summary>
    /// Toggle play/pause command.
    /// </summary>
    public const string ContextPlayerPlayPause = "playerplaypause";

    /// <summary>
    /// Skip to next track command.
    /// </summary>
    public const string ContextPlayerNext = "playernext";

    /// <summary>
    /// Go to previous track command.
    /// </summary>
    public const string ContextPlayerPrevious = "playerprevious";

    /// <summary>
    /// Get/set volume level (0-100).
    /// </summary>
    public const string ContextPlayerVolume = "playervolume";

    /// <summary>
    /// Get/set mute state.
    /// </summary>
    public const string ContextPlayerMute = "playermute";

    /// <summary>
    /// Get/set shuffle mode.
    /// </summary>
    public const string ContextPlayerShuffle = "playershuffle";

    /// <summary>
    /// Get/set repeat mode.
    /// </summary>
    public const string ContextPlayerRepeat = "playerrepeat";

    /// <summary>
    /// Get current track information.
    /// </summary>
    public const string ContextNowPlayingTrack = "nowplayingtrack";

    /// <summary>
    /// Get now playing list.
    /// </summary>
    public const string ContextNowPlayingList = "nowplayinglist";

    /// <summary>
    /// Get album cover art.
    /// </summary>
    public const string ContextNowPlayingCover = "nowplayingcover";

    /// <summary>
    /// Get/set playback position.
    /// </summary>
    public const string ContextNowPlayingPosition = "nowplayingposition";

    /// <summary>
    /// Get/set track rating.
    /// </summary>
    public const string ContextNowPlayingRating = "nowplayingrating";

    /// <summary>
    /// Browse genres in library.
    /// </summary>
    public const string ContextBrowseGenres = "browsegenres";

    /// <summary>
    /// Browse artists in library.
    /// </summary>
    public const string ContextBrowseArtists = "browseartists";

    /// <summary>
    /// Browse albums in library.
    /// </summary>
    public const string ContextBrowseAlbums = "browsealbums";

    /// <summary>
    /// Browse tracks in library.
    /// </summary>
    public const string ContextBrowseTracks = "browsetracks";

    /// <summary>
    /// Search library by artist.
    /// </summary>
    public const string ContextLibrarySearchArtist = "librarysearchartist";

    /// <summary>
    /// Search library by album.
    /// </summary>
    public const string ContextLibrarySearchAlbum = "librarysearchalbum";

    /// <summary>
    /// Search library by title.
    /// </summary>
    public const string ContextLibrarySearchTitle = "librarysearchtitle";

    /// <summary>
    /// Set rating for a library track.
    /// </summary>
    public const string ContextLibrarySetRating = "librarysetrating";

    /// <summary>
    /// Set love/favorite status for a library track.
    /// </summary>
    public const string ContextLibrarySetLove = "librarysetlove";

    /// <summary>
    /// Get list of playlists.
    /// </summary>
    public const string ContextPlaylistList = "playlistlist";

    /// <summary>
    /// Play a specific playlist.
    /// </summary>
    public const string ContextPlaylistPlay = "playlistplay";

    /// <summary>
    /// Get radio stations.
    /// </summary>
    public const string ContextRadioStations = "radiostations";

    /// <summary>
    /// Get/set audio output device.
    /// </summary>
    public const string ContextPlayerOutput = "playeroutput";

    /// <summary>
    /// Player initiator action context.
    /// </summary>
    public const string ContextPlayerInitiatorAction = "playerinitiatoraction";

    /// <summary>
    /// Get plugin storage path.
    /// </summary>
    public const string ContextPluginStoragePath = "pluginstoragepath";

    /// <summary>
    /// Ping request for keep-alive.
    /// </summary>
    public const string ContextPing = "ping";

    /// <summary>
    /// Pong response to ping.
    /// </summary>
    public const string ContextPong = "pong";

    /// <summary>
    /// UDP discovery broadcast/response.
    /// </summary>
    public const string ContextDiscovery = "discovery";

    /// <summary>
    /// Push notification from server.
    /// </summary>
    public const string ContextNotify = "notify";

    /// <summary>
    /// Subscribe to library change notifications.
    /// </summary>
    public const string ContextLibrarySubscribe = "librarysubscribe";

    /// <summary>
    /// Unsubscribe from library change notifications.
    /// </summary>
    public const string ContextLibraryUnsubscribe = "libraryunsubscribe";

    #endregion

    #region Network Configuration

    /// <summary>
    /// Default TCP port for MBRC server.
    /// </summary>
    public const int DefaultPort = 3000;

    /// <summary>
    /// Multicast address for UDP discovery.
    /// </summary>
    public const string DiscoveryMulticastAddress = "239.1.5.10";

    /// <summary>
    /// Port for UDP discovery.
    /// </summary>
    public const int DiscoveryPort = 45345;

    /// <summary>
    /// Full discovery endpoint as "address:port".
    /// </summary>
    public const string DiscoveryEndpoint = "239.1.5.10:45345";

    #endregion

    #region Protocol Versions

    /// <summary>
    /// Protocol version 4.0 (legacy).
    /// </summary>
    public const string ProtocolVersion40 = "4.0";

    /// <summary>
    /// Protocol version 4.5 (current).
    /// </summary>
    public const string ProtocolVersion45 = "4.5";

    /// <summary>
    /// Current/default protocol version.
    /// </summary>
    public const string CurrentProtocolVersion = ProtocolVersion45;

    #endregion

    #region Message Framing

    /// <summary>
    /// Message terminator for NDJSON framing (CRLF).
    /// </summary>
    public const string MessageTerminator = "\r\n";

    /// <summary>
    /// Carriage return character.
    /// </summary>
    public const char CarriageReturn = '\r';

    /// <summary>
    /// Line feed character.
    /// </summary>
    public const char LineFeed = '\n';

    #endregion

    #region All Contexts Collection

    /// <summary>
    /// Array of all valid protocol context strings.
    /// </summary>
    public static readonly string[] AllContexts =
    [
        ContextPlayer,
        ContextProtocol,
        ContextInit,
        ContextPlayerPlay,
        ContextPlayerPause,
        ContextPlayerStop,
        ContextPlayerPlayPause,
        ContextPlayerNext,
        ContextPlayerPrevious,
        ContextPlayerVolume,
        ContextPlayerMute,
        ContextPlayerShuffle,
        ContextPlayerRepeat,
        ContextNowPlayingTrack,
        ContextNowPlayingList,
        ContextNowPlayingCover,
        ContextNowPlayingPosition,
        ContextNowPlayingRating,
        ContextBrowseGenres,
        ContextBrowseArtists,
        ContextBrowseAlbums,
        ContextBrowseTracks,
        ContextLibrarySearchArtist,
        ContextLibrarySearchAlbum,
        ContextLibrarySearchTitle,
        ContextLibrarySetRating,
        ContextLibrarySetLove,
        ContextPlaylistList,
        ContextPlaylistPlay,
        ContextRadioStations,
        ContextPlayerOutput,
        ContextPlayerInitiatorAction,
        ContextPluginStoragePath,
        ContextPing,
        ContextPong,
        ContextDiscovery,
        ContextNotify,
        ContextLibrarySubscribe,
        ContextLibraryUnsubscribe
    ];

    /// <summary>
    /// Checks if a context string is a valid MBRC context.
    /// </summary>
    /// <param name="context">The context string to validate.</param>
    /// <returns>True if the context is valid; otherwise, false.</returns>
    public static bool IsValidContext(string? context)
    {
        if (string.IsNullOrEmpty(context))
            return false;

        return Array.Exists(AllContexts, c => c.Equals(context, StringComparison.OrdinalIgnoreCase));
    }

    #endregion
}
