using System;
using System.Threading.Tasks;
using Mbrcpval.CLI.Commands;
using Spectre.Console;
using Spectre.Console.Cli;

namespace Mbrcpval.CLI
{
    /// <summary>
    /// MBRCPVAL - MusicBee Remote Control Protocol Validator
    /// Command-line tool for validating MBRC protocol implementations.
    /// </summary>
    public class Program
    {
        public static async Task<int> Main(string[] args)
        {
            // Display banner unless --no-color or running in CI
            if (!HasFlag(args, "--no-color") && !HasFlag(args, "-q") && !HasFlag(args, "--quiet"))
            {
                DisplayBanner();
            }

            var app = new CommandApp();

            app.Configure(config =>
            {
                config.SetApplicationName("mbrcpval");
                config.SetApplicationVersion("1.5.26.3");

                // Global settings
                config.PropagateExceptions();
                config.ValidateExamples();

                // Server validation command
                config.AddCommand<ServerCommand>("server")
                    .WithDescription("Validate a running MBRC server/plugin")
                    .WithExample(new[] { "server", "-H", "192.168.1.100" })
                    .WithExample(new[] { "server", "-H", "localhost", "-p", "3000", "-s", "core" })
                    .WithExample(new[] { "server", "-H", "192.168.1.100", "-p", "3000", "--protocol", "4.5", "-r", "html", "-o", "report.html" });

                // Message validation command
                config.AddCommand<ValidateCommand>("validate")
                    .WithDescription("Validate a single MBRC protocol message")
                    .WithExample(new[] { "validate", "--message", "{\"context\":\"player\",\"data\":\"PlayPause\"}" })
                    .WithExample(new[] { "validate", "--file", "message.json", "--direction", "client" })
                    .WithExample(new[] { "validate", "--stdin" });

                // Mock server command
                config.AddCommand<MockCommand>("mock")
                    .WithDescription("Start a mock MBRC server for testing")
                    .WithExample(new[] { "mock", "--port", "3000" })
                    .WithExample(new[] { "mock", "-p", "3000", "--generate", "100", "--protocol", "4.5" })
                    .WithExample(new[] { "mock", "--latency", "50", "--name", "TestServer" });

                // Interactive REPL command
                config.AddCommand<ReplCommand>("repl")
                    .WithDescription("Start an interactive REPL session with an MBRC server")
                    .WithExample(new[] { "repl", "-H", "192.168.1.100" })
                    .WithExample(new[] { "repl", "-H", "localhost", "-p", "3000", "--protocol", "4.5" });

                // Proxy command
                config.AddCommand<ProxyCommand>("proxy")
                    .WithDescription("Start a proxy to intercept and log MBRC messages")
                    .WithExample(new[] { "proxy", "--listen", "3001", "--target", "localhost:3000" })
                    .WithExample(new[] { "proxy", "-l", "3001", "-t", "192.168.1.100:3000", "--output", "traffic.log", "--format", "pretty" });
            });

            try
            {
                return await app.RunAsync(args);
            }
            catch (Exception ex)
            {
                AnsiConsole.MarkupLine($"[red]Error:[/] {ex.Message}");

                if (HasFlag(args, "--verbose") || HasFlag(args, "-v"))
                {
                    AnsiConsole.WriteException(ex);
                }

                return 2; // Error exit code
            }
        }

        /// <summary>
        /// Display the ASCII art banner for MBRCPVAL.
        /// </summary>
        private static void DisplayBanner()
        {
            var banner = @"
[blue]  __  __ ____  ____   ____  ______     ___    _     [/]
[blue] |  \/  | __ )|  _ \ / ___||  _ \ \   / / \  | |    [/]
[blue] | |\/| |  _ \| |_) | |    | |_) \ \ / / _ \ | |    [/]
[blue] | |  | | |_) |  _ <| |___ |  __/ \ V / ___ \| |___ [/]
[blue] |_|  |_|____/|_| \_\\____||_|     \_/_/   \_\_____|[/]
[grey]                                                     [/]
[grey] MusicBee Remote Control Protocol Validator v1.5.26.3[/]
";
            AnsiConsole.Markup(banner);
            AnsiConsole.WriteLine();
        }

        /// <summary>
        /// Check if a flag is present in the command-line arguments.
        /// </summary>
        private static bool HasFlag(string[] args, string flag)
        {
            foreach (var arg in args)
            {
                if (arg.Equals(flag, StringComparison.OrdinalIgnoreCase))
                {
                    return true;
                }
            }
            return false;
        }
    }
}
