using System;
using System.ComponentModel;
using System.IO;
using System.Text.Json;
using System.Threading.Tasks;
using Mbrcpval.CLI.Settings;
using Spectre.Console;
using Spectre.Console.Cli;

namespace Mbrcpval.CLI.Commands
{
    /// <summary>
    /// Direction option for validation context.
    /// </summary>
    public enum ValidateDirection
    {
        /// <summary>Auto-detect based on message content.</summary>
        Auto,
        /// <summary>Message sent from client to server.</summary>
        Client,
        /// <summary>Message sent from server to client.</summary>
        Server
    }

    /// <summary>
    /// Settings for the validate command.
    /// </summary>
    public class ValidateCommandSettings : CommonSettings
    {
        /// <summary>
        /// Inline JSON message to validate.
        /// </summary>
        [CommandOption("-m|--message <JSON>")]
        [Description("Inline JSON message to validate")]
        public string? Message { get; set; }

        /// <summary>
        /// Path to a JSON file containing the message to validate.
        /// </summary>
        [CommandOption("-f|--file <PATH>")]
        [Description("Path to a JSON file containing the message")]
        public string? File { get; set; }

        /// <summary>
        /// Read message from standard input.
        /// </summary>
        [CommandOption("--stdin")]
        [Description("Read message from standard input")]
        [DefaultValue(false)]
        public bool Stdin { get; set; }

        /// <summary>
        /// The direction of the message for context-aware validation.
        /// </summary>
        [CommandOption("-d|--direction <DIRECTION>")]
        [Description("Message direction (auto, client, server)")]
        [DefaultValue(ValidateDirection.Auto)]
        public ValidateDirection Direction { get; set; } = ValidateDirection.Auto;

        /// <summary>
        /// The protocol version for schema selection.
        /// </summary>
        [CommandOption("--protocol <VERSION>")]
        [Description("Protocol version for schema selection (4.0 or 4.5)")]
        [DefaultValue("4.5")]
        public string Protocol { get; set; } = "4.5";

        /// <summary>
        /// Output format for validation results.
        /// </summary>
        [CommandOption("--format <FORMAT>")]
        [Description("Output format (text, json)")]
        [DefaultValue("text")]
        public string Format { get; set; } = "text";

        /// <summary>
        /// Validate the settings.
        /// </summary>
        public override ValidationResult Validate()
        {
            var baseResult = base.Validate();
            if (!baseResult.Successful)
            {
                return baseResult;
            }

            // Count how many input sources are specified
            var inputCount = 0;
            if (!string.IsNullOrEmpty(Message)) inputCount++;
            if (!string.IsNullOrEmpty(File)) inputCount++;
            if (Stdin) inputCount++;

            if (inputCount == 0)
            {
                return ValidationResult.Error("Must specify one of: --message, --file, or --stdin");
            }

            if (inputCount > 1)
            {
                return ValidationResult.Error("Only one input source can be specified (--message, --file, or --stdin)");
            }

            if (!string.IsNullOrEmpty(File) && !System.IO.File.Exists(File))
            {
                return ValidationResult.Error($"File not found: {File}");
            }

            var validProtocols = new[] { "4.0", "4.5" };
            if (!Array.Exists(validProtocols, p => p.Equals(Protocol, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidationResult.Error($"Invalid protocol version '{Protocol}'. Valid options: {string.Join(", ", validProtocols)}");
            }

            var validFormats = new[] { "text", "json" };
            if (!Array.Exists(validFormats, f => f.Equals(Format, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidationResult.Error($"Invalid format '{Format}'. Valid options: {string.Join(", ", validFormats)}");
            }

            return ValidationResult.Success();
        }
    }

    /// <summary>
    /// Command to validate a single MBRC protocol message.
    /// </summary>
    public class ValidateCommand : AsyncCommand<ValidateCommandSettings>
    {
        /// <summary>
        /// Execute the validate command.
        /// </summary>
        /// <param name="context">The command context.</param>
        /// <param name="settings">The command settings.</param>
        /// <returns>Exit code: 0=valid, 1=invalid, 2=error.</returns>
        public override async Task<int> ExecuteAsync(CommandContext context, ValidateCommandSettings settings)
        {
            string jsonMessage;

            try
            {
                // Get the message content from the appropriate source
                if (!string.IsNullOrEmpty(settings.Message))
                {
                    jsonMessage = settings.Message;
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine("[grey]Reading from inline message[/]");
                    }
                }
                else if (!string.IsNullOrEmpty(settings.File))
                {
                    jsonMessage = await System.IO.File.ReadAllTextAsync(settings.File);
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"[grey]Reading from file: {settings.File}[/]");
                    }
                }
                else if (settings.Stdin)
                {
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine("[grey]Reading from stdin[/]");
                    }
                    using var reader = new StreamReader(Console.OpenStandardInput());
                    jsonMessage = await reader.ReadToEndAsync();
                }
                else
                {
                    AnsiConsole.MarkupLine("[red]Error: No input source specified[/]");
                    return 2;
                }

                // Parse JSON to verify it's valid
                JsonDocument? document = null;
                try
                {
                    document = JsonDocument.Parse(jsonMessage);
                }
                catch (JsonException ex)
                {
                    OutputValidationResult(settings, false, "Invalid JSON", ex.Message, jsonMessage);
                    return 1;
                }

                // Extract message properties
                var root = document.RootElement;
                var hasContext = root.TryGetProperty("context", out var contextElement);
                var hasData = root.TryGetProperty("data", out var dataElement);

                var messageContext = hasContext ? contextElement.GetString() : null;

                // Determine direction if auto
                var direction = settings.Direction;
                if (direction == ValidateDirection.Auto)
                {
                    direction = DetectDirection(messageContext, root);
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"[grey]Auto-detected direction: {direction}[/]");
                    }
                }

                // Validate the message structure
                var validationErrors = new System.Collections.Generic.List<string>();

                if (!hasContext)
                {
                    validationErrors.Add("Missing required property: 'context'");
                }
                else if (string.IsNullOrWhiteSpace(messageContext))
                {
                    validationErrors.Add("Property 'context' cannot be empty");
                }

                // TODO: Implement full schema validation
                // var validator = new SchemaValidator(settings.Protocol);
                // var schemaErrors = validator.Validate(document, direction);
                // validationErrors.AddRange(schemaErrors);

                // Validate against known contexts
                if (!string.IsNullOrEmpty(messageContext))
                {
                    var validContexts = new[]
                    {
                        "player", "playerstate", "playerstatus", "playervolume", "playershuffle", "playerrepeat",
                        "playerscrobble", "playermute", "playerplaypause", "playerstop", "playerprevious",
                        "playernext", "playerseek", "nowplayingtrack", "nowplayingcover", "nowplayinglyrics",
                        "nowplayingrating", "nowplayinglist", "nowplayinglistplay", "nowplayinglistremove",
                        "nowplayinglistmove", "nowplayinglistsearch", "nowplayingposition", "playlist",
                        "playlistlist", "playlistplay", "library", "librarysearch", "libraryartists",
                        "libraryalbums", "librarytracks", "librarygenres", "libraryqueue", "output",
                        "outputswitch", "protocol", "ping", "pong", "init", "close", "error"
                    };

                    if (!Array.Exists(validContexts, c => c.Equals(messageContext, StringComparison.OrdinalIgnoreCase)))
                    {
                        validationErrors.Add($"Unknown context: '{messageContext}'");
                    }
                }

                // Output results
                if (validationErrors.Count == 0)
                {
                    OutputValidationResult(settings, true, messageContext ?? "unknown", null, jsonMessage);
                    return 0;
                }
                else
                {
                    OutputValidationResult(settings, false, messageContext ?? "unknown",
                        string.Join("; ", validationErrors), jsonMessage);
                    return 1;
                }
            }
            catch (Exception ex)
            {
                if (settings.Format.Equals("json", StringComparison.OrdinalIgnoreCase))
                {
                    var errorJson = JsonSerializer.Serialize(new
                    {
                        valid = false,
                        error = ex.Message,
                        errorType = "exception"
                    }, new JsonSerializerOptions { WriteIndented = true });
                    AnsiConsole.WriteLine(errorJson);
                }
                else
                {
                    AnsiConsole.MarkupLine($"[red]Error: {ex.Message}[/]");
                    if (settings.Verbose)
                    {
                        AnsiConsole.WriteException(ex);
                    }
                }
                return 2;
            }
        }

        /// <summary>
        /// Detect message direction based on context and content.
        /// </summary>
        private ValidateDirection DetectDirection(string? context, JsonElement root)
        {
            if (string.IsNullOrEmpty(context))
            {
                return ValidateDirection.Auto;
            }

            // Server-to-client contexts typically include state information
            var serverContexts = new[]
            {
                "playerstate", "playerstatus", "nowplayingtrack", "nowplayingcover",
                "nowplayinglyrics", "nowplayinglist", "playlistlist", "libraryartists",
                "libraryalbums", "librarytracks", "librarygenres", "pong", "error"
            };

            // Client-to-server contexts are typically commands
            var clientContexts = new[]
            {
                "playerplaypause", "playerstop", "playerprevious", "playernext",
                "playerseek", "nowplayinglistplay", "nowplayinglistremove",
                "nowplayinglistmove", "playlistplay", "librarysearch", "libraryqueue",
                "outputswitch", "ping", "init", "close"
            };

            if (Array.Exists(serverContexts, c => c.Equals(context, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidateDirection.Server;
            }

            if (Array.Exists(clientContexts, c => c.Equals(context, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidateDirection.Client;
            }

            // For ambiguous contexts, check if data looks like a command or state
            if (root.TryGetProperty("data", out var data))
            {
                if (data.ValueKind == JsonValueKind.String)
                {
                    // String data is typically a command
                    return ValidateDirection.Client;
                }
                else if (data.ValueKind == JsonValueKind.Object)
                {
                    // Object data is typically state
                    return ValidateDirection.Server;
                }
            }

            return ValidateDirection.Auto;
        }

        /// <summary>
        /// Output the validation result in the specified format.
        /// </summary>
        private void OutputValidationResult(ValidateCommandSettings settings, bool isValid,
            string context, string? error, string originalMessage)
        {
            if (settings.Format.Equals("json", StringComparison.OrdinalIgnoreCase))
            {
                var result = new
                {
                    valid = isValid,
                    context = context,
                    direction = settings.Direction.ToString().ToLower(),
                    protocol = settings.Protocol,
                    error = error,
                    message = originalMessage
                };

                var options = new JsonSerializerOptions { WriteIndented = true };
                AnsiConsole.WriteLine(JsonSerializer.Serialize(result, options));
            }
            else
            {
                // Text format
                if (isValid)
                {
                    AnsiConsole.MarkupLine($"[green]VALID[/] - Context: [cyan]{context}[/]");

                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"  Direction: [grey]{settings.Direction}[/]");
                        AnsiConsole.MarkupLine($"  Protocol: [grey]{settings.Protocol}[/]");

                        // Pretty-print the message
                        AnsiConsole.WriteLine();
                        AnsiConsole.MarkupLine("[grey]Message:[/]");
                        try
                        {
                            var doc = JsonDocument.Parse(originalMessage);
                            var formatted = JsonSerializer.Serialize(doc.RootElement,
                                new JsonSerializerOptions { WriteIndented = true });
                            AnsiConsole.Write(new Panel(formatted)
                                .Border(BoxBorder.Rounded)
                                .BorderColor(Color.Grey));
                        }
                        catch
                        {
                            AnsiConsole.WriteLine(originalMessage);
                        }
                    }
                }
                else
                {
                    AnsiConsole.MarkupLine($"[red]INVALID[/] - Context: [cyan]{context}[/]");
                    if (!string.IsNullOrEmpty(error))
                    {
                        AnsiConsole.MarkupLine($"  [red]Error: {error}[/]");
                    }

                    if (settings.Verbose)
                    {
                        AnsiConsole.WriteLine();
                        AnsiConsole.MarkupLine("[grey]Original message:[/]");
                        AnsiConsole.WriteLine(originalMessage);
                    }
                }
            }
        }
    }
}
