using System;
using System.ComponentModel;
using System.Diagnostics;
using System.Threading.Tasks;
using Mbrcpval.CLI.Settings;
using Mbrcpval.Testing;
using Spectre.Console;
using Spectre.Console.Cli;

namespace Mbrcpval.CLI.Commands
{
    /// <summary>
    /// Settings for the server validation command.
    /// </summary>
    public class ServerCommandSettings : CommonSettings
    {
        /// <summary>
        /// The host/IP address of the MBRC server to validate.
        /// </summary>
        [CommandOption("-H|--host <HOST>")]
        [Description("The host/IP address of the MBRC server (required)")]
        public string? Host { get; set; }

        /// <summary>
        /// The port number of the MBRC server.
        /// </summary>
        [CommandOption("-p|--port <PORT>")]
        [Description("The port number of the MBRC server")]
        [DefaultValue(3000)]
        public int Port { get; set; } = 3000;

        /// <summary>
        /// The test suite to run.
        /// </summary>
        [CommandOption("-s|--suite <SUITE>")]
        [Description("The test suite to run (all, core, playback, library, playlist, nowplaying, output, cover, lyrics, rating, protocol)")]
        [DefaultValue("all")]
        public string Suite { get; set; } = "all";

        /// <summary>
        /// Run a specific test by ID.
        /// </summary>
        [CommandOption("-t|--test <TEST>")]
        [Description("Run a specific test by ID (e.g., 'core.handshake')")]
        public string? Test { get; set; }

        /// <summary>
        /// The protocol version to test against.
        /// </summary>
        [CommandOption("--protocol <VERSION>")]
        [Description("The protocol version to test (4.0 or 4.5)")]
        [DefaultValue("4.5")]
        public string Protocol { get; set; } = "4.5";

        /// <summary>
        /// The output report format.
        /// </summary>
        [CommandOption("-r|--report <FORMAT>")]
        [Description("Report format (text, json, html, junit)")]
        [DefaultValue("text")]
        public string Report { get; set; } = "text";

        /// <summary>
        /// Output file path for the report.
        /// </summary>
        [CommandOption("-o|--output <FILE>")]
        [Description("Output file path for the report")]
        public string? Output { get; set; }

        /// <summary>
        /// Connection and test timeout in seconds.
        /// </summary>
        [CommandOption("--timeout <SECONDS>")]
        [Description("Connection and test timeout in seconds")]
        [DefaultValue(30)]
        public int Timeout { get; set; } = 30;

        /// <summary>
        /// Stop on first failure.
        /// </summary>
        [CommandOption("--fail-fast")]
        [Description("Stop execution on first test failure")]
        [DefaultValue(false)]
        public bool FailFast { get; set; }

        /// <summary>
        /// List all tests without running them.
        /// </summary>
        [CommandOption("--list")]
        [Description("List all available tests without running them")]
        [DefaultValue(false)]
        public bool ListTests { get; set; }

        /// <summary>
        /// Validate the settings.
        /// </summary>
        public override ValidationResult Validate()
        {
            var baseResult = base.Validate();
            if (!baseResult.Successful)
            {
                return baseResult;
            }

            // --list doesn't require host
            if (ListTests)
            {
                return ValidationResult.Success();
            }

            if (string.IsNullOrWhiteSpace(Host))
            {
                return ValidationResult.Error("Host is required. Use --host <HOST> to specify the server address.");
            }

            if (Port < 1 || Port > 65535)
            {
                return ValidationResult.Error("Port must be between 1 and 65535.");
            }

            var validSuites = new[] { "all", "core", "playback", "library", "playlist", "nowplaying", "output", "cover", "lyrics", "rating", "protocol" };
            if (!Array.Exists(validSuites, s => s.Equals(Suite, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidationResult.Error($"Invalid suite '{Suite}'. Valid options: {string.Join(", ", validSuites)}");
            }

            var validProtocols = new[] { "4.0", "4.5" };
            if (!Array.Exists(validProtocols, p => p.Equals(Protocol, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidationResult.Error($"Invalid protocol version '{Protocol}'. Valid options: {string.Join(", ", validProtocols)}");
            }

            var validReports = new[] { "text", "json", "html", "junit" };
            if (!Array.Exists(validReports, r => r.Equals(Report, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidationResult.Error($"Invalid report format '{Report}'. Valid options: {string.Join(", ", validReports)}");
            }

            if (Timeout < 1 || Timeout > 300)
            {
                return ValidationResult.Error("Timeout must be between 1 and 300 seconds.");
            }

            return ValidationResult.Success();
        }
    }

    /// <summary>
    /// Command to validate a running MBRC server/plugin.
    /// </summary>
    public class ServerCommand : AsyncCommand<ServerCommandSettings>
    {
        /// <summary>
        /// Execute the server validation command.
        /// </summary>
        /// <param name="context">The command context.</param>
        /// <param name="settings">The command settings.</param>
        /// <returns>Exit code: 0=pass, 1=fail, 2=error.</returns>
        public override async Task<int> ExecuteAsync(CommandContext context, ServerCommandSettings settings)
        {
            // Handle --list option
            if (settings.ListTests)
            {
                return ListAllTests(settings);
            }

            var exitCode = 0;
            var results = new List<TestResult>();
            var stopwatch = Stopwatch.StartNew();
            string? detectedProtocol = null;
            bool? ariaEnabled = null;
            bool? experimentalEnabled = null;

            // Create log file for protocol messages
            var logFile = Path.Combine(Path.GetTempPath(), $"mbrcpval_{DateTime.Now:yyyyMMdd_HHmmss}.log");
            StreamWriter? logWriter = null;

            try
            {
                logWriter = new StreamWriter(logFile, false, System.Text.Encoding.UTF8);
                logWriter.WriteLine($"MBRCPVAL Protocol Log - {DateTime.Now:yyyy-MM-dd HH:mm:ss}");
                logWriter.WriteLine($"Host: {settings.Host}:{settings.Port}");
                logWriter.WriteLine($"Protocol: {settings.Protocol}");
                logWriter.WriteLine(new string('-', 60));
                logWriter.Flush();

                // First, detect the server's protocol version and plugin settings
                (detectedProtocol, ariaEnabled, experimentalEnabled) = await DetectProtocolAndSettingsAsync(settings);

                // Get test counts for display
                var (core, v40, v45, total) = TestSuiteFactory.GetTestCounts(settings.Protocol);

                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine($"[bold]Validating MBRC Server[/]");
                    AnsiConsole.MarkupLine($"  Host: [cyan]{settings.Host}:{settings.Port}[/]");
                    AnsiConsole.MarkupLine($"  Suite: [cyan]{settings.Suite}[/]");
                    AnsiConsole.MarkupLine($"  Testing Protocol: [cyan]{settings.Protocol}[/]");
                    AnsiConsole.MarkupLine($"  Detected Protocol: [cyan]{detectedProtocol ?? "unknown"}[/]");

                    // Show plugin settings (only for v4.5+ plugins)
                    var isV45 = detectedProtocol != null &&
                        (detectedProtocol.Contains("4.5") ||
                         (double.TryParse(detectedProtocol, out var ver) && ver >= 4.5));

                    if (isV45)
                    {
                        var ariaStatus = ariaEnabled switch
                        {
                            true => "[green]Enabled[/]",
                            false => "[yellow]Disabled[/]",
                            null => "[grey]Unknown[/]"
                        };
                        var expStatus = experimentalEnabled switch
                        {
                            true => "[green]Enabled[/]",
                            false => "[yellow]Disabled[/]",
                            null => "[grey]Unknown[/]"
                        };
                        AnsiConsole.MarkupLine($"  ARiA: {ariaStatus}  |  Experimental: {expStatus}");
                    }
                    else
                    {
                        AnsiConsole.MarkupLine($"  ARiA: [grey]N/A (v4.5+)[/]  |  Experimental: [grey]N/A (v4.5+)[/]");
                    }

                    AnsiConsole.MarkupLine($"  Tests: [cyan]{total}[/] (Core: {core}, v4.0: {v40}" + (v45 > 0 ? $", v4.5: {v45}" : "") + ")");
                    AnsiConsole.MarkupLine($"  Log: [cyan]{logFile}[/]");
                    if (!string.IsNullOrEmpty(settings.Test))
                    {
                        AnsiConsole.MarkupLine($"  Filter: [cyan]{settings.Test}[/]");
                    }
                    AnsiConsole.WriteLine();
                }

                // Load test cases
                var testCases = TestSuiteFactory.CreateTestSuite(settings.Protocol, settings.Suite);

                // Filter by specific test ID if provided
                if (!string.IsNullOrEmpty(settings.Test))
                {
                    testCases = testCases.Where(t =>
                        t.Id.Equals(settings.Test, StringComparison.OrdinalIgnoreCase) ||
                        t.Name.Contains(settings.Test, StringComparison.OrdinalIgnoreCase)).ToList();

                    if (testCases.Count == 0)
                    {
                        AnsiConsole.MarkupLine($"[yellow]No tests found matching '{settings.Test}'[/]");
                        return 1;
                    }
                }

                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine($"[grey]Running {testCases.Count} tests...[/]");
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"[yellow]Verbose logging enabled - showing all protocol messages[/]");
                    }
                    AnsiConsole.WriteLine();
                }

                // Create test runner
                var runner = new TestRunner
                {
                    DefaultTimeout = TimeSpan.FromSeconds(settings.Timeout),
                    FailFast = settings.FailFast
                };

                // Subscribe to events for progress
                runner.TestStarted += (sender, e) =>
                {
                    if (settings.Verbose)
                    {
                        var name = EscapeMarkup(e.TestCase.DisplayName);
                        AnsiConsole.MarkupLine($"[grey]Running: {name}[/]");
                    }
                };

                runner.TestCompleted += (sender, e) =>
                {
                    var status = e.Result.Status switch
                    {
                        TestStatus.Pass => "[green]PASS[/]",
                        TestStatus.Fail => "[red]FAIL[/]",
                        TestStatus.Skip => "[yellow]SKIP[/]",
                        TestStatus.Error => "[red]ERROR[/]",
                        _ => "[grey]???[/]"
                    };

                    var name = EscapeMarkup(e.TestCase.DisplayName);
                    if (!settings.Quiet)
                    {
                        AnsiConsole.MarkupLine($"  {status} {name} ({e.Result.Duration.TotalMilliseconds:F0}ms)");
                    }

                    if (e.Result.Failed && !string.IsNullOrEmpty(e.Result.ErrorMessage))
                    {
                        var error = EscapeMarkup(e.Result.ErrorMessage);
                        AnsiConsole.MarkupLine($"       [red]{error}[/]");
                    }
                };

                // Connect and run tests
                await AnsiConsole.Status()
                    .AutoRefresh(true)
                    .Spinner(Spinner.Known.Dots)
                    .SpinnerStyle(Style.Parse("blue"))
                    .StartAsync("Connecting to server...", async ctx =>
                    {
                        ctx.Status($"Connecting to {settings.Host}:{settings.Port}...");
                    });

                // Run all tests with logging
                Action<string> onSend = msg =>
                {
                    var ts = DateTime.Now.ToString("HH:mm:ss.fff");
                    logWriter.WriteLine($"[{ts}] >>> SEND: {msg}");
                    logWriter.Flush();
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"[grey]>>> SEND: {EscapeMarkup(msg)}[/]");
                    }
                };

                Action<string> onReceive = msg =>
                {
                    var ts = DateTime.Now.ToString("HH:mm:ss.fff");
                    logWriter.WriteLine($"[{ts}] <<< RECV: {msg}");
                    logWriter.Flush();
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"[grey]<<< RECV: {EscapeMarkup(msg)}[/]");
                    }
                };

                results = await runner.RunAllAsync(settings.Host!, settings.Port, testCases, onSend, onReceive);

                stopwatch.Stop();

                // Close log file
                logWriter.WriteLine(new string('-', 60));
                logWriter.WriteLine($"Completed: {DateTime.Now:yyyy-MM-dd HH:mm:ss}");
                logWriter.Close();

                // Generate report if output specified
                if (!string.IsNullOrEmpty(settings.Output))
                {
                    await GenerateReportAsync(results, settings);
                }
            }
            catch (Exception ex)
            {
                stopwatch.Stop();
                AnsiConsole.MarkupLine($"[red]Error: {ex.Message}[/]");
                if (settings.Verbose)
                {
                    AnsiConsole.WriteException(ex);
                }
                return 2;
            }
            finally
            {
                logWriter?.Close();
            }

            // Display results summary
            if (!settings.Quiet)
            {
                AnsiConsole.WriteLine();

                var passed = results.Count(r => r.Passed);
                var failed = results.Count(r => r.Failed);
                var skipped = results.Count(r => r.Skipped);
                var errored = results.Count(r => r.Errored);

                var table = new Table()
                    .Border(TableBorder.Rounded)
                    .AddColumn("Metric")
                    .AddColumn("Value");

                table.AddRow("Protocol", settings.Protocol);
                table.AddRow("Total Tests", results.Count.ToString());
                table.AddRow("[green]Passed[/]", $"[green]{passed}[/]");
                table.AddRow("[red]Failed[/]", $"[red]{failed}[/]");
                table.AddRow("[yellow]Skipped[/]", $"[yellow]{skipped}[/]");
                if (errored > 0)
                {
                    table.AddRow("[red]Errors[/]", $"[red]{errored}[/]");
                }
                table.AddRow("Duration", $"{stopwatch.Elapsed.TotalSeconds:F2}s");

                AnsiConsole.Write(table);
                AnsiConsole.WriteLine();

                if (failed == 0 && errored == 0)
                {
                    AnsiConsole.MarkupLine("[green]All tests passed![/]");
                    exitCode = 0;
                }
                else
                {
                    AnsiConsole.MarkupLine($"[red]{failed + errored} test(s) failed[/]");
                    exitCode = 1;

                    // Show failed test details
                    if (settings.Verbose)
                    {
                        AnsiConsole.WriteLine();
                        AnsiConsole.MarkupLine("[bold]Failed Tests:[/]");
                        foreach (var result in results.Where(r => r.Failed || r.Errored))
                        {
                            var failName = EscapeMarkup(result.TestCase.DisplayName);
                            AnsiConsole.MarkupLine($"  [red]{failName}[/]");
                            if (!string.IsNullOrEmpty(result.ErrorMessage))
                            {
                                var failError = EscapeMarkup(result.ErrorMessage);
                                AnsiConsole.MarkupLine($"    {failError}");
                            }
                        }
                    }
                }
            }
            else
            {
                // Quiet mode - just return exit code
                exitCode = results.Any(r => r.Failed || r.Errored) ? 1 : 0;
            }

            return exitCode;
        }

        /// <summary>
        /// Generate a report file.
        /// </summary>
        private async Task GenerateReportAsync(List<TestResult> results, ServerCommandSettings settings)
        {
            try
            {
                var reportContent = settings.Report.ToLowerInvariant() switch
                {
                    "json" => GenerateJsonReport(results, settings),
                    "html" => GenerateHtmlReport(results, settings),
                    "junit" => GenerateJUnitReport(results, settings),
                    _ => GenerateTextReport(results, settings)
                };

                await System.IO.File.WriteAllTextAsync(settings.Output!, reportContent);

                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[grey]Report written to: {settings.Output}[/]");
                }
            }
            catch (Exception ex)
            {
                AnsiConsole.MarkupLine($"[yellow]Warning: Could not write report: {ex.Message}[/]");
            }
        }

        private string GenerateTextReport(List<TestResult> results, ServerCommandSettings settings)
        {
            var sb = new System.Text.StringBuilder();
            sb.AppendLine($"MBRCPVAL Test Report");
            sb.AppendLine($"====================");
            sb.AppendLine($"Host: {settings.Host}:{settings.Port}");
            sb.AppendLine($"Protocol: {settings.Protocol}");
            sb.AppendLine($"Date: {DateTime.Now:yyyy-MM-dd HH:mm:ss}");
            sb.AppendLine();

            var passed = results.Count(r => r.Passed);
            var failed = results.Count(r => r.Failed);
            sb.AppendLine($"Results: {passed} passed, {failed} failed, {results.Count} total");
            sb.AppendLine();

            foreach (var result in results)
            {
                var status = result.Status.ToString().ToUpper();
                sb.AppendLine($"[{status}] {result.TestCase.DisplayName} ({result.Duration.TotalMilliseconds:F0}ms)");
                if (!string.IsNullOrEmpty(result.ErrorMessage))
                {
                    sb.AppendLine($"  Error: {result.ErrorMessage}");
                }
            }

            return sb.ToString();
        }

        private string GenerateJsonReport(List<TestResult> results, ServerCommandSettings settings)
        {
            var report = new
            {
                host = $"{settings.Host}:{settings.Port}",
                protocol = settings.Protocol,
                timestamp = DateTime.UtcNow.ToString("o"),
                summary = new
                {
                    total = results.Count,
                    passed = results.Count(r => r.Passed),
                    failed = results.Count(r => r.Failed),
                    skipped = results.Count(r => r.Skipped),
                    errors = results.Count(r => r.Errored)
                },
                tests = results.Select(r => new
                {
                    id = r.TestCase.Id,
                    name = r.TestCase.Name,
                    category = r.TestCase.Category,
                    status = r.Status.ToString().ToLower(),
                    duration_ms = r.Duration.TotalMilliseconds,
                    error = r.ErrorMessage
                })
            };

            return System.Text.Json.JsonSerializer.Serialize(report, new System.Text.Json.JsonSerializerOptions { WriteIndented = true });
        }

        private string GenerateHtmlReport(List<TestResult> results, ServerCommandSettings settings)
        {
            var passed = results.Count(r => r.Passed);
            var failed = results.Count(r => r.Failed);
            var passRate = results.Count > 0 ? (passed * 100.0 / results.Count) : 0;

            var sb = new System.Text.StringBuilder();
            sb.AppendLine("<!DOCTYPE html><html><head><title>MBRCPVAL Report</title>");
            sb.AppendLine("<style>body{font-family:sans-serif;margin:20px}table{border-collapse:collapse;width:100%}th,td{border:1px solid #ddd;padding:8px;text-align:left}.pass{color:green}.fail{color:red}.skip{color:orange}</style>");
            sb.AppendLine("</head><body>");
            sb.AppendLine($"<h1>MBRCPVAL Test Report</h1>");
            sb.AppendLine($"<p>Host: {settings.Host}:{settings.Port} | Protocol: {settings.Protocol} | {DateTime.Now:yyyy-MM-dd HH:mm:ss}</p>");
            sb.AppendLine($"<p><strong>Results:</strong> {passed} passed, {failed} failed ({passRate:F1}% pass rate)</p>");
            sb.AppendLine("<table><tr><th>Status</th><th>ID</th><th>Name</th><th>Category</th><th>Duration</th><th>Error</th></tr>");

            foreach (var result in results)
            {
                var statusClass = result.Passed ? "pass" : (result.Failed ? "fail" : "skip");
                sb.AppendLine($"<tr><td class='{statusClass}'>{result.Status}</td><td>{result.TestCase.Id}</td><td>{result.TestCase.Name}</td><td>{result.TestCase.Category}</td><td>{result.Duration.TotalMilliseconds:F0}ms</td><td>{result.ErrorMessage ?? ""}</td></tr>");
            }

            sb.AppendLine("</table></body></html>");
            return sb.ToString();
        }

        private string GenerateJUnitReport(List<TestResult> results, ServerCommandSettings settings)
        {
            var sb = new System.Text.StringBuilder();
            var totalTime = results.Sum(r => r.Duration.TotalSeconds);
            var failures = results.Count(r => r.Failed);
            var errors = results.Count(r => r.Errored);

            sb.AppendLine("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
            sb.AppendLine($"<testsuite name=\"MBRCPVAL\" tests=\"{results.Count}\" failures=\"{failures}\" errors=\"{errors}\" time=\"{totalTime:F3}\">");

            foreach (var result in results)
            {
                sb.AppendLine($"  <testcase name=\"{result.TestCase.Name}\" classname=\"{result.TestCase.Category}\" time=\"{result.Duration.TotalSeconds:F3}\">");
                if (result.Failed)
                {
                    sb.AppendLine($"    <failure message=\"{System.Security.SecurityElement.Escape(result.ErrorMessage ?? "Test failed")}\"/>");
                }
                else if (result.Errored)
                {
                    sb.AppendLine($"    <error message=\"{System.Security.SecurityElement.Escape(result.ErrorMessage ?? "Test error")}\"/>");
                }
                else if (result.Skipped)
                {
                    sb.AppendLine($"    <skipped message=\"{System.Security.SecurityElement.Escape(result.SkipReason ?? "Skipped")}\"/>");
                }
                sb.AppendLine("  </testcase>");
            }

            sb.AppendLine("</testsuite>");
            return sb.ToString();
        }

        /// <summary>
        /// Detect the protocol version and plugin settings from the server.
        /// Uses proper handshake sequence: player first, then protocol.
        /// </summary>
        /// <returns>Tuple of (protocol version, ARiA enabled, Experimental enabled)</returns>
        private async Task<(string? protocol, bool? ariaEnabled, bool? experimentalEnabled)> DetectProtocolAndSettingsAsync(ServerCommandSettings settings)
        {
            string? protocolVersion = null;
            bool? ariaEnabled = null;
            bool? experimentalEnabled = null;

            try
            {
                using var client = new TcpMbrcSocketClient();

                await AnsiConsole.Status()
                    .Spinner(Spinner.Known.Dots)
                    .SpinnerStyle(Style.Parse("blue"))
                    .StartAsync("Connecting to server...", async ctx =>
                    {
                        await client.ConnectAsync(settings.Host!, settings.Port);
                    });

                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine("[green]Connected[/]");
                }

                using var cts = new System.Threading.CancellationTokenSource(TimeSpan.FromSeconds(15));

                // CRITICAL: Old plugin requires strict handshake sequence
                // Packet 0 MUST be 'player' context, otherwise server disconnects!
                await client.SendAsync("{\"context\":\"player\",\"data\":\"\"}", cts.Token);
                await client.ReceiveAsync("player", cts.Token);

                // Packet 1: Now send protocol request with v4.5 to negotiate highest version
                await client.SendAsync("{\"context\":\"protocol\",\"data\":{\"protocol_version\":4.5}}", cts.Token);

                // Wait for protocol response
                var response = await client.ReceiveAsync("protocol", cts.Token);

                if (response?.Data != null)
                {
                    var dataStr = response.Data.ToString();
                    if (dataStr != null)
                    {
                        // Plugin returns just a number: {"context":"protocol","data":4} or {"data":4.5}
                        if (double.TryParse(dataStr, out var version))
                        {
                            protocolVersion = version.ToString();
                        }
                        else if (int.TryParse(dataStr, out var intVersion))
                        {
                            protocolVersion = intVersion.ToString();
                        }
                    }
                }

                // Detect ARiA by querying pluginstoragepath
                // When ARiA disabled: {"error": "Not Enabled", "message": "ARiA must be enabled..."}
                // When ARiA enabled: {"path": "..."}
                try
                {
                    await client.SendAsync("{\"context\":\"pluginstoragepath\",\"data\":null}", cts.Token);
                    using var ariaCts = new System.Threading.CancellationTokenSource(TimeSpan.FromSeconds(3));
                    var ariaResponse = await client.ReceiveAsync("pluginstoragepath", ariaCts.Token);

                    if (ariaResponse?.Context == "pluginstoragepath" && ariaResponse.Data != null)
                    {
                        var dataStr = ariaResponse.Data.ToString();
                        if (dataStr != null && dataStr.Contains("Not Enabled", StringComparison.OrdinalIgnoreCase))
                        {
                            ariaEnabled = false;
                        }
                        else
                        {
                            ariaEnabled = true;
                        }
                    }
                }
                catch
                {
                    // Can't determine - might be old plugin without this context
                    ariaEnabled = null;
                }

                // Detect experimental features by querying visualizer list
                // When disabled, plugin returns: {"error": "Not Enabled", "message": "...", "visualizers": []}
                // When enabled, plugin returns: {"visualizers": [...], "default": "...", "state": "..."}
                try
                {
                    await client.SendAsync("{\"context\":\"playervisualizerlist\",\"data\":null}", cts.Token);

                    // Wait briefly for response
                    using var expCts = new System.Threading.CancellationTokenSource(TimeSpan.FromSeconds(3));
                    var vizResponse = await client.ReceiveAsync("playervisualizerlist", expCts.Token);

                    if (vizResponse?.Context == "playervisualizerlist" && vizResponse.Data != null)
                    {
                        var dataStr = vizResponse.Data.ToString();
                        // Check for "Not Enabled" error indicating experimental features are disabled
                        if (dataStr != null && dataStr.Contains("Not Enabled", StringComparison.OrdinalIgnoreCase))
                        {
                            experimentalEnabled = false;
                        }
                        else
                        {
                            experimentalEnabled = true;
                        }
                    }
                }
                catch
                {
                    // Timeout or error - can't determine status
                    experimentalEnabled = null;
                }

                await client.DisconnectAsync();
            }
            catch (Exception ex)
            {
                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[yellow]Detection failed: {ex.Message}[/]");
                }
            }

            return (protocolVersion, ariaEnabled, experimentalEnabled);
        }

        /// <summary>
        /// Escape Spectre.Console markup characters in text.
        /// </summary>
        private static string EscapeMarkup(string text) => text.Replace("[", "[[").Replace("]", "]]");

        /// <summary>
        /// List all available tests with their properties.
        /// </summary>
        private int ListAllTests(ServerCommandSettings settings)
        {
            var testCases = TestSuiteFactory.CreateTestSuite(settings.Protocol, settings.Suite);

            AnsiConsole.MarkupLine($"[bold]Available Tests for Protocol {settings.Protocol}[/]");
            AnsiConsole.MarkupLine($"Suite: [cyan]{settings.Suite}[/]");
            AnsiConsole.WriteLine();

            // Priority legend
            AnsiConsole.MarkupLine("[bold]Priority Levels:[/]");
            AnsiConsole.MarkupLine("  [green]1-10[/]: Critical (core protocol, must pass)");
            AnsiConsole.MarkupLine("  [yellow]11-30[/]: Important (key features)");
            AnsiConsole.MarkupLine("  [blue]31-60[/]: Standard (normal functionality)");
            AnsiConsole.MarkupLine("  [grey]61+[/]: Optional (edge cases, stress tests)");
            AnsiConsole.WriteLine();

            // Group by category
            var categories = testCases.GroupBy(t => t.Category).OrderBy(g => g.Key);

            foreach (var category in categories)
            {
                AnsiConsole.MarkupLine($"[bold underline]{category.Key.ToUpper()}[/]");

                var table = new Table()
                    .Border(TableBorder.Simple)
                    .AddColumn("ID")
                    .AddColumn("Name")
                    .AddColumn("Pri")
                    .AddColumn("Tags")
                    .AddColumn("Status");

                foreach (var test in category.OrderBy(t => t.Priority))
                {
                    var id = EscapeMarkup(test.Id);
                    var name = EscapeMarkup(test.Name);

                    // Color-code priority
                    string priColor = test.Priority switch
                    {
                        <= 10 => "green",
                        <= 30 => "yellow",
                        <= 60 => "blue",
                        _ => "grey"
                    };
                    var pri = $"[{priColor}]{test.Priority}[/]";

                    var tags = string.Join(", ", test.Tags.Take(3));
                    if (test.Tags.Count > 3) tags += "...";

                    var status = test.Skip ? "[yellow]SKIP[/]" : "[green]AUTO[/]";

                    table.AddRow(id, name, pri, tags, status);
                }

                AnsiConsole.Write(table);
                AnsiConsole.WriteLine();
            }

            // Summary
            var total = testCases.Count;
            var auto = testCases.Count(t => !t.Skip);
            var skip = testCases.Count(t => t.Skip);
            var critical = testCases.Count(t => t.Priority <= 10);
            var important = testCases.Count(t => t.Priority > 10 && t.Priority <= 30);
            var standard = testCases.Count(t => t.Priority > 30 && t.Priority <= 60);
            var optional = testCases.Count(t => t.Priority > 60);

            AnsiConsole.MarkupLine("[bold]Summary:[/]");
            AnsiConsole.MarkupLine($"  Total: {total} | Auto: [green]{auto}[/] | Skip: [yellow]{skip}[/]");
            AnsiConsole.MarkupLine($"  Critical: [green]{critical}[/] | Important: [yellow]{important}[/] | Standard: [blue]{standard}[/] | Optional: [grey]{optional}[/]");

            return 0;
        }
    }
}
