using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Net.Sockets;
using System.Text;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using Mbrcpval.CLI.Settings;
using Spectre.Console;
using Spectre.Console.Cli;

namespace Mbrcpval.CLI.Commands
{
    /// <summary>
    /// Settings for the REPL command.
    /// </summary>
    public class ReplCommandSettings : CommonSettings
    {
        /// <summary>
        /// The host/IP address of the MBRC server.
        /// </summary>
        [CommandOption("-H|--host <HOST>")]
        [Description("The host/IP address of the MBRC server (required)")]
        public string? Host { get; set; }

        /// <summary>
        /// The port number of the MBRC server.
        /// </summary>
        [CommandOption("-p|--port <PORT>")]
        [Description("The port number of the MBRC server")]
        [DefaultValue(3000)]
        public int Port { get; set; } = 3000;

        /// <summary>
        /// The protocol version.
        /// </summary>
        [CommandOption("--protocol <VERSION>")]
        [Description("Protocol version (4.0 or 4.5)")]
        [DefaultValue("4.5")]
        public string Protocol { get; set; } = "4.5";

        /// <summary>
        /// History file path.
        /// </summary>
        [CommandOption("--history <FILE>")]
        [Description("Path to command history file")]
        public string? HistoryFile { get; set; }

        /// <summary>
        /// Connect automatically on start.
        /// </summary>
        [CommandOption("--auto-connect")]
        [Description("Connect automatically on start")]
        [DefaultValue(true)]
        public bool AutoConnect { get; set; } = true;

        /// <summary>
        /// Validate the settings.
        /// </summary>
        public override ValidationResult Validate()
        {
            var baseResult = base.Validate();
            if (!baseResult.Successful)
            {
                return baseResult;
            }

            if (string.IsNullOrWhiteSpace(Host))
            {
                return ValidationResult.Error("Host is required. Use --host <HOST> to specify the server address.");
            }

            if (Port < 1 || Port > 65535)
            {
                return ValidationResult.Error("Port must be between 1 and 65535.");
            }

            var validProtocols = new[] { "4.0", "4.5" };
            if (!Array.Exists(validProtocols, p => p.Equals(Protocol, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidationResult.Error($"Invalid protocol version '{Protocol}'. Valid options: {string.Join(", ", validProtocols)}");
            }

            return ValidationResult.Success();
        }
    }

    /// <summary>
    /// Command for interactive REPL session with an MBRC server.
    /// Thread-safe implementation with proper state synchronization.
    /// </summary>
    public class ReplCommand : AsyncCommand<ReplCommandSettings>
    {
        private TcpClient? _client;
        private NetworkStream? _stream;
        private CancellationTokenSource? _listenerCts;
        private readonly List<string> _history = new();
        private readonly object _stateLock = new();  // Synchronizes state access
        private volatile bool _isConnected;   // Volatile for cross-thread visibility
        private volatile bool _isListening;   // Volatile for cross-thread visibility

        /// <summary>
        /// Execute the REPL command.
        /// </summary>
        /// <param name="context">The command context.</param>
        /// <param name="settings">The command settings.</param>
        /// <returns>Exit code.</returns>
        public override async Task<int> ExecuteAsync(CommandContext context, ReplCommandSettings settings)
        {
            // Load history if specified
            if (!string.IsNullOrEmpty(settings.HistoryFile) && File.Exists(settings.HistoryFile))
            {
                try
                {
                    var historyLines = await File.ReadAllLinesAsync(settings.HistoryFile);
                    _history.AddRange(historyLines);
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"[grey]Loaded {_history.Count} history entries[/]");
                    }
                }
                catch
                {
                    // Ignore history load errors
                }
            }

            if (!settings.Quiet)
            {
                AnsiConsole.MarkupLine("[bold]MBRCPVAL Interactive REPL[/]");
                AnsiConsole.MarkupLine($"  Target: [cyan]{settings.Host}:{settings.Port}[/]");
                AnsiConsole.MarkupLine($"  Protocol: [cyan]{settings.Protocol}[/]");
                AnsiConsole.WriteLine();
                AnsiConsole.MarkupLine("[grey]Type .help for available commands, .exit to quit[/]");
                AnsiConsole.WriteLine();
            }

            try
            {
                // Auto-connect if requested
                if (settings.AutoConnect)
                {
                    await ConnectAsync(settings);
                }

                // Main REPL loop
                while (true)
                {
                    var prompt = _isConnected
                        ? $"[green]{settings.Host}:{settings.Port}>[/] "
                        : "[yellow]disconnected>[/] ";

                    AnsiConsole.Markup(prompt);
                    var input = Console.ReadLine()?.Trim();

                    if (string.IsNullOrEmpty(input))
                    {
                        continue;
                    }

                    // Add to history
                    if (!input.StartsWith(".exit", StringComparison.OrdinalIgnoreCase))
                    {
                        _history.Add(input);
                    }

                    // Process commands
                    if (input.StartsWith("."))
                    {
                        var handled = await ProcessCommandAsync(input, settings);
                        if (!handled)
                        {
                            break; // Exit requested
                        }
                    }
                    else
                    {
                        // Treat as raw JSON if connected
                        if (_isConnected)
                        {
                            await SendRawAsync(input, settings);
                        }
                        else
                        {
                            AnsiConsole.MarkupLine("[yellow]Not connected. Use .connect to connect first.[/]");
                        }
                    }
                }

                // Save history if specified
                if (!string.IsNullOrEmpty(settings.HistoryFile))
                {
                    try
                    {
                        // Keep last 1000 entries
                        var toSave = _history.Count > 1000 ? _history.GetRange(_history.Count - 1000, 1000) : _history;
                        await File.WriteAllLinesAsync(settings.HistoryFile, toSave);
                    }
                    catch
                    {
                        // Ignore history save errors
                    }
                }
            }
            finally
            {
                // Always cleanup resources
                await DisconnectAsync(settings);
            }

            return 0;
        }

        /// <summary>
        /// Process a REPL command.
        /// </summary>
        private async Task<bool> ProcessCommandAsync(string input, ReplCommandSettings settings)
        {
            var parts = input.Split(' ', 2, StringSplitOptions.RemoveEmptyEntries);
            var command = parts[0].ToLowerInvariant();
            var args = parts.Length > 1 ? parts[1] : string.Empty;

            switch (command)
            {
                case ".help":
                    ShowHelp();
                    return true;

                case ".exit":
                case ".quit":
                case ".q":
                    return false; // Signal exit

                case ".connect":
                    await ConnectAsync(settings);
                    return true;

                case ".disconnect":
                case ".close":
                    await DisconnectAsync(settings);
                    return true;

                case ".send":
                    await SendContextMessageAsync(args, settings);
                    return true;

                case ".raw":
                    await SendRawAsync(args, settings);
                    return true;

                case ".listen":
                    await ToggleListenModeAsync(settings);
                    return true;

                case ".status":
                    ShowStatus(settings);
                    return true;

                case ".history":
                    ShowHistory();
                    return true;

                case ".clear":
                    AnsiConsole.Clear();
                    return true;

                default:
                    AnsiConsole.MarkupLine($"[yellow]Unknown command: {command}. Type .help for available commands.[/]");
                    return true;
            }
        }

        /// <summary>
        /// Show help information.
        /// </summary>
        private void ShowHelp()
        {
            var table = new Table()
                .Border(TableBorder.Rounded)
                .AddColumn("Command")
                .AddColumn("Description");

            table.AddRow(".connect", "Connect to the MBRC server");
            table.AddRow(".disconnect", "Disconnect from the server");
            table.AddRow(".send <context> [data]", "Send a message with context and optional data");
            table.AddRow(".raw <json>", "Send raw JSON message");
            table.AddRow(".listen", "Toggle background listening mode");
            table.AddRow(".status", "Show connection status");
            table.AddRow(".history", "Show command history");
            table.AddRow(".clear", "Clear the screen");
            table.AddRow(".help", "Show this help");
            table.AddRow(".exit", "Exit the REPL");

            AnsiConsole.Write(table);

            AnsiConsole.WriteLine();
            AnsiConsole.MarkupLine("[bold]Examples:[/]");
            AnsiConsole.MarkupLine("  [grey].send player[/]                      - Get player state");
            AnsiConsole.MarkupLine("  [grey].send player PlayPause[/]            - Toggle play/pause");
            AnsiConsole.MarkupLine("  [grey].send playervolume 80[/]             - Set volume to 80");
            AnsiConsole.MarkupLine("  [grey].raw {\"context\":\"ping\"}[/]           - Send raw ping");
        }

        /// <summary>
        /// Connect to the server.
        /// </summary>
        private async Task ConnectAsync(ReplCommandSettings settings)
        {
            if (_isConnected)
            {
                AnsiConsole.MarkupLine("[yellow]Already connected. Use .disconnect first.[/]");
                return;
            }

            try
            {
                await AnsiConsole.Status()
                    .Spinner(Spinner.Known.Dots)
                    .StartAsync($"Connecting to {settings.Host}:{settings.Port}...", async ctx =>
                    {
                        _client = new TcpClient();
                        await _client.ConnectAsync(settings.Host!, settings.Port);
                        _stream = _client.GetStream();
                        _isConnected = true;
                    });

                AnsiConsole.MarkupLine($"[green]Connected to {settings.Host}:{settings.Port}[/]");

                // Send protocol init
                var initMessage = $"{{\"context\":\"protocol\",\"data\":{{\"protocol_version\":\"{settings.Protocol}\"}}}}";
                await SendRawAsync(initMessage, settings, silent: true);
            }
            catch (Exception ex)
            {
                AnsiConsole.MarkupLine($"[red]Connection failed: {ex.Message}[/]");
                _isConnected = false;
            }
        }

        /// <summary>
        /// Disconnect from the server and dispose all resources.
        /// </summary>
        private async Task DisconnectAsync(ReplCommandSettings settings)
        {
            // Stop listener if running
            if (_isListening)
            {
                _listenerCts?.Cancel();
                _listenerCts?.Dispose();
                _listenerCts = null;
                _isListening = false;
            }

            if (!_isConnected)
            {
                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine("[yellow]Not connected.[/]");
                }
                return;
            }

            try
            {
                // Send close message
                await SendRawAsync("{\"context\":\"close\"}", settings, silent: true);
            }
            catch
            {
                // Ignore close errors
            }

            // Dispose resources properly
            _stream?.Dispose();
            _client?.Dispose();
            _stream = null;
            _client = null;
            _isConnected = false;

            if (!settings.Quiet)
            {
                AnsiConsole.MarkupLine("[grey]Disconnected[/]");
            }
        }

        /// <summary>
        /// Send a context message.
        /// </summary>
        private async Task SendContextMessageAsync(string args, ReplCommandSettings settings)
        {
            if (!_isConnected)
            {
                AnsiConsole.MarkupLine("[yellow]Not connected. Use .connect first.[/]");
                return;
            }

            var parts = args.Split(' ', 2, StringSplitOptions.RemoveEmptyEntries);
            if (parts.Length == 0)
            {
                AnsiConsole.MarkupLine("[yellow]Usage: .send <context> [data][/]");
                return;
            }

            var context = parts[0];
            var data = parts.Length > 1 ? parts[1] : string.Empty;

            // Build message
            string message;
            if (string.IsNullOrEmpty(data))
            {
                message = $"{{\"context\":\"{context}\"}}";
            }
            else if (data.StartsWith("{") || data.StartsWith("["))
            {
                // Data is JSON
                message = $"{{\"context\":\"{context}\",\"data\":{data}}}";
            }
            else
            {
                // Data is a simple string
                message = $"{{\"context\":\"{context}\",\"data\":\"{data}\"}}";
            }

            await SendRawAsync(message, settings);
        }

        /// <summary>
        /// Send a raw JSON message.
        /// </summary>
        private async Task SendRawAsync(string json, ReplCommandSettings settings, bool silent = false)
        {
            if (!_isConnected || _stream == null)
            {
                if (!silent)
                {
                    AnsiConsole.MarkupLine("[yellow]Not connected. Use .connect first.[/]");
                }
                return;
            }

            try
            {
                // Validate JSON
                try
                {
                    JsonDocument.Parse(json);
                }
                catch (JsonException)
                {
                    if (!silent)
                    {
                        AnsiConsole.MarkupLine("[red]Invalid JSON[/]");
                    }
                    return;
                }

                // Send message
                var bytes = Encoding.UTF8.GetBytes(json + "\r\n");
                await _stream.WriteAsync(bytes);

                if (settings.Verbose && !silent)
                {
                    AnsiConsole.MarkupLine($"[grey]>>> {EscapeMarkup(json)}[/]");
                }

                // Wait for response
                await ReceiveResponseAsync(settings, silent);
            }
            catch (Exception ex)
            {
                if (!silent)
                {
                    AnsiConsole.MarkupLine($"[red]Send failed: {ex.Message}[/]");
                }
                _isConnected = false;
            }
        }

        /// <summary>
        /// Receive and display response.
        /// </summary>
        private async Task ReceiveResponseAsync(ReplCommandSettings settings, bool silent = false)
        {
            if (_stream == null) return;

            try
            {
                var buffer = new byte[8192];
                using var cts = new CancellationTokenSource(TimeSpan.FromSeconds(5));

                // Check if data is available
                _stream.ReadTimeout = 2000;
                var bytesRead = await _stream.ReadAsync(buffer, 0, buffer.Length, cts.Token);

                if (bytesRead > 0)
                {
                    var response = Encoding.UTF8.GetString(buffer, 0, bytesRead).Trim();

                    if (!silent)
                    {
                        PrettyPrintResponse(response, settings);
                    }
                }
            }
            catch (OperationCanceledException)
            {
                // Timeout - no response
            }
            catch (IOException)
            {
                // Read timeout
            }
        }

        /// <summary>
        /// Pretty-print a response.
        /// </summary>
        private void PrettyPrintResponse(string response, ReplCommandSettings settings)
        {
            // Split by newlines in case of multiple messages
            var messages = response.Split(new[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);

            foreach (var msg in messages)
            {
                try
                {
                    var doc = JsonDocument.Parse(msg);
                    var formatted = JsonSerializer.Serialize(doc.RootElement,
                        new JsonSerializerOptions { WriteIndented = true });

                    // Extract context for coloring
                    var context = doc.RootElement.TryGetProperty("context", out var ctx) ? ctx.GetString() : null;

                    var color = context?.ToLowerInvariant() switch
                    {
                        "error" => Color.Red,
                        "pong" => Color.Yellow,
                        "playerstate" or "playerstatus" => Color.Green,
                        "nowplayingtrack" => Color.Aqua,
                        _ => Color.White
                    };

                    AnsiConsole.Write(new Panel(formatted)
                        .Header($"[{color.ToMarkup()}]{context ?? "response"}[/]")
                        .Border(BoxBorder.Rounded)
                        .BorderColor(color));
                }
                catch
                {
                    // Not valid JSON, print as-is
                    AnsiConsole.MarkupLine($"[grey]{EscapeMarkup(msg)}[/]");
                }
            }
        }

        /// <summary>
        /// Toggle background listening mode.
        /// </summary>
        private async Task ToggleListenModeAsync(ReplCommandSettings settings)
        {
            if (!_isConnected)
            {
                AnsiConsole.MarkupLine("[yellow]Not connected. Use .connect first.[/]");
                return;
            }

            if (_isListening)
            {
                _listenerCts?.Cancel();
                _isListening = false;
                AnsiConsole.MarkupLine("[grey]Listening mode disabled[/]");
            }
            else
            {
                _listenerCts = new CancellationTokenSource();
                _isListening = true;
                AnsiConsole.MarkupLine("[green]Listening mode enabled. Press .listen again to stop.[/]");

                // Start background listener
                _ = BackgroundListenAsync(settings, _listenerCts.Token);
            }

            await Task.CompletedTask;
        }

        /// <summary>
        /// Background listener for incoming messages.
        /// </summary>
        private async Task BackgroundListenAsync(ReplCommandSettings settings, CancellationToken ct)
        {
            var buffer = new byte[8192];

            while (!ct.IsCancellationRequested && _isConnected && _stream != null)
            {
                try
                {
                    if (_stream.DataAvailable)
                    {
                        var bytesRead = await _stream.ReadAsync(buffer, 0, buffer.Length, ct);
                        if (bytesRead > 0)
                        {
                            var message = Encoding.UTF8.GetString(buffer, 0, bytesRead).Trim();
                            AnsiConsole.WriteLine();
                            AnsiConsole.MarkupLine("[cyan]<<< Incoming message:[/]");
                            PrettyPrintResponse(message, settings);
                        }
                    }

                    await Task.Delay(100, ct);
                }
                catch (OperationCanceledException)
                {
                    break;
                }
                catch
                {
                    // Ignore read errors
                }
            }
        }

        /// <summary>
        /// Show connection status.
        /// </summary>
        private void ShowStatus(ReplCommandSettings settings)
        {
            var table = new Table()
                .Border(TableBorder.Rounded)
                .AddColumn("Property")
                .AddColumn("Value");

            table.AddRow("Host", settings.Host ?? "not set");
            table.AddRow("Port", settings.Port.ToString());
            table.AddRow("Protocol", settings.Protocol);
            table.AddRow("Connected", _isConnected ? "[green]Yes[/]" : "[red]No[/]");
            table.AddRow("Listening", _isListening ? "[green]Yes[/]" : "[grey]No[/]");
            table.AddRow("History Entries", _history.Count.ToString());

            AnsiConsole.Write(table);
        }

        /// <summary>
        /// Show command history.
        /// </summary>
        private void ShowHistory()
        {
            if (_history.Count == 0)
            {
                AnsiConsole.MarkupLine("[grey]No history[/]");
                return;
            }

            var recentHistory = _history.Count > 20 ? _history.GetRange(_history.Count - 20, 20) : _history;
            var startIndex = _history.Count - recentHistory.Count;

            for (int i = 0; i < recentHistory.Count; i++)
            {
                AnsiConsole.MarkupLine($"[grey]{startIndex + i + 1}:[/] {EscapeMarkup(recentHistory[i])}");
            }
        }

        /// <summary>
        /// Escape Spectre.Console markup characters.
        /// </summary>
        private static string EscapeMarkup(string text)
        {
            return text.Replace("[", "[[").Replace("]", "]]");
        }
    }
}
