using System;
using System.Collections.Concurrent;
using System.ComponentModel;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Text.Json;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using Mbrcpval.CLI.Settings;
using Mbrcpval.Core;
using Spectre.Console;
using Spectre.Console.Cli;

namespace Mbrcpval.CLI.Commands
{
    /// <summary>
    /// Output format for proxy logging.
    /// </summary>
    public enum ProxyFormat
    {
        /// <summary>Pretty-printed JSON with colors.</summary>
        Pretty,
        /// <summary>Raw JSON one line per message.</summary>
        Json,
        /// <summary>Raw unformatted output.</summary>
        Raw
    }

    /// <summary>
    /// Settings for the proxy command.
    /// </summary>
    public class ProxyCommandSettings : CommonSettings
    {
        /// <summary>
        /// The port to listen on.
        /// </summary>
        [CommandOption("-l|--listen <PORT>")]
        [Description("The port to listen on for client connections")]
        [DefaultValue(3001)]
        public int ListenPort { get; set; } = 3001;

        /// <summary>
        /// The target server (host:port).
        /// </summary>
        [CommandOption("-t|--target <TARGET>")]
        [Description("The target MBRC server (host:port)")]
        public string? Target { get; set; }

        /// <summary>
        /// Output file for logging.
        /// </summary>
        [CommandOption("-o|--output <FILE>")]
        [Description("Output file for message logging")]
        public string? Output { get; set; }

        /// <summary>
        /// Output format.
        /// </summary>
        [CommandOption("-f|--format <FORMAT>")]
        [Description("Output format (pretty, json, raw)")]
        [DefaultValue(ProxyFormat.Pretty)]
        public ProxyFormat Format { get; set; } = ProxyFormat.Pretty;

        /// <summary>
        /// Context filter pattern (regex).
        /// </summary>
        [CommandOption("--filter <PATTERN>")]
        [Description("Context filter pattern (regex)")]
        public string? Filter { get; set; }

        /// <summary>
        /// Validate messages against schema.
        /// </summary>
        [CommandOption("--validate")]
        [Description("Validate messages against protocol schema")]
        [DefaultValue(false)]
        public bool ValidateMessages { get; set; }

        /// <summary>
        /// Show timestamps in output.
        /// </summary>
        [CommandOption("--timestamps")]
        [Description("Show timestamps in output")]
        [DefaultValue(true)]
        public bool Timestamps { get; set; } = true;

        /// <summary>
        /// Bind to all interfaces.
        /// </summary>
        [CommandOption("--bind-all")]
        [Description("Bind to all network interfaces")]
        [DefaultValue(false)]
        public bool BindAll { get; set; }

        /// <summary>
        /// Validate the settings.
        /// </summary>
        public override ValidationResult Validate()
        {
            var baseResult = base.Validate();
            if (!baseResult.Successful)
            {
                return baseResult;
            }

            if (ListenPort < 1 || ListenPort > 65535)
            {
                return ValidationResult.Error("Listen port must be between 1 and 65535.");
            }

            if (string.IsNullOrWhiteSpace(Target))
            {
                return ValidationResult.Error("Target is required. Use --target <HOST:PORT> to specify the server.");
            }

            // Parse target
            var targetParts = Target.Split(':');
            if (targetParts.Length != 2 || !int.TryParse(targetParts[1], out var targetPort) || targetPort < 1 || targetPort > 65535)
            {
                return ValidationResult.Error("Target must be in format HOST:PORT (e.g., localhost:3000).");
            }

            // Validate filter regex
            if (!string.IsNullOrEmpty(Filter))
            {
                try
                {
                    new Regex(Filter);
                }
                catch (ArgumentException)
                {
                    return ValidationResult.Error($"Invalid filter regex: {Filter}");
                }
            }

            return ValidationResult.Success();
        }

        /// <summary>
        /// Get target host.
        /// </summary>
        public string TargetHost => Target?.Split(':')[0] ?? "localhost";

        /// <summary>
        /// Get target port.
        /// </summary>
        public int TargetPort => int.TryParse(Target?.Split(':')[1], out var port) ? port : 3000;
    }

    /// <summary>
    /// Command to start a proxy for intercepting and logging MBRC messages.
    /// </summary>
    public class ProxyCommand : AsyncCommand<ProxyCommandSettings>
    {
        private TcpListener? _listener;
        private CancellationTokenSource? _cts;
        private StreamWriter? _logWriter;
        private readonly SemaphoreSlim _logWriterLock = new(1, 1);
        private readonly ConcurrentDictionary<int, ProxySession> _sessions = new();
        private int _sessionCount;        // Accessed via Interlocked.Increment
        private long _messageCount;       // Accessed via Interlocked.Increment
        private Regex? _filterRegex;
        private SchemaValidator? _schemaValidator;

        /// <summary>
        /// Execute the proxy command.
        /// </summary>
        /// <param name="context">The command context.</param>
        /// <param name="settings">The command settings.</param>
        /// <returns>Exit code.</returns>
        public override async Task<int> ExecuteAsync(CommandContext context, ProxyCommandSettings settings)
        {
            _cts = new CancellationTokenSource();

            // Handle Ctrl+C
            Console.CancelKeyPress += (sender, e) =>
            {
                e.Cancel = true;
                AnsiConsole.MarkupLine("\n[yellow]Shutting down proxy...[/]");
                _cts.Cancel();
            };

            // Compile filter regex with timeout to prevent ReDoS attacks
            if (!string.IsNullOrEmpty(settings.Filter))
            {
                _filterRegex = new Regex(settings.Filter, RegexOptions.IgnoreCase | RegexOptions.Compiled, TimeSpan.FromSeconds(1));
            }

            // Open log file if specified
            if (!string.IsNullOrEmpty(settings.Output))
            {
                try
                {
                    _logWriter = new StreamWriter(settings.Output, append: true) { AutoFlush = true };
                }
                catch (Exception ex)
                {
                    AnsiConsole.MarkupLine($"[red]Error opening log file: {ex.Message}[/]");
                    return 2;
                }
            }

            var bindAddress = settings.BindAll ? IPAddress.Any : IPAddress.Loopback;

            if (!settings.Quiet)
            {
                AnsiConsole.MarkupLine("[bold]Starting MBRC Proxy[/]");
                AnsiConsole.MarkupLine($"  Listen: [cyan]{(settings.BindAll ? "0.0.0.0" : "127.0.0.1")}:{settings.ListenPort}[/]");
                AnsiConsole.MarkupLine($"  Target: [cyan]{settings.Target}[/]");
                AnsiConsole.MarkupLine($"  Format: [cyan]{settings.Format}[/]");
                if (!string.IsNullOrEmpty(settings.Filter))
                {
                    AnsiConsole.MarkupLine($"  Filter: [cyan]{settings.Filter}[/]");
                }
                if (!string.IsNullOrEmpty(settings.Output))
                {
                    AnsiConsole.MarkupLine($"  Log File: [cyan]{settings.Output}[/]");
                }
                if (settings.ValidateMessages)
                {
                    AnsiConsole.MarkupLine($"  Validation: [cyan]Enabled[/]");
                }
                AnsiConsole.WriteLine();
                AnsiConsole.MarkupLine("[grey]Press Ctrl+C to stop the proxy[/]");
                AnsiConsole.WriteLine();
            }

            // Initialize schema validator if validation is enabled
            if (settings.ValidateMessages)
            {
                try
                {
                    _schemaValidator = await SchemaValidator.CreateAsync();
                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine("[grey]Schema validator initialized[/]");
                    }
                }
                catch (Exception ex)
                {
                    AnsiConsole.MarkupLine($"[yellow]Warning: Could not initialize schema validator: {ex.Message}[/]");
                    AnsiConsole.MarkupLine("[yellow]Validation will be disabled[/]");
                }
            }

            try
            {
                _listener = new TcpListener(bindAddress, settings.ListenPort);
                _listener.Start();

                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine($"[green]Proxy listening on {bindAddress}:{settings.ListenPort}[/]");
                    AnsiConsole.MarkupLine($"[grey]Forwarding to {settings.Target}[/]");
                    AnsiConsole.WriteLine();
                }

                // Accept connections
                while (!_cts.Token.IsCancellationRequested)
                {
                    try
                    {
                        var client = await _listener.AcceptTcpClientAsync();
                        _ = HandleClientAsync(client, settings, _cts.Token);
                    }
                    catch (ObjectDisposedException)
                    {
                        break;
                    }
                    catch (OperationCanceledException)
                    {
                        break;
                    }
                }
            }
            catch (SocketException ex) when (ex.SocketErrorCode == SocketError.AddressAlreadyInUse)
            {
                AnsiConsole.MarkupLine($"[red]Error: Port {settings.ListenPort} is already in use[/]");
                return 2;
            }
            catch (Exception ex)
            {
                AnsiConsole.MarkupLine($"[red]Error: {ex.Message}[/]");
                if (settings.Verbose)
                {
                    AnsiConsole.WriteException(ex);
                }
                return 2;
            }
            finally
            {
                _listener?.Stop();
                _logWriter?.Close();

                if (!settings.Quiet)
                {
                    AnsiConsole.WriteLine();
                    AnsiConsole.MarkupLine($"[grey]Proxy stopped. Total messages: {_messageCount}[/]");
                }
            }

            return 0;
        }

        /// <summary>
        /// Handle a client connection.
        /// </summary>
        private async Task HandleClientAsync(TcpClient client, ProxyCommandSettings settings, CancellationToken ct)
        {
            var sessionId = Interlocked.Increment(ref _sessionCount);
            var clientEndpoint = client.Client.RemoteEndPoint?.ToString() ?? "unknown";

            if (!settings.Quiet)
            {
                AnsiConsole.MarkupLine($"[green][[{GetTimestamp(settings)}]] Session #{sessionId} started[/] - Client: {clientEndpoint}");
            }

            TcpClient? server = null;
            ProxySession? session = null;

            try
            {
                // Connect to target server
                server = new TcpClient();
                await server.ConnectAsync(settings.TargetHost, settings.TargetPort);

                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[grey][[{GetTimestamp(settings)}]] Session #{sessionId} connected to {settings.Target}[/]");
                }

                session = new ProxySession(sessionId, client, server);
                _sessions[sessionId] = session;

                // Start bidirectional forwarding
                var clientToServer = ForwardDataAsync(session, true, settings, ct);
                var serverToClient = ForwardDataAsync(session, false, settings, ct);

                await Task.WhenAny(clientToServer, serverToClient);
            }
            catch (Exception ex)
            {
                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[red][[{GetTimestamp(settings)}]] Session #{sessionId} error: {ex.Message}[/]");
                }
            }
            finally
            {
                // Clean up session if it was registered
                if (_sessions.TryRemove(sessionId, out var removedSession))
                {
                    removedSession.Dispose();
                }
                else if (session != null)
                {
                    // Session was created but not registered (shouldn't happen, but be safe)
                    session.Dispose();
                }
                else
                {
                    // Session was never created - dispose client and server directly
                    client.Dispose();
                    server?.Dispose();
                }

                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine($"[yellow][[{GetTimestamp(settings)}]] Session #{sessionId} ended[/]");
                }
            }
        }

        /// <summary>
        /// Forward data between client and server.
        /// </summary>
        private async Task ForwardDataAsync(ProxySession session, bool clientToServer, ProxyCommandSettings settings, CancellationToken ct)
        {
            var source = clientToServer ? session.ClientStream : session.ServerStream;
            var dest = clientToServer ? session.ServerStream : session.ClientStream;
            var direction = clientToServer ? "CLIENT->SERVER" : "SERVER->CLIENT";
            var arrowColor = clientToServer ? "cyan" : "green";

            var buffer = new byte[8192];

            try
            {
                while (!ct.IsCancellationRequested)
                {
                    int bytesRead;
                    try
                    {
                        bytesRead = await source.ReadAsync(buffer, 0, buffer.Length, ct);
                    }
                    catch (OperationCanceledException)
                    {
                        break;
                    }

                    if (bytesRead == 0)
                    {
                        break; // Connection closed
                    }

                    // Forward to destination
                    await dest.WriteAsync(buffer, 0, bytesRead, ct);

                    // Process and log the message
                    var message = Encoding.UTF8.GetString(buffer, 0, bytesRead);
                    var messages = message.Split(new[] { '\r', '\n' }, StringSplitOptions.RemoveEmptyEntries);

                    foreach (var msg in messages)
                    {
                        if (!ShouldLog(msg, settings))
                        {
                            continue;
                        }

                        Interlocked.Increment(ref _messageCount);

                        // Validate if requested
                        bool? isValid = null;
                        string? validationError = null;
                        if (settings.ValidateMessages)
                        {
                            (isValid, validationError) = ValidateMessage(msg);
                        }

                        // Output to console
                        OutputMessage(session.Id, direction, arrowColor, msg, isValid, validationError, settings);

                        // Log to file
                        if (_logWriter != null)
                        {
                            await LogMessageAsync(session.Id, direction, msg, isValid, settings);
                        }
                    }
                }
            }
            catch (IOException)
            {
                // Connection closed
            }
            catch (Exception ex)
            {
                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[red]Forward error ({direction}): {ex.Message}[/]");
                }
            }
        }

        /// <summary>
        /// Check if a message should be logged based on filter.
        /// </summary>
        private bool ShouldLog(string message, ProxyCommandSettings settings)
        {
            if (_filterRegex == null)
            {
                return true;
            }

            try
            {
                var doc = JsonDocument.Parse(message);
                if (doc.RootElement.TryGetProperty("context", out var contextElement))
                {
                    var context = contextElement.GetString() ?? "";
                    return _filterRegex.IsMatch(context);
                }
            }
            catch
            {
                // Not JSON, include it
            }

            return true;
        }

        /// <summary>
        /// Validate a message against the protocol schema.
        /// </summary>
        private (bool isValid, string? error) ValidateMessage(string message)
        {
            if (_schemaValidator == null)
            {
                return (true, null); // No validator available, skip validation
            }

            try
            {
                // Perform full schema validation
                var result = _schemaValidator.ValidateMessageJson(message, MessageDirection.Response);

                if (result.IsValid)
                {
                    return (true, null);
                }

                var errorMessages = string.Join("; ", result.Errors.Select(e => e.Message).Take(3));
                return (false, errorMessages);
            }
            catch (Exception ex)
            {
                return (false, $"Validation error: {ex.Message}");
            }
        }

        /// <summary>
        /// Output a message to the console.
        /// </summary>
        private void OutputMessage(int sessionId, string direction, string arrowColor, string message,
            bool? isValid, string? validationError, ProxyCommandSettings settings)
        {
            var timestamp = GetTimestamp(settings);
            var arrow = direction.Contains("CLIENT") ? ">>>" : "<<<";

            switch (settings.Format)
            {
                case ProxyFormat.Pretty:
                    OutputPrettyMessage(sessionId, timestamp, direction, arrowColor, arrow, message, isValid, validationError, settings);
                    break;

                case ProxyFormat.Json:
                    OutputJsonMessage(sessionId, timestamp, direction, message, isValid, validationError);
                    break;

                case ProxyFormat.Raw:
                    AnsiConsole.MarkupLine($"[grey]{timestamp}[/] [{arrowColor}]{arrow}[/] #{sessionId} {EscapeMarkup(message)}");
                    break;
            }
        }

        /// <summary>
        /// Output a pretty-formatted message.
        /// </summary>
        private void OutputPrettyMessage(int sessionId, string timestamp, string direction, string arrowColor,
            string arrow, string message, bool? isValid, string? validationError, ProxyCommandSettings settings)
        {
            var prefix = settings.Timestamps ? $"[grey]{timestamp}[/] " : "";

            try
            {
                var doc = JsonDocument.Parse(message);
                var context = doc.RootElement.TryGetProperty("context", out var ctx) ? ctx.GetString() : "unknown";
                var formatted = JsonSerializer.Serialize(doc.RootElement, new JsonSerializerOptions { WriteIndented = true });

                // Validation indicator
                var validIndicator = "";
                if (isValid.HasValue)
                {
                    validIndicator = isValid.Value ? " [green]VALID[/]" : $" [red]INVALID: {validationError}[/]";
                }

                AnsiConsole.MarkupLine($"{prefix}[{arrowColor}]{arrow}[/] [grey]#{sessionId}[/] [bold]{context}[/]{validIndicator}");

                if (settings.Verbose)
                {
                    AnsiConsole.Write(new Panel(formatted)
                        .Border(BoxBorder.Rounded)
                        .BorderColor(direction.Contains("CLIENT") ? Color.Aqua : Color.Green));
                }
            }
            catch
            {
                // Not JSON
                AnsiConsole.MarkupLine($"{prefix}[{arrowColor}]{arrow}[/] [grey]#{sessionId}[/] {EscapeMarkup(message)}");
            }
        }

        /// <summary>
        /// Output a JSON-formatted log entry.
        /// </summary>
        private void OutputJsonMessage(int sessionId, string timestamp, string direction, string message,
            bool? isValid, string? validationError)
        {
            var logEntry = new
            {
                timestamp = timestamp,
                session = sessionId,
                direction = direction.ToLowerInvariant().Replace("->", "_to_"),
                valid = isValid,
                validationError = validationError,
                message = message
            };

            var json = JsonSerializer.Serialize(logEntry);
            AnsiConsole.WriteLine(json);
        }

        /// <summary>
        /// Log a message to the log file (thread-safe).
        /// </summary>
        private async Task LogMessageAsync(int sessionId, string direction, string message, bool? isValid, ProxyCommandSettings settings)
        {
            if (_logWriter == null) return;

            var logLine = settings.Format switch
            {
                ProxyFormat.Json => JsonSerializer.Serialize(new
                {
                    timestamp = DateTime.Now.ToString("o"),
                    session = sessionId,
                    direction = direction,
                    valid = isValid,
                    message = message
                }),
                _ => $"[{DateTime.Now:yyyy-MM-dd HH:mm:ss.fff}] [{direction}] #{sessionId} {message}"
            };

            await _logWriterLock.WaitAsync();
            try
            {
                await _logWriter.WriteLineAsync(logLine);
            }
            finally
            {
                _logWriterLock.Release();
            }
        }

        /// <summary>
        /// Get formatted timestamp.
        /// </summary>
        private string GetTimestamp(ProxyCommandSettings settings)
        {
            return settings.Timestamps ? DateTime.Now.ToString("HH:mm:ss.fff") : "";
        }

        /// <summary>
        /// Escape Spectre.Console markup characters.
        /// </summary>
        private static string EscapeMarkup(string text)
        {
            return text.Replace("[", "[[").Replace("]", "]]");
        }
    }

    /// <summary>
    /// Represents a proxy session between client and server.
    /// </summary>
    internal sealed class ProxySession : IDisposable
    {
        public int Id { get; }
        public TcpClient Client { get; }
        public TcpClient Server { get; }
        public NetworkStream ClientStream { get; }
        public NetworkStream ServerStream { get; }
        private bool _disposed;

        public ProxySession(int id, TcpClient client, TcpClient server)
        {
            Id = id;
            Client = client;
            Server = server;
            ClientStream = client.GetStream();
            ServerStream = server.GetStream();
        }

        public void Dispose()
        {
            if (_disposed) return;
            _disposed = true;

            ClientStream.Dispose();
            ServerStream.Dispose();
            Client.Dispose();
            Server.Dispose();
        }
    }
}
