using System;
using System.ComponentModel;
using System.Net;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Mbrcpval.CLI.Settings;
using Spectre.Console;
using Spectre.Console.Cli;

namespace Mbrcpval.CLI.Commands
{
    /// <summary>
    /// Settings for the mock server command.
    /// </summary>
    public class MockCommandSettings : CommonSettings
    {
        /// <summary>
        /// The port to listen on.
        /// </summary>
        [CommandOption("-p|--port <PORT>")]
        [Description("The port to listen on")]
        [DefaultValue(3000)]
        public int Port { get; set; } = 3000;

        /// <summary>
        /// Number of mock tracks to generate in the library.
        /// </summary>
        [CommandOption("-g|--generate <COUNT>")]
        [Description("Number of mock tracks to generate")]
        [DefaultValue(50)]
        public int Generate { get; set; } = 50;

        /// <summary>
        /// The protocol version to emulate.
        /// </summary>
        [CommandOption("--protocol <VERSION>")]
        [Description("Protocol version to emulate (4.0 or 4.5)")]
        [DefaultValue("4.5")]
        public string Protocol { get; set; } = "4.5";

        /// <summary>
        /// Enable service discovery broadcast.
        /// </summary>
        [CommandOption("--discovery")]
        [Description("Enable service discovery broadcast")]
        [DefaultValue(true)]
        public bool Discovery { get; set; } = true;

        /// <summary>
        /// The server name for discovery.
        /// </summary>
        [CommandOption("-n|--name <NAME>")]
        [Description("Server name for discovery")]
        [DefaultValue("MBRCPVAL Mock Server")]
        public string Name { get; set; } = "MBRCPVAL Mock Server";

        /// <summary>
        /// Artificial latency in milliseconds.
        /// </summary>
        [CommandOption("-l|--latency <MS>")]
        [Description("Artificial latency in milliseconds")]
        [DefaultValue(0)]
        public int Latency { get; set; } = 0;

        /// <summary>
        /// Log file path.
        /// </summary>
        [CommandOption("--log <FILE>")]
        [Description("Log file path for message logging")]
        public string? LogFile { get; set; }

        /// <summary>
        /// Bind to all interfaces instead of just localhost.
        /// </summary>
        [CommandOption("--bind-all")]
        [Description("Bind to all network interfaces (0.0.0.0)")]
        [DefaultValue(false)]
        public bool BindAll { get; set; }

        /// <summary>
        /// Validate the settings.
        /// </summary>
        public override ValidationResult Validate()
        {
            var baseResult = base.Validate();
            if (!baseResult.Successful)
            {
                return baseResult;
            }

            if (Port < 1 || Port > 65535)
            {
                return ValidationResult.Error("Port must be between 1 and 65535.");
            }

            if (Generate < 0 || Generate > 10000)
            {
                return ValidationResult.Error("Generate count must be between 0 and 10000.");
            }

            var validProtocols = new[] { "4.0", "4.5" };
            if (!Array.Exists(validProtocols, p => p.Equals(Protocol, StringComparison.OrdinalIgnoreCase)))
            {
                return ValidationResult.Error($"Invalid protocol version '{Protocol}'. Valid options: {string.Join(", ", validProtocols)}");
            }

            if (Latency < 0 || Latency > 10000)
            {
                return ValidationResult.Error("Latency must be between 0 and 10000 milliseconds.");
            }

            return ValidationResult.Success();
        }
    }

    /// <summary>
    /// Command to start a mock MBRC server for testing.
    /// </summary>
    public class MockCommand : AsyncCommand<MockCommandSettings>
    {
        private TcpListener? _listener;
        private CancellationTokenSource? _cts;
        private int _connectionCount = 0;
        private readonly SemaphoreSlim _logFileLock = new(1, 1);

        /// <summary>
        /// Execute the mock server command.
        /// </summary>
        /// <param name="context">The command context.</param>
        /// <param name="settings">The command settings.</param>
        /// <returns>Exit code.</returns>
        public override async Task<int> ExecuteAsync(CommandContext context, MockCommandSettings settings)
        {
            _cts = new CancellationTokenSource();

            // Handle Ctrl+C
            Console.CancelKeyPress += (sender, e) =>
            {
                e.Cancel = true;
                AnsiConsole.MarkupLine("\n[yellow]Shutting down...[/]");
                _cts.Cancel();
            };

            var bindAddress = settings.BindAll ? IPAddress.Any : IPAddress.Loopback;

            if (!settings.Quiet)
            {
                AnsiConsole.MarkupLine("[bold]Starting Mock MBRC Server[/]");
                AnsiConsole.MarkupLine($"  Name: [cyan]{settings.Name}[/]");
                AnsiConsole.MarkupLine($"  Bind: [cyan]{(settings.BindAll ? "0.0.0.0" : "127.0.0.1")}:{settings.Port}[/]");
                AnsiConsole.MarkupLine($"  Protocol: [cyan]{settings.Protocol}[/]");
                AnsiConsole.MarkupLine($"  Mock Tracks: [cyan]{settings.Generate}[/]");
                AnsiConsole.MarkupLine($"  Discovery: [cyan]{(settings.Discovery ? "Enabled" : "Disabled")}[/]");
                if (settings.Latency > 0)
                {
                    AnsiConsole.MarkupLine($"  Latency: [cyan]{settings.Latency}ms[/]");
                }
                if (!string.IsNullOrEmpty(settings.LogFile))
                {
                    AnsiConsole.MarkupLine($"  Log File: [cyan]{settings.LogFile}[/]");
                }
                AnsiConsole.WriteLine();
                AnsiConsole.MarkupLine("[grey]Press Ctrl+C to stop the server[/]");
                AnsiConsole.WriteLine();
            }

            try
            {
                // Initialize mock server
                // TODO: Initialize MockServer from Mock project
                // var mockServer = new MockServer(settings.Protocol, settings.Generate);

                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[grey]Generated {settings.Generate} mock tracks[/]");
                }

                // Start TCP listener
                _listener = new TcpListener(bindAddress, settings.Port);
                _listener.Start();

                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine($"[green]Server listening on {bindAddress}:{settings.Port}[/]");
                }

                // Start discovery broadcast if enabled
                if (settings.Discovery)
                {
                    _ = StartDiscoveryBroadcastAsync(settings, _cts.Token);
                }

                // Accept connections
                while (!_cts.Token.IsCancellationRequested)
                {
                    try
                    {
                        var client = await _listener.AcceptTcpClientAsync();
                        _ = HandleClientAsync(client, settings, _cts.Token);
                    }
                    catch (ObjectDisposedException)
                    {
                        // Listener was stopped
                        break;
                    }
                    catch (OperationCanceledException)
                    {
                        break;
                    }
                }
            }
            catch (SocketException ex) when (ex.SocketErrorCode == SocketError.AddressAlreadyInUse)
            {
                AnsiConsole.MarkupLine($"[red]Error: Port {settings.Port} is already in use[/]");
                return 2;
            }
            catch (Exception ex)
            {
                AnsiConsole.MarkupLine($"[red]Error: {ex.Message}[/]");
                if (settings.Verbose)
                {
                    AnsiConsole.WriteException(ex);
                }
                return 2;
            }
            finally
            {
                _listener?.Stop();
                _cts?.Dispose();
                _cts = null;

                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine("[grey]Server stopped[/]");
                }
            }

            return 0;
        }

        /// <summary>
        /// Handle a client connection.
        /// </summary>
        private async Task HandleClientAsync(TcpClient client, MockCommandSettings settings, CancellationToken ct)
        {
            var connectionId = Interlocked.Increment(ref _connectionCount);
            var endpoint = client.Client.RemoteEndPoint?.ToString() ?? "unknown";

            if (!settings.Quiet)
            {
                AnsiConsole.MarkupLine($"[green][[{DateTime.Now:HH:mm:ss}]] Client connected:[/] {endpoint} [grey](#{connectionId})[/]");
            }

            try
            {
                using var stream = client.GetStream();
                var buffer = new byte[4096];

                while (!ct.IsCancellationRequested && client.Connected)
                {
                    int bytesRead;
                    try
                    {
                        bytesRead = await stream.ReadAsync(buffer, 0, buffer.Length, ct);
                    }
                    catch (OperationCanceledException)
                    {
                        break;
                    }

                    if (bytesRead == 0)
                    {
                        break; // Client disconnected
                    }

                    var message = Encoding.UTF8.GetString(buffer, 0, bytesRead).Trim();

                    if (settings.Verbose)
                    {
                        AnsiConsole.MarkupLine($"[grey][[{DateTime.Now:HH:mm:ss}]] Received from #{connectionId}:[/] {EscapeMarkup(message)}");
                    }

                    // Log to file if specified
                    if (!string.IsNullOrEmpty(settings.LogFile))
                    {
                        await LogMessageAsync(settings.LogFile, "RECV", connectionId, message);
                    }

                    // Apply artificial latency
                    if (settings.Latency > 0)
                    {
                        await Task.Delay(settings.Latency, ct);
                    }

                    // TODO: Process message through MockServer
                    // var response = mockServer.ProcessMessage(message);
                    var response = GenerateMockResponse(message, settings);

                    if (!string.IsNullOrEmpty(response))
                    {
                        var responseBytes = Encoding.UTF8.GetBytes(response + "\r\n");
                        await stream.WriteAsync(responseBytes, 0, responseBytes.Length, ct);

                        if (settings.Verbose)
                        {
                            AnsiConsole.MarkupLine($"[grey][[{DateTime.Now:HH:mm:ss}]] Sent to #{connectionId}:[/] {EscapeMarkup(response)}");
                        }

                        if (!string.IsNullOrEmpty(settings.LogFile))
                        {
                            await LogMessageAsync(settings.LogFile, "SEND", connectionId, response);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[red][[{DateTime.Now:HH:mm:ss}]] Error with client #{connectionId}: {ex.Message}[/]");
                }
            }
            finally
            {
                client.Close();
                if (!settings.Quiet)
                {
                    AnsiConsole.MarkupLine($"[yellow][[{DateTime.Now:HH:mm:ss}]] Client disconnected:[/] {endpoint} [grey](#{connectionId})[/]");
                }
            }
        }

        /// <summary>
        /// Generate a mock response for a message.
        /// </summary>
        private string GenerateMockResponse(string message, MockCommandSettings settings)
        {
            // Basic mock responses - TODO: Replace with full MockServer implementation
            try
            {
                var doc = System.Text.Json.JsonDocument.Parse(message);
                var root = doc.RootElement;

                if (root.TryGetProperty("context", out var contextElement))
                {
                    var context = contextElement.GetString()?.ToLowerInvariant();

                    return context switch
                    {
                        "protocol" => $"{{\"context\":\"protocol\",\"data\":{{\"protocol_version\":\"{settings.Protocol}\",\"player_name\":\"MusicBee\"}}}}",
                        "ping" => "{\"context\":\"pong\",\"data\":\"\"}",
                        "player" or "playerstate" => "{\"context\":\"playerstate\",\"data\":{\"playerstate\":\"playing\",\"playermute\":false,\"playervolume\":80,\"playershuffle\":\"off\",\"playerrepeat\":\"none\",\"playerscrobble\":true}}",
                        "nowplayingtrack" => "{\"context\":\"nowplayingtrack\",\"data\":{\"artist\":\"Mock Artist\",\"album\":\"Mock Album\",\"title\":\"Mock Track\",\"year\":\"2024\",\"path\":\"/mock/path/track.mp3\"}}",
                        "nowplayingcover" => "{\"context\":\"nowplayingcover\",\"data\":\"\"}",
                        "nowplayinglyrics" => "{\"context\":\"nowplayinglyrics\",\"data\":\"\"}",
                        "nowplayingrating" => "{\"context\":\"nowplayingrating\",\"data\":\"0\"}",
                        "nowplayingposition" => "{\"context\":\"nowplayingposition\",\"data\":{\"current\":0,\"total\":300000}}",
                        "output" => "{\"context\":\"output\",\"data\":[{\"name\":\"Default\",\"active\":true}]}",
                        "init" or "close" => string.Empty,  // No response needed for init/close
                        _ => $"{{\"context\":\"error\",\"data\":\"Unknown context: {context}\"}}"
                    };
                }
            }
            catch
            {
                return "{\"context\":\"error\",\"data\":\"Invalid JSON\"}";
            }

            return "{\"context\":\"error\",\"data\":\"Missing context\"}";
        }

        /// <summary>
        /// Start the discovery broadcast loop.
        /// </summary>
        private async Task StartDiscoveryBroadcastAsync(MockCommandSettings settings, CancellationToken ct)
        {
            try
            {
                using var udpClient = new UdpClient();
                udpClient.EnableBroadcast = true;

                var discoveryMessage = $"{{\"name\":\"{settings.Name}\",\"port\":{settings.Port},\"protocol\":\"{settings.Protocol}\"}}";
                var messageBytes = Encoding.UTF8.GetBytes(discoveryMessage);
                var broadcastEndpoint = new IPEndPoint(IPAddress.Broadcast, 45345);

                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine("[grey]Discovery broadcast started on port 45345[/]");
                }

                while (!ct.IsCancellationRequested)
                {
                    try
                    {
                        await udpClient.SendAsync(messageBytes, messageBytes.Length, broadcastEndpoint);
                    }
                    catch (SocketException)
                    {
                        // Broadcast may fail on some networks
                    }

                    await Task.Delay(5000, ct); // Broadcast every 5 seconds
                }
            }
            catch (OperationCanceledException)
            {
                // Expected on shutdown
            }
            catch (Exception ex)
            {
                if (settings.Verbose)
                {
                    AnsiConsole.MarkupLine($"[yellow]Discovery broadcast error: {ex.Message}[/]");
                }
            }
        }

        /// <summary>
        /// Log a message to the log file with synchronization.
        /// </summary>
        private async Task LogMessageAsync(string logFile, string direction, int connectionId, string message)
        {
            var logLine = $"[{DateTime.Now:yyyy-MM-dd HH:mm:ss.fff}] [{direction}] #{connectionId} {message}\n";
            await _logFileLock.WaitAsync();
            try
            {
                await System.IO.File.AppendAllTextAsync(logFile, logLine);
            }
            finally
            {
                _logFileLock.Release();
            }
        }

        /// <summary>
        /// Escape Spectre.Console markup characters.
        /// </summary>
        private static string EscapeMarkup(string text)
        {
            return text.Replace("[", "[[").Replace("]", "]]");
        }
    }
}
