# MBRCPVAL

**MusicBee Remote Control Protocol Validator**

A comprehensive CLI toolkit for testing and validating MBRC protocol implementations.

---

## Project Scope

MBRCPVAL is part of the `mbrc-plugin-main-server` project. The parent project contains:

| Folder | Purpose | Scope |
|--------|---------|-------|
| `plugin/` | MBRC Server Plugin source code | Production plugin for MusicBee |
| `mbrcpval/` | Protocol Validator toolkit | Developer/QA testing tools |
| `firewall-utility/` | Windows Firewall configuration helper | End-user utility |
| `mbrc-plugin.Tests/` | Unit tests for plugin | Development testing |
| `Documentation/` | Legacy documentation | Reference materials |
| `deploy/` | Build output and packages | Distribution |

---

## Quick Start

```bash
# Build the project
cd mbrcpval
dotnet build

# Run with help
dotnet run -- --help

# Validate a running MBRC server
dotnet run -- server --host localhost --port 3000 --suite all

# Run only mandatory tests (Bronze certification)
dotnet run -- server --host localhost --tags mandatory

# Start a mock MBRC server for client development
dotnet run -- mock --port 3000 --generate 100

# Interactive REPL session
dotnet run -- repl --host localhost --port 3000
```

---

## Documentation Index

All documentation is located in the `Docs/` subfolder.

### Product Documentation

Documents about **using** the MBRCPVAL tools and understanding the protocol.

| Document | Purpose | Audience |
|----------|---------|----------|
| [MBRCPVAL-QUICKSTART.md](Docs/MBRCPVAL-QUICKSTART.md) | Quick start guide with testing ecosystem overview | All users |
| [MBRCPVAL-Overview.md](Docs/MBRCPVAL-Overview.md) | What is MBRCPVAL and why use it | New users |
| [INTEGRATION-PLAN.md](Docs/INTEGRATION-PLAN.md) | Step-by-step validation workflow | QA, CI/CD |
| [CRITICALITY-FRAMEWORK.md](Docs/CRITICALITY-FRAMEWORK.md) | Mandatory/Recommended/Optional test levels | Developers, QA |
| [PROTOCOL-SDK-MIRROR.md](Docs/PROTOCOL-SDK-MIRROR.md) | MBRC Protocol v4/4.5 reference (MD mirror) | Client developers |
| [MBRCPVAL-HTML-MIRROR.md](Docs/MBRCPVAL-HTML-MIRROR.md) | Validator documentation (MD mirror of HTML) | All users |

### Project Documentation

Documents about **building and developing** the MBRCPVAL platform.

| Document | Purpose | Audience |
|----------|---------|----------|
| [MBRCPVAL-SPEC.md](Docs/MBRCPVAL-SPEC.md) | Full technical specification | Contributors |
| [MBRCPVAL-IMPLEMENTATION.md](Docs/MBRCPVAL-IMPLEMENTATION.md) | Implementation details and architecture | Contributors |
| [MBRCPVAL-MASTERPLAN.md](Docs/MBRCPVAL-MASTERPLAN.md) | Project roadmap and vision | Contributors |
| [MBRCPVAL-CHANGELOG.md](Docs/MBRCPVAL-CHANGELOG.md) | Version history and changes | All users |
| [BACKLOG.md](Docs/BACKLOG.md) | Missing features and planned work | Contributors |
| [ISSUES.md](Docs/ISSUES.md) | Known bugs and limitations | Contributors, QA |

### Legacy/Deprecated

| Document | Status | Notes |
|----------|--------|-------|
| [Docs/QUICKSTART.md](Docs/QUICKSTART.md) | Deprecated | Use root MBRCPVAL-QUICKSTART.md instead |

---

## Commands

### `server` - Validate MBRC Server

Test a running MBRC server/plugin against protocol test suites.

```bash
mbrcpval server -h <host> -p <port> [options]

Options:
  -H, --host <HOST>       Server host/IP (required)
  -p, --port <PORT>       Server port [default: 3000]
  -s, --suite <SUITE>     Test suite to run [default: all]
  -t, --test <TEST>       Run a specific test by ID
  --list                  List all available tests without running them
  --protocol <VERSION>    Protocol version to test: 4.0 or 4.5 [default: 4.5]
  -r, --report <FORMAT>   Report format: text, json, html, junit [default: text]
  -o, --output <FILE>     Output file for report
  --timeout <SECONDS>     Timeout in seconds [default: 30]
  --fail-fast             Stop on first test failure
  -v, --verbose           Verbose output
```

### `validate` - Validate Single Message

Validate a single MBRC protocol JSON message against the schema.

```bash
mbrcpval validate [options]

Options:
  -m, --message <JSON>    Inline JSON message to validate
  -f, --file <PATH>       Path to JSON file to validate
  --stdin                 Read JSON from standard input
  -d, --direction <DIR>   Message direction: auto, client, server [default: auto]
  --protocol <VERSION>    Protocol version [default: 4.5]
```

### `mock` - Mock MBRC Server

Start a mock MBRC server for testing client applications.

```bash
mbrcpval mock [options]

Options:
  -p, --port <PORT>       Port to listen on [default: 3000]
  -g, --generate <COUNT>  Number of mock tracks to generate [default: 50]
  --protocol <VERSION>    Protocol version to emulate [default: 4.5]
  -n, --name <NAME>       Server name for discovery
  -l, --latency <MS>      Artificial latency in milliseconds [default: 0]
  --discovery             Enable service discovery broadcast [default: true]
  --log <FILE>            Log file path for message logging
```

### `repl` - Interactive Session

Interactive REPL for exploring and testing MBRC protocol commands.

```bash
mbrcpval repl -h <host> [options]

REPL Commands:
  .connect                Connect to server
  .disconnect             Disconnect from server
  .send <context> [data]  Send message (e.g., .send player PlayPause)
  .raw <json>             Send raw JSON
  .listen                 Toggle background message listening
  .status                 Show connection status
  .help                   Show help
  .exit                   Exit REPL
```

### `proxy` - Traffic Interceptor

Proxy mode to intercept and log traffic between client and server.

```bash
mbrcpval proxy [options]

Options:
  -l, --listen <PORT>     Port to listen on (required)
  -t, --target <HOST:PORT> Target server address (required)
  -o, --output <FILE>     Output file for captured traffic
  --format <FORMAT>       Output format: raw, pretty, json [default: pretty]
  --filter <REGEX>        Filter messages by context (regex pattern)
  --validate              Validate messages against schema
```

---

## Test Suites & Criticality

### Available Test Suites

| Suite | File | Tests | Description |
|-------|------|-------|-------------|
| Handshake | `handshake.yaml` | 8 | Connection & protocol negotiation |
| Player Control | `player-control.yaml` | 14 | Play, pause, volume, modes |
| Now Playing | `now-playing.yaml` | 10 | Track info, cover, lyrics, rating |
| Library | `library.yaml` | 10 | Browse, search, pagination |
| Errors | `errors.yaml` | 10 | Error handling, robustness |

### Criticality Levels

| Level | Tag | Priority | Description |
|-------|-----|----------|-------------|
| **MANDATORY** | `mandatory` | 1-30 | Must pass for basic protocol compliance |
| **RECOMMENDED** | `recommended` | 31-70 | Should pass for good user experience |
| **OPTIONAL** | `optional` | 71-100 | May pass for full feature parity |

### Certification Levels

| Level | Badge | Requirements |
|-------|-------|--------------|
| Bronze | 🥉 | 100% mandatory tests pass |
| Silver | 🥈 | 100% mandatory + 90% recommended |
| Gold | 🥇 | 100% mandatory + 100% recommended + 80% optional |
| Platinum | 💎 | 100% all tests pass |

---

## Project Structure

```
mbrcpval/
├── src/
│   ├── CLI/              # Command-line interface (System.CommandLine)
│   │   └── Commands/     # server, mock, repl, proxy, validate commands
│   ├── Core/             # TCP client, message parsing, schema validation
│   │   └── Reliability.cs # Circuit breaker, retry policy, health monitor
│   ├── Testing/          # Test runner, YAML loader, assertion evaluator
│   │   └── ProtocolFuzzer.cs # Protocol fuzzing for robustness
│   └── Reporting/        # Text, JSON, HTML, JUnit reporters
├── schemas/              # 41 JSON Schema definitions
├── tests/suites/         # 5 YAML test suites with criticality tags
├── Docs/                 # Documentation (see index above)
│   └── MBRCPVAL-QUICKSTART.md # Main quick start guide
└── README.md             # This file
```

---

## Requirements

- .NET 8.0 or later
- Windows, Linux, or macOS

---

## Building

```bash
cd mbrcpval
dotnet restore
dotnet build
```

For a single-file executable:
```bash
dotnet publish -c Release -r win-x64 --self-contained true -p:PublishSingleFile=true
```

---

## Examples

```bash
# === CRITICALITY-BASED VALIDATION ===

# Bronze certification (mandatory only)
mbrcpval server -h localhost --tags mandatory

# Silver certification (mandatory + recommended)
mbrcpval server -h localhost --tags mandatory,recommended --report html -o report.html

# Full validation
mbrcpval server -h localhost --suite all

# === DEVELOPMENT ===

# Validate a player state message
mbrcpval validate -m '{"context":"playerstate","data":{"playerstate":"playing"}}'

# Start mock server with 500 tracks
mbrcpval mock -p 3000 -g 500 --log server.log

# Interactive exploration
mbrcpval repl -h localhost -p 3000

# Capture traffic for debugging
mbrcpval proxy -l 3001 -t localhost:3000 -o traffic.jsonl --validate
```

---

## Related Documentation

- [Plugin README](../README.md) - Main plugin documentation
- [Plugin Changelog](../plugin_CHANGELOG.md) - Plugin version history
- [Protocol SDK](https://halrad.com/MBXRemote/MBRC-Server-Plugin-Protocol.html) - Online protocol reference

---

## Dependencies

| Package | Author | License | Purpose |
|---------|--------|---------|---------|
| [Spectre.Console](https://github.com/spectreconsole/spectre.console) | Patrik Svensson | MIT | Beautiful console output and CLI |
| [NJsonSchema](https://github.com/RicoSuter/NJsonSchema) | Rico Suter | MIT | JSON Schema validation |
| [Newtonsoft.Json](https://github.com/JamesNK/Newtonsoft.Json) | James Newton-King | MIT | JSON protocol parsing |
| [YamlDotNet](https://github.com/aaubry/YamlDotNet) | Antoine Aubry | MIT | YAML test suite parsing |
| [FsCheck](https://github.com/fscheck/FsCheck) | Kurt Schelfthout | BSD-3-Clause | Property-based fuzzing |

## Attribution

**Protocol Specification**: Based on [MusicBee Remote Protocol](https://mbrc.kelsos.net/) by [Konstantinos Paparas](https://github.com/kelsos)

**MusicBee**: [Steven Mayall](https://getmusicbee.com/) - The ultimate music manager and player

## License

[![License: GPL v3](https://img.shields.io/badge/License-GPLv3-blue.svg)](https://www.gnu.org/licenses/gpl-3.0)

Part of the MBRC Plugin project. Licensed under GPLv3.

---

*MBRCPVAL v1.5.26.3 | Updated 2026-01-04*
