# MBRCPVAL Quick Start Guide

Get up and running with MBRCPVAL in minutes.

---

## Installation

### Requirements

- .NET 8.0 or later
- Windows, Linux, or macOS
- Network access to target MBRC server (for server validation)

### Building from Source

```bash
cd mbrcpval
dotnet restore
dotnet build
```

### Create Standalone Executable

```bash
# Windows x64
dotnet publish -c Release -r win-x64 --self-contained true -p:PublishSingleFile=true

# Linux x64
dotnet publish -c Release -r linux-x64 --self-contained true -p:PublishSingleFile=true

# macOS x64
dotnet publish -c Release -r osx-x64 --self-contained true -p:PublishSingleFile=true
```

### Verify Installation

```bash
# If running from build directory
dotnet run -- --help

# If using published executable
mbrcpval --help
```

---

## First Steps

### 1. Validate Your Plugin

Connect to a running MBRC server plugin and run the core test suite:

```bash
mbrcpval server --host localhost --port 3000 --suite core
```

**Expected output:**

```
Validating MBRC Server
  Host: localhost:3000
  Protocol: 4.5
  Suite: core

Running Handshake tests...
Running Player State tests...
Running Playback Control tests...
...

┌────────┬───────┐
│ Metric │ Value │
├────────┼───────┤
│ Total  │ 25    │
│ Passed │ 25    │
│ Failed │ 0     │
│ Skip   │ 0     │
│ Time   │ 2.5s  │
└────────┴───────┘

All tests passed!
```

### 2. Run Full Test Suite with Report

Test all protocol commands and generate an HTML report:

```bash
mbrcpval server -h localhost -p 3000 -s all -r html -o report.html
```

Open `report.html` in your browser to view detailed results with expandable test cases.

### 3. Start Mock Server (for Client Development)

If you are developing a client application, start a mock MBRC server:

```bash
mbrcpval mock --port 3000 --generate 500
```

**Output:**

```
Starting Mock MBRC Server
  Name: MBRCPVAL Mock Server
  Bind: 127.0.0.1:3000
  Protocol: 4.5
  Mock Tracks: 500
  Discovery: Enabled

Press Ctrl+C to stop the server

Server listening on 127.0.0.1:3000
```

**Mock server features:**
- Full handshake support (v4.0 and v4.5)
- Simulated player state (play, pause, stop, volume, etc.)
- Browsable mock library (genres, artists, albums, tracks)
- Now playing queue operations
- Service discovery broadcast on UDP 45345

### 4. Interactive REPL Session

Use the REPL for interactive protocol exploration:

```bash
mbrcpval repl --host localhost --port 3000
```

**REPL session example:**

```
MBRCPVAL Interactive REPL
  Target: localhost:3000
  Protocol: 4.5

Type .help for available commands, .exit to quit

Connected to localhost:3000

localhost:3000> .send player
┌─────────────playerstate─────────────┐
│ {                                   │
│   "context": "playerstate",         │
│   "data": {                         │
│     "playerstate": "playing",       │
│     "playermute": false,            │
│     "playervolume": 80              │
│   }                                 │
│ }                                   │
└─────────────────────────────────────┘

localhost:3000> .send playervolume 50
┌──────────playervolume───────────┐
│ {                               │
│   "context": "playervolume",    │
│   "data": "50"                  │
│ }                               │
└─────────────────────────────────┘

localhost:3000> .help
┌────────────────────────┬───────────────────────────────────────────┐
│ Command                │ Description                               │
├────────────────────────┼───────────────────────────────────────────┤
│ .connect               │ Connect to the MBRC server                │
│ .disconnect            │ Disconnect from the server                │
│ .send <context> [data] │ Send a message with context and data      │
│ .raw <json>            │ Send raw JSON message                     │
│ .listen                │ Toggle background listening mode          │
│ .status                │ Show connection status                    │
│ .history               │ Show command history                      │
│ .clear                 │ Clear the screen                          │
│ .help                  │ Show this help                            │
│ .exit                  │ Exit the REPL                             │
└────────────────────────┴───────────────────────────────────────────┘

localhost:3000> .exit
Disconnected
```

---

## Validate Single Messages

The `validate` command checks JSON messages against the MBRC protocol schema:

```bash
# Inline JSON
mbrcpval validate -m '{"context":"player","data":"PlayPause"}'

# From file
mbrcpval validate -f message.json

# From stdin (useful for piping)
echo '{"context":"ping"}' | mbrcpval validate --stdin

# With direction hint
mbrcpval validate -m '{"context":"playerstate","data":{...}}' --direction server

# JSON output for scripting
mbrcpval validate -m '{"context":"player"}' --format json
```

**Output examples:**

```
VALID - Context: player
```

```
INVALID - Context: unknown
  Error: Unknown context: 'foobar'
```

---

## Understanding Results

### Test Statuses

| Status | Meaning |
|--------|---------|
| **PASS** | Test completed successfully, all assertions passed |
| **FAIL** | Test completed but assertions did not match expected values |
| **SKIP** | Test was skipped (prerequisites not met or feature disabled) |
| **ERROR** | Test encountered an unexpected error (timeout, connection lost) |

### Reading Failure Messages

```
[FAIL] playback.volume_set
       Expected: volume = 75
       Actual: volume = 80
       Path: $.data.playervolume
```

- **Expected**: What the test was looking for
- **Actual**: What the server actually returned
- **Path**: JSONPath to the failing assertion

### Report Formats

| Format | Option | Use Case |
|--------|--------|----------|
| Text | `-r text` | Console output (default) |
| JSON | `-r json -o results.json` | Programmatic processing |
| HTML | `-r html -o report.html` | Human-readable with details |
| JUnit | `-r junit -o results.xml` | CI/CD integration |

---

## Common Tasks

### Test Specific Suites

```bash
# Core protocol tests (handshake, ping/pong)
mbrcpval server -h localhost -p 3000 -s core

# Playback control tests
mbrcpval server -h localhost -p 3000 -s playback

# Library browsing tests
mbrcpval server -h localhost -p 3000 -s library

# Now playing queue tests
mbrcpval server -h localhost -p 3000 -s nowplaying

# Playlist tests
mbrcpval server -h localhost -p 3000 -s playlist

# Audio output tests
mbrcpval server -h localhost -p 3000 -s output

# Album art tests
mbrcpval server -h localhost -p 3000 -s cover

# Lyrics tests
mbrcpval server -h localhost -p 3000 -s lyrics

# Rating tests
mbrcpval server -h localhost -p 3000 -s rating

# All tests
mbrcpval server -h localhost -p 3000 -s all
```

### Test Specific Test Case

```bash
# Run only a specific test by ID
mbrcpval server -h localhost -p 3000 -t CORE-001
```

### List Available Tests

```bash
# List all tests with priority levels (without running them)
mbrcpval server --list --protocol 4.5

# List v4.0 tests only
mbrcpval server --list --protocol 4.0
```

### Protocol Version Selection

```bash
# Test with protocol v4.0 (legacy mode)
mbrcpval server -h localhost -p 3000 --protocol 4.0 -s all

# Test with protocol v4.5 (default, extended features)
mbrcpval server -h localhost -p 3000 --protocol 4.5 -s all
```

### Debug with Proxy Mode

Capture all traffic between your client and the server:

```bash
# Start proxy on port 3001, forwarding to server on port 3000
mbrcpval proxy -l 3001 -t localhost:3000 -o traffic.jsonl

# With schema validation on each message
mbrcpval proxy -l 3001 -t localhost:3000 --validate

# Filter to only player-related messages
mbrcpval proxy -l 3001 -t localhost:3000 --filter "player.*"

# Connect your client to port 3001 instead of 3000
# All messages are logged to traffic.jsonl
```

### Mock Server Options

```bash
# Basic mock server
mbrcpval mock -p 3000

# With more tracks in library
mbrcpval mock -p 3000 -g 1000

# Simulate network latency (50ms delay)
mbrcpval mock -p 3000 -l 50

# Custom server name (for discovery)
mbrcpval mock -p 3000 -n "My Test Server"

# Bind to all interfaces (accessible from network)
mbrcpval mock -p 3000 --bind-all

# Log all messages to file
mbrcpval mock -p 3000 --log server.log

# Disable service discovery
mbrcpval mock -p 3000 --discovery false
```

---

## Troubleshooting

### Connection Refused

```
Error: Connection refused to localhost:3000
```

- Ensure MusicBee is running with the MBRC plugin installed
- Check the plugin settings for the correct port number
- Verify firewall settings allow connections on the specified port

### Firewall Configuration

If external devices (e.g., Android phone) cannot connect to the proxy or mock server, you may need to open firewall ports:

**Windows (PowerShell as Administrator):**
```powershell
# Allow proxy port (3001)
netsh advfirewall firewall add rule name="MBRCPVAL Proxy" dir=in action=allow protocol=tcp localport=3001

# Allow mock server port (3000)
netsh advfirewall firewall add rule name="MBRCPVAL Mock" dir=in action=allow protocol=tcp localport=3000

# Remove rules when done
netsh advfirewall firewall delete rule name="MBRCPVAL Proxy"
netsh advfirewall firewall delete rule name="MBRCPVAL Mock"
```

**Linux:**
```bash
# Allow port 3001
sudo ufw allow 3001/tcp

# Remove when done
sudo ufw delete allow 3001/tcp
```

### Timeout on Tests

```
Error: Timeout waiting for response
```

- The server may be slow to respond; try increasing timeout: `--timeout 60`
- Check that no other client is connected (some servers allow only one connection)
- Maximum timeout is 300 seconds

### Schema Validation Errors

```
[FAIL] core.message_format
       Error: 'context' field is missing
```

- The server response did not match the expected schema
- This indicates a protocol compliance issue in the server implementation

### Port Already in Use (Mock Server)

```
Error: Port 3000 is already in use
```

- Another application is using the port
- Choose a different port: `mbrcpval mock -p 3001`
- Or stop the other application

---

## Global Options

All commands support these options:

| Option | Description |
|--------|-------------|
| `-v, --verbose` | Show detailed output and debug information |
| `-q, --quiet` | Minimal output (errors only) |
| `--no-color` | Disable colored output |
| `--help` | Show help for command |

---

## Quick Reference

```bash
# Server validation
mbrcpval server -h <host> -p <port> -s <suite> -r <format> -o <file>
mbrcpval server -h localhost -p 3000 -s all -r html -o report.html

# Validate single message
mbrcpval validate -m '{"context":"player"}'
mbrcpval validate -f message.json
echo '{"context":"ping"}' | mbrcpval validate --stdin

# Mock server
mbrcpval mock -p <port> -g <tracks> --log <file>
mbrcpval mock -p 3000 -g 500 --bind-all

# Interactive REPL
mbrcpval repl -h <host> -p <port>
mbrcpval repl -h localhost -p 3000 --history ~/.mbrcpval_history

# Proxy mode
mbrcpval proxy -l <listen_port> -t <target:port> -o <file>
mbrcpval proxy -l 3001 -t localhost:3000 -o traffic.jsonl --validate

# Get help
mbrcpval --help
mbrcpval server --help
mbrcpval mock --help
```

---

*Version 1.5.26.3 | Updated 2026-01-04*
