# MBRCPVAL - MusicBee Remote Control Protocol Validator

> **Related Documentation:**
> - [Quick Start](MBRCPVAL-QUICKSTART.md) - Getting started guide
> - [Overview](MBRCPVAL-Overview.md) - High-level introduction
> - [Implementation Guide](MBRCPVAL-IMPLEMENTATION.md) - Technical details
> - [Master Plan](MBRCPVAL-MASTERPLAN.md) - Implementation roadmap
> - [Changelog](MBRCPVAL-CHANGELOG.md) - Version history

---

## Project Specification & Implementation Plan

**Version:** 1.0.0-draft
**Date:** 2026-01-01
**Status:** PROPOSAL

---

## 1. Executive Summary

### 1.1 Purpose

MBRCPVAL (MusicBee Remote Control Protocol Validator) is a standalone validation toolkit designed to verify conformance of both **client** and **server** implementations against the MBRC Protocol specification (v4.0/v4.5). Similar to UPNP/DLNA certification tools, mbrcpval provides:

- **Protocol conformance testing** - Verify message formats, sequences, and responses
- **Behavioral validation** - Test state machine correctness and edge cases
- **Interoperability testing** - Ensure clients and servers work together correctly
- **Regression testing** - Automated test suites for CI/CD integration
- **Developer tooling** - Message inspector, protocol explorer, mock server/client

### 1.2 Problem Statement

Currently, MBRC protocol implementations are validated through manual testing with existing clients (Android, iOS, MBXRemote). This leads to:

- Inconsistent behavior between implementations
- Undocumented edge cases and protocol quirks
- No formal verification of protocol compliance
- Difficulty developing new clients without reference implementations
- Hidden bugs that only appear in specific client/server combinations

### 1.3 Goals

1. **Codify the protocol** - Create machine-readable protocol definitions (JSON Schema)
2. **Standardize testing** - Provide comprehensive test suites for all protocol commands
3. **Enable development** - Mock server/client for isolated development and testing
4. **Document behavior** - Capture observed behaviors and edge cases
5. **Ensure compatibility** - Validate backward compatibility with legacy clients

---

## 2. Protocol Fundamentals (Reference)

### 2.1 Transport Layer

| Property | Value |
|----------|-------|
| Transport | TCP |
| Default Port | 3000 (configurable) |
| Discovery | UDP Multicast 239.1.5.10:45345 |
| Encoding | UTF-8 (no BOM) |
| Line Terminator | CRLF (`\r\n`) |
| Message Format | JSON (Newtonsoft.Json compatible) |
| Framing | Newline-delimited JSON (NDJSON) |

### 2.2 Message Structure

```json
{
  "context": "<command_name>",
  "data": <payload>
}
```

**Key constraints:**
- `context` - Required string, identifies command type
- `data` - Required, can be: null, string, number, boolean, object, or array
- Messages are terminated with `\r\n`
- One JSON message per line (no embedded newlines in JSON)

### 2.3 Connection Handshake (Required Sequence)

```
1. TCP Connect
2. Send: {"context":"player","data":"android"}\r\n
3. Send: {"context":"protocol","data":{"protocol_version":4.5}}\r\n
4. Send: {"context":"init","data":null}\r\n
5. Receive: 6 init response messages (track, rating, love, status, cover, lyrics)
```

**Validation Points:**
- Handshake packet order is enforced
- Protocol version negotiation affects response format
- Init response must include all 6 messages

### 2.4 Protocol Versions

| Version | Features |
|---------|----------|
| 4.0 | Base functionality, backward compatible |
| 4.5 | Extended metadata, ARiA, library subscriptions, two-way sync |

---

## 3. Architecture

### 3.1 Component Overview

```
┌─────────────────────────────────────────────────────────────────┐
│                         MBRCPVAL                                 │
├─────────────────────────────────────────────────────────────────┤
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐              │
│  │   Schema    │  │    Test     │  │   Report    │              │
│  │  Validator  │  │   Runner    │  │  Generator  │              │
│  └──────┬──────┘  └──────┬──────┘  └──────┬──────┘              │
│         │                │                │                      │
│  ┌──────┴────────────────┴────────────────┴──────┐              │
│  │              Protocol Engine                   │              │
│  │  ┌─────────┐ ┌─────────┐ ┌─────────────────┐  │              │
│  │  │ Message │ │  State  │ │    Sequence     │  │              │
│  │  │ Parser  │ │ Machine │ │    Validator    │  │              │
│  │  └─────────┘ └─────────┘ └─────────────────┘  │              │
│  └───────────────────────────────────────────────┘              │
│                           │                                      │
│  ┌────────────────────────┴───────────────────────┐             │
│  │              Network Layer                      │             │
│  │  ┌───────────┐ ┌───────────┐ ┌───────────────┐ │             │
│  │  │TCP Client │ │TCP Server │ │ UDP Discovery │ │             │
│  │  │  (SUT)    │ │  (Mock)   │ │   (Mock)      │ │             │
│  │  └───────────┘ └───────────┘ └───────────────┘ │             │
│  └────────────────────────────────────────────────┘             │
├─────────────────────────────────────────────────────────────────┤
│  ┌─────────────┐  ┌─────────────┐  ┌─────────────┐              │
│  │    CLI      │  │    GUI      │  │    API      │              │
│  │  Interface  │  │  (Optional) │  │  (Library)  │              │
│  └─────────────┘  └─────────────┘  └─────────────┘              │
└─────────────────────────────────────────────────────────────────┘

SUT = System Under Test (the client or server being validated)
```

### 3.2 Operating Modes

#### 3.2.1 Server Validation Mode
```
┌──────────┐     TCP      ┌─────────────┐
│ MBRCPVAL ├──────────────► Plugin/Server│
│ (Client) │◄─────────────┤    (SUT)    │
└──────────┘              └─────────────┘
```
MBRCPVAL acts as a protocol-aware client, sending test commands and validating server responses.

#### 3.2.2 Client Validation Mode
```
┌─────────────┐     TCP      ┌──────────┐
│ Client App  ├──────────────► MBRCPVAL │
│   (SUT)     │◄─────────────┤ (Server) │
└─────────────┘              └──────────┘
```
MBRCPVAL acts as a mock server, validating client requests and testing client behavior.

#### 3.2.3 Passthrough/Proxy Mode
```
┌─────────────┐     TCP      ┌──────────┐     TCP      ┌──────────┐
│ Client App  ├──────────────► MBRCPVAL ├──────────────► Plugin   │
│             │◄─────────────┤ (Proxy)  │◄─────────────┤ (Server) │
└─────────────┘              └──────────┘              └──────────┘
```
MBRCPVAL sits between client and server, logging and validating all traffic.

---

## 4. Schema Definitions

### 4.1 JSON Schema Approach

All protocol messages will be defined using **JSON Schema Draft 2020-12**. This enables:
- Automated validation of message structure
- Documentation generation
- Code generation for multiple languages
- Test case generation

### 4.2 Schema Organization

```
schemas/
├── common/
│   ├── message.schema.json        # Base message envelope
│   ├── pagination.schema.json     # offset/limit patterns
│   └── track.schema.json          # Track metadata object
├── requests/
│   ├── player.schema.json         # player command
│   ├── protocol.schema.json       # protocol negotiation
│   ├── init.schema.json           # init request
│   ├── playerplay.schema.json     # playback controls
│   ├── browsetracks.schema.json   # library browse
│   └── ...
├── responses/
│   ├── nowplayingtrack.schema.json
│   ├── playerstatus.schema.json
│   └── ...
├── events/
│   ├── librarytagchanged.schema.json
│   ├── libraryratingchanged.schema.json
│   └── ...
└── discovery/
    ├── discovery-request.schema.json
    └── discovery-response.schema.json
```

### 4.3 Example Schema: Base Message

```json
{
  "$schema": "https://json-schema.org/draft/2020-12/schema",
  "$id": "https://mbrc.halrad.com/schemas/message.schema.json",
  "title": "MBRC Protocol Message",
  "description": "Base envelope for all MBRC protocol messages",
  "type": "object",
  "required": ["context", "data"],
  "properties": {
    "context": {
      "type": "string",
      "description": "Command identifier",
      "minLength": 1,
      "maxLength": 64,
      "pattern": "^[a-z][a-z0-9]*$"
    },
    "data": {
      "description": "Command payload (type varies by context)"
    }
  },
  "additionalProperties": false
}
```

### 4.4 Example Schema: Protocol Negotiation Request

```json
{
  "$schema": "https://json-schema.org/draft/2020-12/schema",
  "$id": "https://mbrc.halrad.com/schemas/requests/protocol.schema.json",
  "title": "Protocol Negotiation Request",
  "allOf": [{ "$ref": "../common/message.schema.json" }],
  "properties": {
    "context": { "const": "protocol" },
    "data": {
      "type": "object",
      "required": ["protocol_version"],
      "properties": {
        "protocol_version": {
          "type": "number",
          "enum": [2.0, 2.1, 3.0, 4.0, 4.5],
          "description": "Requested protocol version"
        },
        "no_broadcast": {
          "type": "boolean",
          "default": false,
          "description": "Disable UDP broadcast responses"
        }
      },
      "additionalProperties": false
    }
  }
}
```

### 4.5 Example Schema: Player Status Response

```json
{
  "$schema": "https://json-schema.org/draft/2020-12/schema",
  "$id": "https://mbrc.halrad.com/schemas/responses/playerstatus.schema.json",
  "title": "Player Status Response",
  "allOf": [{ "$ref": "../common/message.schema.json" }],
  "properties": {
    "context": { "const": "playerstatus" },
    "data": {
      "type": "object",
      "required": ["playerstate", "playervolume"],
      "properties": {
        "playerrepeat": {
          "type": "string",
          "enum": ["None", "All", "One"]
        },
        "playermute": { "type": "boolean" },
        "playershuffle": {
          "oneOf": [
            { "type": "boolean" },
            { "type": "string", "enum": ["off", "shuffle", "autodj"] }
          ]
        },
        "playerscrobble": { "type": "boolean" },
        "playerstate": {
          "type": "string",
          "enum": ["Playing", "Paused", "Stopped"]
        },
        "playervolume": {
          "type": "string",
          "pattern": "^([0-9]|[1-9][0-9]|100)$"
        }
      }
    }
  }
}
```

---

## 5. Test Categories

### 5.1 Category 1: Message Format Validation

Tests that each message conforms to JSON Schema:

| Test ID | Description | Validation |
|---------|-------------|------------|
| MSG-001 | Valid JSON syntax | Parse without error |
| MSG-002 | Required fields present | `context` and `data` exist |
| MSG-003 | Context is valid string | Non-empty, lowercase alphanumeric |
| MSG-004 | CRLF line termination | Message ends with `\r\n` |
| MSG-005 | UTF-8 encoding (no BOM) | First bytes not 0xEF 0xBB 0xBF |
| MSG-006 | No embedded newlines | Single line per message |

### 5.2 Category 2: Handshake Sequence

| Test ID | Description | Expected Result |
|---------|-------------|-----------------|
| HSK-001 | Correct handshake order | Connection established |
| HSK-002 | Missing player packet | Connection rejected/no response |
| HSK-003 | Missing protocol packet | Fallback to v2.0 behavior |
| HSK-004 | Protocol v4.0 negotiation | Server responds with v4.0 |
| HSK-005 | Protocol v4.5 negotiation | Server responds with v4.5 |
| HSK-006 | Init response burst (6 messages) | All 6 messages received |
| HSK-007 | Invalid protocol version | Server uses closest supported |

### 5.3 Category 3: Player Control Commands

| Test ID | Command | Validation |
|---------|---------|------------|
| PLY-001 | playerplay | Returns success, state changes to Playing |
| PLY-002 | playerpause | Returns success, state changes to Paused |
| PLY-003 | playerstop | Returns success, state changes to Stopped |
| PLY-004 | playerplaypause | Toggles between Playing/Paused |
| PLY-005 | playernext | Track changes, nowplayingtrack event fired |
| PLY-006 | playerprevious | Track changes, nowplayingtrack event fired |
| PLY-007 | playervolume (absolute) | Volume set to exact value |
| PLY-008 | playervolume (relative +) | Volume increases |
| PLY-009 | playervolume (relative -) | Volume decreases |
| PLY-010 | playervolume (boundary 0) | Volume at minimum |
| PLY-011 | playervolume (boundary 100) | Volume at maximum |
| PLY-012 | playermute toggle | Mute state toggles |
| PLY-013 | playershuffle modes | Cycles through off/shuffle/autodj |
| PLY-014 | playerrepeat modes | Cycles through None/All/One |

### 5.4 Category 4: Now Playing Commands

| Test ID | Command | Validation |
|---------|---------|------------|
| NP-001 | nowplayingtrack | Returns track metadata object |
| NP-002 | nowplayingcover | Returns Base64 image data |
| NP-003 | nowplayingcover (no art) | Returns empty string or null |
| NP-004 | nowplayingposition (get) | Returns current position in ms |
| NP-005 | nowplayingposition (seek) | Position changes to specified value |
| NP-006 | nowplayingrating (get) | Returns rating 0-5 or -1 |
| NP-007 | nowplayingrating (set) | Rating updated, event fired |
| NP-008 | nowplayinglyrics | Returns lyrics object with status |
| NP-009 | nowplayingdetails | Returns extended metadata |

### 5.5 Category 5: Queue Operations

| Test ID | Command | Validation |
|---------|---------|------------|
| QUE-001 | nowplayinglist | Returns paginated queue |
| QUE-002 | nowplayinglistplay | Track at index starts playing |
| QUE-003 | nowplayinglistremove | Track removed from queue |
| QUE-004 | nowplayinglistmove | Track position changes |
| QUE-005 | nowplayinglistclear | Queue emptied |
| QUE-006 | nowplayingqueuenext | Track inserted after current |
| QUE-007 | nowplayingqueuelast | Track appended to end |

### 5.6 Category 6: Library Browse

| Test ID | Command | Validation |
|---------|---------|------------|
| LIB-001 | browsegenres | Returns genre list with counts |
| LIB-002 | browseartists | Returns artist list with counts |
| LIB-003 | browsealbums | Returns album list |
| LIB-004 | browsetracks (v4.0) | Returns base track fields only |
| LIB-005 | browsetracks (v4.5) | Returns extended metadata fields |
| LIB-006 | Pagination (offset/limit) | Correct subset returned |
| LIB-007 | librarysearchartist | Partial match search works |
| LIB-008 | libraryalbumcover | Returns album art by name |

### 5.7 Category 7: v4.5 Extended Features

| Test ID | Command | Validation |
|---------|---------|------------|
| V45-001 | Extended metadata in browsetracks | playcount, skipcount, etc. present |
| V45-002 | librarysetrating | Rating updated on server |
| V45-003 | librarysetlove | Love status updated on server |
| V45-004 | librarysubscribe | Subscription acknowledged |
| V45-005 | librarytagchanged event | Event fired on tag change |
| V45-006 | libraryratingchanged event | Event fired on rating change |
| V45-007 | libraryunsubscribe | Unsubscription acknowledged |

### 5.8 Category 8: ARiA (Requires ARiA Enabled)

| Test ID | Command | Validation |
|---------|---------|------------|
| ARIA-001 | playerinitiatoraction (enabled) | Returns "Ok" |
| ARIA-002 | playerinitiatoraction (disabled) | Returns "Not Enabled" |
| ARIA-003 | pluginstoragepath (enabled) | Returns path info |
| ARIA-004 | pluginstoragepath (disabled) | Returns error |
| ARIA-005 | configureariahotkeys check | Returns hotkey status |

### 5.9 Category 9: Discovery Protocol (UDP)

| Test ID | Description | Validation |
|---------|-------------|------------|
| DSC-001 | Discovery request format | Valid JSON with context/address |
| DSC-002 | Discovery response format | Contains name, address, port |
| DSC-003 | Subnet matching | Correct interface IP returned |
| DSC-004 | Missing address field | Error response returned |
| DSC-005 | Multiple servers | All servers respond |

### 5.10 Category 10: Error Handling

| Test ID | Scenario | Expected |
|---------|----------|----------|
| ERR-001 | Invalid JSON | Connection continues, message ignored |
| ERR-002 | Unknown context | No response (silent ignore) |
| ERR-003 | Malformed data payload | Error response or ignore |
| ERR-004 | Missing required field | Error response |
| ERR-005 | Type mismatch in data | Error or coercion |
| ERR-006 | Connection timeout | Reconnection possible |
| ERR-007 | Missed ping/pong | Connection declared dead |

### 5.11 Category 11: Performance & Stress

| Test ID | Description | Threshold |
|---------|-------------|-----------|
| PERF-001 | Handshake latency | < 500ms |
| PERF-002 | Command response time | < 200ms typical |
| PERF-003 | Large library browse (10k tracks) | < 30s total |
| PERF-004 | Concurrent connections (10) | All functional |
| PERF-005 | Rapid fire commands (100/s) | No crashes, queue handles |
| PERF-006 | Cover art response time | < 2s |
| PERF-007 | Memory stability (1hr test) | No significant growth |

---

## 6. Test Case Definition Format

### 6.1 Test Case Structure

```yaml
test:
  id: "PLY-001"
  category: "Player Control"
  name: "Play Command Success"
  description: "Verify playerplay command starts playback"
  protocol_version: [4.0, 4.5]
  prerequisites:
    - "Connected and handshake complete"
    - "Track in now playing queue"
    - "Player in Stopped or Paused state"

  steps:
    - action: send
      message:
        context: "playerplay"
        data: null
      timeout: 5000

    - action: expect
      message:
        context: "playerstate"
        data:
          playerstate: "Playing"
      within: 2000

  validation:
    - type: schema
      schema: "responses/playerstate.schema.json"
    - type: field_value
      path: "$.data.playerstate"
      equals: "Playing"

  cleanup:
    - action: send
      message:
        context: "playerstop"
        data: null
```

### 6.2 Test Suite Organization

```
tests/
├── suites/
│   ├── handshake.yaml         # Connection tests
│   ├── player-control.yaml    # Transport commands
│   ├── now-playing.yaml       # Track info commands
│   ├── queue.yaml             # Queue operations
│   ├── library.yaml           # Browse commands
│   ├── v45-features.yaml      # Extended features
│   ├── aria.yaml              # ARiA tests
│   ├── discovery.yaml         # UDP discovery
│   ├── errors.yaml            # Error handling
│   └── performance.yaml       # Stress tests
├── fixtures/
│   ├── mock-library.json      # Test library data
│   ├── mock-track.json        # Sample track
│   └── mock-cover.base64      # Sample cover art
└── reports/
    └── (generated reports)
```

---

## 7. Implementation Plan

### 7.1 Phase 1: Core Infrastructure (Foundation)

**Deliverables:**
1. JSON Schema definitions for all protocol messages
2. Schema validator component
3. TCP client library with message framing
4. Basic CLI interface

**Tasks:**
- [ ] Create schema directory structure
- [ ] Define base message schema
- [ ] Define request schemas (40+ commands)
- [ ] Define response schemas (30+ responses)
- [ ] Define event schemas (10+ events)
- [ ] Implement schema loading and caching
- [ ] Implement AJV-based validator (or NJsonSchema for .NET)
- [ ] Implement TCP client with NDJSON framing
- [ ] Implement handshake sequence
- [ ] Create CLI framework with command structure

### 7.2 Phase 2: Server Validation Mode

**Deliverables:**
1. Server test runner
2. Test case parser and executor
3. Basic test suites (handshake, player, now playing)
4. Report generator (text, JSON, HTML)

**Tasks:**
- [ ] Implement test case YAML parser
- [ ] Implement test step executor
- [ ] Implement assertion library
- [ ] Create handshake test suite
- [ ] Create player control test suite
- [ ] Create now playing test suite
- [ ] Implement text report generator
- [ ] Implement JSON report generator
- [ ] Implement HTML report generator

### 7.3 Phase 3: Mock Server & Client Validation

**Deliverables:**
1. Mock MBRC server implementation
2. Client validation framework
3. Behavioral state machine

**Tasks:**
- [ ] Implement TCP server with client management
- [ ] Implement mock player state
- [ ] Implement mock library with test data
- [ ] Implement handshake validation for incoming clients
- [ ] Implement message validation for client requests
- [ ] Implement state machine for player behavior
- [ ] Create client test cases

### 7.4 Phase 4: Advanced Features

**Deliverables:**
1. UDP discovery mock
2. Library subscription testing
3. ARiA testing framework
4. Performance/stress testing

**Tasks:**
- [ ] Implement UDP multicast discovery responder
- [ ] Implement library event simulation
- [ ] Implement ARiA command validation
- [ ] Implement performance metrics collection
- [ ] Create stress test scenarios
- [ ] Implement concurrent connection testing

### 7.5 Phase 5: Developer Tools

**Deliverables:**
1. Interactive protocol explorer
2. Message inspector/logger
3. Proxy mode with traffic logging
4. GUI interface (optional)

**Tasks:**
- [ ] Implement REPL for interactive testing
- [ ] Implement proxy mode with passthrough
- [ ] Implement message capture and export
- [ ] Implement protocol diff tool
- [ ] Consider GUI implementation (WinForms/WPF or Electron)

---

## 8. Technology Recommendations

### 8.1 Primary Platform: .NET 8+

**Rationale:**
- Same platform as existing plugin (code reuse)
- Excellent JSON handling (System.Text.Json, Newtonsoft.Json)
- Strong async networking (System.Net.Sockets)
- Cross-platform via .NET Core
- NJsonSchema for JSON Schema validation

**Libraries:**
- `NJsonSchema` - JSON Schema validation for .NET
- `YamlDotNet` - YAML test case parsing
- `Spectre.Console` - Rich CLI output
- `xUnit` - Unit testing framework
- `BenchmarkDotNet` - Performance testing

### 8.2 Alternative: Node.js/TypeScript

**Rationale:**
- Native JSON handling
- Excellent protocol testing ecosystem
- Cross-platform
- Easy GUI via Electron if needed

**Libraries:**
- `ajv` - JSON Schema validation (fastest)
- `jest` - Testing framework
- `commander` - CLI framework
- `ws` - WebSocket (for future expansion)

### 8.3 Schema Validation Tools

| Tool | Language | Performance | Features |
|------|----------|-------------|----------|
| AJV | JavaScript | Excellent | Full Draft 2020-12, code generation |
| NJsonSchema | C# | Good | Draft 4/6/7, code generation |
| json-schema-validator | Java | Good | Full spec compliance |
| jsonschema | Python | Moderate | Full spec compliance |

---

## 9. Usage Guidelines

### 9.1 Validating a Server Plugin

```bash
# Basic server validation against localhost
mbrcpval server --host localhost --port 3000 --suite handshake,player

# Full validation with v4.5 protocol
mbrcpval server --host 192.168.1.10 --port 3000 --protocol 4.5 --suite all

# Generate HTML report
mbrcpval server --host localhost --port 3000 --report html --output report.html

# Test specific categories
mbrcpval server --host localhost --port 3000 --suite library --tests LIB-001,LIB-002
```

### 9.2 Validating a Client Application

```bash
# Start mock server for client testing
mbrcpval mock-server --port 3000 --library fixtures/test-library.json

# Start mock server with specific behavior
mbrcpval mock-server --port 3000 --behavior strict --log traffic.jsonl

# Validate client connection (observe mode)
mbrcpval client --port 3000 --expect handshake --timeout 30
```

### 9.3 Proxy Mode for Debugging

```bash
# Proxy between client and server, log all traffic
mbrcpval proxy --listen 3001 --target localhost:3000 --log traffic.jsonl

# Proxy with validation
mbrcpval proxy --listen 3001 --target localhost:3000 --validate --report proxy-report.json
```

### 9.4 Interactive Explorer

```bash
# Start interactive REPL
mbrcpval repl --host localhost --port 3000

# In REPL:
> connect
Connected to localhost:3000, protocol v4.5
> send playerplay
< {"context":"playerstate","data":{"playerstate":"Playing",...}}
> browse genres
< [{"genre":"Rock","count":250},{"genre":"Jazz","count":100},...]
> exit
```

### 9.5 CI/CD Integration

```yaml
# GitHub Actions example
- name: Protocol Validation
  run: |
    mbrcpval server \
      --host ${{ secrets.MUSICBEE_HOST }} \
      --port 3000 \
      --suite all \
      --protocol 4.5 \
      --report junit \
      --output test-results.xml

- name: Upload Results
  uses: actions/upload-artifact@v4
  with:
    name: protocol-test-results
    path: test-results.xml
```

---

## 10. Best Practices

### 10.1 For Plugin Developers

1. **Run full validation before release**
   ```bash
   mbrcpval server --suite all --protocol 4.5 --report html
   ```

2. **Test backward compatibility**
   ```bash
   mbrcpval server --protocol 4.0 --suite handshake,player
   mbrcpval server --protocol 4.5 --suite handshake,player
   ```

3. **Validate edge cases**
   - Empty library
   - Very large library (100k+ tracks)
   - Unicode in metadata
   - Missing cover art
   - Rapid command sequences

4. **Test ARiA separately** (security feature)
   ```bash
   mbrcpval server --suite aria --aria-enabled
   mbrcpval server --suite aria --aria-disabled
   ```

### 10.2 For Client Developers

1. **Use mock server for development**
   ```bash
   mbrcpval mock-server --port 3000 --library test-data.json
   ```

2. **Test protocol negotiation**
   - Request v4.5, verify extended fields
   - Request v4.0, verify no extended fields

3. **Validate message format before sending**
   ```bash
   echo '{"context":"playerplay","data":null}' | mbrcpval validate --schema requests
   ```

4. **Test reconnection handling**
   - Server restarts
   - Network interruption
   - Timeout scenarios

### 10.3 For QA/Testing

1. **Generate comprehensive reports**
   ```bash
   mbrcpval server --suite all --report html,json --output results/
   ```

2. **Compare versions**
   ```bash
   mbrcpval server --host old-server:3000 --report json --output v1.4.json
   mbrcpval server --host new-server:3000 --report json --output v1.5.json
   mbrcpval diff v1.4.json v1.5.json
   ```

3. **Performance regression testing**
   ```bash
   mbrcpval server --suite performance --baseline baseline.json --output current.json
   ```

---

## 11. Report Formats

### 11.1 Text Report

```
MBRCPVAL Protocol Validation Report
====================================
Target: localhost:3000
Protocol: v4.5
Plugin Version: 1.5.25
Date: 2026-01-01 12:00:00

SUMMARY
-------
Total:  150 tests
Passed: 145 tests (96.7%)
Failed: 3 tests
Skipped: 2 tests

FAILED TESTS
------------
[FAIL] PLY-008 playervolume (relative +)
       Expected: volume increased by 5
       Actual: volume unchanged
       Message: {"context":"playervolume","data":"75"}

[FAIL] V45-003 librarysetlove
       Expected: success response
       Actual: no response (timeout)

[FAIL] PERF-003 Large library browse
       Expected: < 30s
       Actual: 45.2s
```

### 11.2 JSON Report

```json
{
  "report": {
    "tool": "mbrcpval",
    "version": "1.0.0",
    "timestamp": "2026-01-01T12:00:00Z",
    "target": {
      "host": "localhost",
      "port": 3000,
      "protocol": 4.5,
      "plugin_version": "1.5.25"
    },
    "summary": {
      "total": 150,
      "passed": 145,
      "failed": 3,
      "skipped": 2,
      "duration_ms": 45230
    },
    "results": [
      {
        "id": "PLY-001",
        "name": "Play Command Success",
        "status": "passed",
        "duration_ms": 125
      },
      {
        "id": "PLY-008",
        "name": "playervolume (relative +)",
        "status": "failed",
        "duration_ms": 5002,
        "error": {
          "expected": "volume increased by 5",
          "actual": "volume unchanged",
          "message": "{\"context\":\"playervolume\",\"data\":\"75\"}"
        }
      }
    ]
  }
}
```

---

## 12. Appendix A: Command Catalog

### 12.1 Connection Commands

| Context | Direction | Description |
|---------|-----------|-------------|
| player | C→S | Client identification |
| protocol | C↔S | Version negotiation |
| init | C→S | Request initial state |
| ping | C→S | Keepalive request |
| pong | S→C | Keepalive response |
| verifyconnection | C→S | Connection check |
| pluginversion | C→S | Get plugin version |
| plugininstanceid | C→S | Get unique instance ID |

### 12.2 Player Control Commands

| Context | Direction | Description |
|---------|-----------|-------------|
| playerplay | C→S | Start playback |
| playerpause | C→S | Pause playback |
| playerplaypause | C→S | Toggle play/pause |
| playerstop | C→S | Stop playback |
| playernext | C→S | Next track |
| playerprevious | C→S | Previous track |
| playernextalbum | C→S | Next album (v4.5) |
| playerpreviousalbum | C→S | Previous album (v4.5) |
| playervolume | C↔S | Get/set volume |
| playermute | C↔S | Get/set mute |
| playershuffle | C↔S | Get/set shuffle |
| playerrepeat | C↔S | Get/set repeat |
| playerautodj | C↔S | Get/set auto DJ |
| playerstatus | C→S | Get full player state |
| playerstate | S→C | Player state event |
| playeroutput | C→S | Get/set output device |

### 12.3 Now Playing Commands

| Context | Direction | Description |
|---------|-----------|-------------|
| nowplayingtrack | C↔S | Track metadata |
| nowplayingcover | C↔S | Album art (Base64) |
| nowplayingposition | C↔S | Position/seek |
| nowplayingrating | C↔S | Track rating |
| nowplayinglovestatus | S→C | Love status |
| nowplayinglfmrating | C↔S | Last.fm love/ban |
| nowplayinglyrics | C↔S | Track lyrics |
| nowplayingdetails | C→S | Extended details |
| nowplayingtagchange | C→S | Edit metadata |

### 12.4 Queue Commands

| Context | Direction | Description |
|---------|-----------|-------------|
| nowplayinglist | C→S | Get queue (paginated) |
| nowplayinglistplay | C→S | Play at index |
| nowplayinglistremove | C→S | Remove track |
| nowplayinglistmove | C→S | Move track |
| nowplayinglistclear | C→S | Clear queue |
| nowplayingqueuenext | C→S | Queue next |
| nowplayingqueuelast | C→S | Queue last |

### 12.5 Library Commands

| Context | Direction | Description |
|---------|-----------|-------------|
| browsegenres | C→S | List genres |
| browseartists | C→S | List artists |
| browsealbums | C→S | List albums |
| browsetracks | C→S | List tracks |
| librarysearchartist | C→S | Search artists |
| librarysearchalbum | C→S | Search albums |
| librarysearchgenre | C→S | Search genres |
| librarysearchtitle | C→S | Search tracks |
| libraryartistalbums | C→S | Albums by artist |
| libraryalbumtracks | C→S | Tracks by album |
| librarygenreartists | C→S | Artists by genre |
| libraryqueuegenre | C→S | Play genre |
| libraryqueueartist | C→S | Play artist |
| libraryqueuealbum | C→S | Play album |
| libraryqueuetrack | C→S | Play track |
| libraryplayall | C→S | Play all |
| libraryalbumcover | C→S | Get album cover |
| librarysetrating | C→S | Set track rating (v4.5) |
| librarysetlove | C→S | Set love status (v4.5) |

### 12.6 Library Subscription Events (v4.5)

| Context | Direction | Description |
|---------|-----------|-------------|
| librarysubscribe | C→S | Subscribe to events |
| libraryunsubscribe | C→S | Unsubscribe |
| librarytagchanged | S→C | Tag modified |
| libraryratingchanged | S→C | Rating changed |
| libraryplaycounterchanged | S→C | Play/skip count |
| libraryfiledeleted | S→C | Track deleted |
| libraryfileadded | S→C | Track added |

### 12.7 Playlist Commands

| Context | Direction | Description |
|---------|-----------|-------------|
| playlistlist | C→S | List playlists |
| playlistplay | C→S | Play playlist |
| playlistqueue | C→S | Queue playlist |
| playlistcreated | S→C | Playlist created event |
| playlistupdated | S→C | Playlist updated event |
| playlistdeleted | S→C | Playlist deleted event |

### 12.8 Advanced Commands (v4.5)

| Context | Direction | Description |
|---------|-----------|-------------|
| playerinitiatoraction | C→S | Execute ARiA script |
| pluginstoragepath | C→S | Get storage paths |
| configureariahotkeys | C→S | Configure hotkeys |
| playervisualizer | C→S | Set visualizer |
| playervisualizerlist | C→S | List visualizers |
| podcastsubscriptions | C→S | List podcasts |
| podcastepisodes | C→S | List episodes |
| podcastplay | C→S | Play episode |
| radiostations | C→S | List radio stations |

---

## 13. Appendix B: State Machine

### 13.1 Player State Machine

```
                 ┌──────────────────┐
                 │     STOPPED      │
                 └────────┬─────────┘
                          │
              ┌───────────┼───────────┐
              │ playerplay│ (queued)  │
              ▼           │           │
        ┌──────────┐      │     ┌─────────────┐
        │ PLAYING  │◄─────┴─────│ Queue Track │
        └────┬─────┘            └─────────────┘
             │
     playerpause
             │
             ▼
        ┌──────────┐
        │  PAUSED  │
        └────┬─────┘
             │
             └────► playerplay ────► PLAYING
                    playerstop ────► STOPPED
```

### 13.2 Connection State Machine

```
┌──────────────┐
│ DISCONNECTED │
└──────┬───────┘
       │ TCP Connect
       ▼
┌──────────────┐
│  CONNECTED   │ (raw TCP, no handshake)
└──────┬───────┘
       │ Send player/protocol/init
       ▼
┌──────────────┐
│ HANDSHAKING  │
└──────┬───────┘
       │ Receive init burst (6 messages)
       ▼
┌──────────────┐
│    READY     │ (fully operational)
└──────┬───────┘
       │
       ├──── ping/pong ──► (keepalive)
       │
       └──── timeout ──► DISCONNECTED
```

---

## 14. Revision History

| Version | Date | Author | Changes |
|---------|------|--------|---------|
| 1.0.0-draft | 2026-01-01 | Claude/MBRC Team | Initial specification |

---

## 15. References

1. MBRC-Server-Plugin-Protocol.html - Protocol SDK Documentation
2. MBXRemote plugin.html - Plugin installation and features
3. MBXRemote appendix.html - Technical appendix
4. mbrc-plugin-main-server source code - Reference implementation
5. tntctl (MBXRemote) source code - Client reference implementation
6. JSON Schema Draft 2020-12 Specification
7. UPNP/DLNA Device Certification Process

---

*This specification is a living document and will be updated as the protocol evolves.*
