# MBRCPVAL Master Implementation Plan

> **Related Documentation:**
> - [Quick Start](MBRCPVAL-QUICKSTART.md) - Getting started guide
> - [Overview](MBRCPVAL-Overview.md) - High-level introduction
> - [Full Specification](MBRCPVAL-SPEC.md) - Protocol specification and test categories
> - [Implementation Guide](MBRCPVAL-IMPLEMENTATION.md) - Technical details
> - [Changelog](MBRCPVAL-CHANGELOG.md) - Version history

---

## Execution Roadmap - All Phases

**Created:** 2026-01-01
**Status:** ACTIVE IMPLEMENTATION

---

## Phase Overview

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                        MBRCPVAL IMPLEMENTATION TIMELINE                      │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                              │
│  PHASE 1: Foundation          PHASE 2: Testing        PHASE 3: Mock Server  │
│  ─────────────────────        ───────────────         ──────────────────── │
│  • JSON Schemas               • Test Runner           • MockMbrcServer      │
│  • Schema Validator           • YAML Parser           • MockPlayerState     │
│  • TCP Client                 • Assertions            • MockLibrary         │
│  • Message Parser             • Test Suites           • Client Validation   │
│  • CLI Framework              • Reporters             • State Machine       │
│                                                                              │
│  PHASE 4: Advanced            PHASE 5: Developer Tools                      │
│  ────────────────             ──────────────────────                        │
│  • UDP Discovery              • Interactive REPL                             │
│  • Library Subscriptions      • Proxy Mode                                   │
│  • ARiA Testing               • Message Inspector                            │
│  • Performance Tests          • GUI (Optional)                               │
│  • Stress Tests               • Protocol Diff Tool                           │
│                                                                              │
└─────────────────────────────────────────────────────────────────────────────┘
```

---

## Phase 1: Core Infrastructure

### 1.1 Project Setup
- [ ] Create mbrcpval solution and project structure
- [ ] Add NuGet packages: NJsonSchema, Newtonsoft.Json, Spectre.Console, YamlDotNet
- [ ] Set up project references

### 1.2 JSON Schemas
- [ ] Create `schemas/common/message.schema.json` - Base envelope
- [ ] Create `schemas/common/pagination.schema.json` - Offset/limit
- [ ] Create `schemas/common/track.schema.json` - Track metadata
- [ ] Create request schemas (40+ commands):
  - [ ] player, protocol, init, ping, pong
  - [ ] playerplay, playerpause, playerstop, playernext, playerprevious
  - [ ] playervolume, playermute, playershuffle, playerrepeat
  - [ ] nowplayingtrack, nowplayingcover, nowplayingposition, nowplayingrating
  - [ ] nowplayinglist, nowplayinglistplay, nowplayinglistremove
  - [ ] browsegenres, browseartists, browsealbums, browsetracks
  - [ ] librarysearchartist, librarysearchalbum, librarysearchtitle
  - [ ] librarysetrating, librarysetlove
  - [ ] playlistlist, playlistplay
  - [ ] radiostations, playeroutput
  - [ ] playerinitiatoraction, pluginstoragepath (ARiA)
  - [ ] podcastsubscriptions, podcastepisodes (v4.5)
- [ ] Create response schemas (30+ responses)
- [ ] Create event schemas (15+ events)
- [ ] Create discovery schemas (UDP)

### 1.3 Core Components
- [ ] `MbrcSocketClient.cs` - TCP client with NDJSON framing
- [ ] `MessageParser.cs` - JSON parsing with error handling
- [ ] `SchemaValidator.cs` - NJsonSchema wrapper
- [ ] `Constants.cs` - All protocol context strings

### 1.4 CLI Framework
- [ ] Program.cs with Spectre.Console command structure
- [ ] `ServerCommand.cs` - Validate server mode
- [ ] `ValidateCommand.cs` - Validate single message
- [ ] Options and configuration handling

**Deliverables:**
```
mbrcpval/
├── mbrcpval.csproj
├── src/
│   ├── Core/
│   │   ├── MbrcSocketClient.cs
│   │   ├── MessageParser.cs
│   │   ├── SchemaValidator.cs
│   │   └── Constants.cs
│   └── CLI/
│       ├── Program.cs
│       └── Commands/
│           ├── ServerCommand.cs
│           └── ValidateCommand.cs
└── schemas/
    ├── common/*.schema.json
    ├── requests/*.schema.json
    ├── responses/*.schema.json
    └── events/*.schema.json
```

---

## Phase 2: Test Framework

### 2.1 Test Case Format
- [ ] Define YAML test case schema
- [ ] Create `TestCase.cs` model
- [ ] Create `TestStep.cs` model
- [ ] Create `Assertion.cs` model
- [ ] Implement `YamlTestLoader.cs`

### 2.2 Test Runner
- [ ] `TestRunner.cs` - Execution engine
- [ ] `TestResult.cs` - Result model
- [ ] `StepExecutor.cs` - Step execution
- [ ] `AssertionEvaluator.cs` - Assertion evaluation

### 2.3 Test Suites
- [ ] `handshake.yaml` - Connection/handshake tests (HSK-001 to HSK-010)
- [ ] `player-control.yaml` - Playback tests (PLY-001 to PLY-020)
- [ ] `now-playing.yaml` - Track info tests (NP-001 to NP-015)
- [ ] `queue.yaml` - Queue operation tests (QUE-001 to QUE-010)
- [ ] `library.yaml` - Library browse tests (LIB-001 to LIB-020)
- [ ] `v45-features.yaml` - Extended features (V45-001 to V45-015)
- [ ] `errors.yaml` - Error handling tests (ERR-001 to ERR-010)

### 2.4 Reporters
- [ ] `TextReporter.cs` - Console output with Spectre.Console
- [ ] `JsonReporter.cs` - JSON export
- [ ] `HtmlReporter.cs` - HTML report with template
- [ ] `JunitReporter.cs` - CI/CD integration

**Deliverables:**
```
mbrcpval/
├── src/
│   ├── Testing/
│   │   ├── TestRunner.cs
│   │   ├── TestCase.cs
│   │   ├── TestStep.cs
│   │   ├── Assertion.cs
│   │   ├── YamlTestLoader.cs
│   │   └── AssertionEvaluator.cs
│   └── Reporting/
│       ├── TextReporter.cs
│       ├── JsonReporter.cs
│       ├── HtmlReporter.cs
│       └── JunitReporter.cs
└── tests/
    └── suites/
        ├── handshake.yaml
        ├── player-control.yaml
        ├── now-playing.yaml
        ├── queue.yaml
        ├── library.yaml
        ├── v45-features.yaml
        └── errors.yaml
```

---

## Phase 3: Mock Server

### 3.1 Mock Server Core
- [ ] `MockMbrcServer.cs` - TCP listener with client management
- [ ] `MockClientConnection.cs` - Per-client state
- [ ] `MessageRouter.cs` - Context-based routing

### 3.2 Mock State
- [ ] `MockPlayerState.cs` - Player state simulation
- [ ] `MockLibrary.cs` - Library data store
- [ ] `MockTrack.cs`, `MockAlbum.cs`, `MockArtist.cs` - Data models

### 3.3 Client Validation
- [ ] `ClientValidator.cs` - Validate incoming client messages
- [ ] `HandshakeValidator.cs` - Validate handshake sequence
- [ ] `ConnectionStateMachine.cs` - State tracking

### 3.4 CLI Integration
- [ ] `MockCommand.cs` - Start mock server
- [ ] Library generation from JSON or random

**Deliverables:**
```
mbrcpval/
├── src/
│   ├── Mock/
│   │   ├── MockMbrcServer.cs
│   │   ├── MockClientConnection.cs
│   │   ├── MessageRouter.cs
│   │   ├── MockPlayerState.cs
│   │   ├── MockLibrary.cs
│   │   └── Models/
│   │       ├── MockTrack.cs
│   │       ├── MockAlbum.cs
│   │       └── MockArtist.cs
│   └── Validation/
│       ├── ClientValidator.cs
│       ├── HandshakeValidator.cs
│       └── ConnectionStateMachine.cs
└── tests/
    └── fixtures/
        ├── mock-library.json
        └── mock-library-large.json
```

---

## Phase 4: Advanced Features

### 4.1 UDP Discovery
- [ ] `UdpDiscoveryClient.cs` - Discovery protocol client
- [ ] `UdpDiscoveryServer.cs` - Mock discovery responder
- [ ] `discovery.yaml` - Discovery test suite (DSC-001 to DSC-010)

### 4.2 Library Subscriptions (v4.5)
- [ ] Event simulation in mock server
- [ ] `librarysubscribe`/`libraryunsubscribe` handling
- [ ] Event generation: `librarytagchanged`, `libraryratingchanged`, etc.

### 4.3 ARiA Testing
- [ ] `aria.yaml` - ARiA test suite (ARIA-001 to ARIA-010)
- [ ] Mock ARiA command handling
- [ ] Security setting simulation

### 4.4 Performance Testing
- [ ] `performance.yaml` - Performance test suite (PERF-001 to PERF-010)
- [ ] `StressTestRunner.cs` - Concurrent connection testing
- [ ] Latency measurement and reporting
- [ ] Throughput benchmarking

**Deliverables:**
```
mbrcpval/
├── src/
│   ├── Discovery/
│   │   ├── UdpDiscoveryClient.cs
│   │   └── UdpDiscoveryServer.cs
│   └── Performance/
│       ├── StressTestRunner.cs
│       └── LatencyMeasurement.cs
└── tests/
    └── suites/
        ├── discovery.yaml
        ├── aria.yaml
        └── performance.yaml
```

---

## Phase 5: Developer Tools

### 5.1 Interactive REPL
- [ ] `ReplCommand.cs` - REPL entry point
- [ ] `ReplSession.cs` - Interactive session management
- [ ] Command parsing and execution
- [ ] History and auto-complete

### 5.2 Proxy Mode
- [ ] `ProxyCommand.cs` - Proxy entry point
- [ ] `ProxyServer.cs` - Bidirectional proxy
- [ ] Traffic logging to JSONL
- [ ] Real-time validation

### 5.3 Message Inspector
- [ ] `InspectCommand.cs` - Analyze captured traffic
- [ ] Message diff and comparison
- [ ] Protocol statistics

### 5.4 GUI (Optional)
- [ ] Electron or WPF wrapper
- [ ] Visual test runner
- [ ] Traffic visualization

**Deliverables:**
```
mbrcpval/
└── src/
    ├── CLI/
    │   └── Commands/
    │       ├── ReplCommand.cs
    │       ├── ProxyCommand.cs
    │       └── InspectCommand.cs
    ├── Repl/
    │   └── ReplSession.cs
    └── Proxy/
        └── ProxyServer.cs
```

---

## Documentation Deliverables

### HTML Pages (for tntctl/deploy/www/)
- [ ] `mbrcpval.html` - Main tool page with overview, usage, examples
- [ ] Update navigation in all pages to include MBRCPVAL link

### Markdown Documentation
- [ ] `MBRCPVAL-SPEC.md` - Protocol specification (COMPLETE)
- [ ] `MBRCPVAL-IMPLEMENTATION.md` - Technical guide (COMPLETE)
- [ ] `MBRCPVAL-MASTERPLAN.md` - This document (COMPLETE)
- [ ] `MBRCPVAL-CHANGELOG.md` - Version history
- [ ] `MBRCPVAL-QUICKSTART.md` - Getting started guide

---

## Success Criteria

### Phase 1 Complete When:
- [ ] All JSON schemas validate against test fixtures
- [ ] TCP client can connect and complete handshake
- [ ] Single message validation works via CLI

### Phase 2 Complete When:
- [ ] Test runner executes all test suites
- [ ] 80%+ of test cases pass against live plugin
- [ ] Reports generate in all formats

### Phase 3 Complete When:
- [ ] Mock server accepts client connections
- [ ] Basic player commands work
- [ ] Library browse returns data

### Phase 4 Complete When:
- [ ] UDP discovery works
- [ ] Subscription events fire correctly
- [ ] Performance metrics collected

### Phase 5 Complete When:
- [ ] REPL interactive mode works
- [ ] Proxy logs all traffic
- [ ] Tool is usable for real development

---

## Parallel Agent Assignments

| Agent | Responsibility | Files |
|-------|---------------|-------|
| Schema Agent | Create all JSON schemas | `schemas/**/*.schema.json` |
| Core Agent | Build core infrastructure | `src/Core/*.cs` |
| Test Agent | Build test framework | `src/Testing/*.cs`, `tests/suites/*.yaml` |
| Docs Agent | Keep documentation updated | `Documentation/*.md`, `www/mbrcpval.html` |

---

*Last Updated: 2026-01-01*
