# MBRCPVAL Implementation Guide

> **Related Documentation:**
> - [Quick Start](MBRCPVAL-QUICKSTART.md) - Getting started guide
> - [Overview](MBRCPVAL-Overview.md) - High-level introduction
> - [Full Specification](MBRCPVAL-SPEC.md) - Protocol specification and test categories
> - [Master Plan](MBRCPVAL-MASTERPLAN.md) - Implementation roadmap
> - [Changelog](MBRCPVAL-CHANGELOG.md) - Version history

---

## Technical Implementation Details & Research Findings

**Version:** 1.0.0-draft
**Date:** 2026-01-01

---

## 1. Recommended Technology Stack

### 1.1 Primary Platform: .NET 8+

Based on research and alignment with the existing MBRC codebase, .NET 8+ is the recommended platform:

| Component | Library | Rationale |
|-----------|---------|-----------|
| JSON Schema | NJsonSchema | Free, code generation, .NET native |
| JSON Parsing | Newtonsoft.Json | Already used in MBRC plugin |
| Unit Testing | NUnit | Already used in mbrc-plugin.Tests |
| Property Testing | FsCheck | .NET native fuzzing |
| CLI Framework | Spectre.Console | Rich terminal output |
| Load Testing | NBomber | C# native load testing |
| YAML Parsing | YamlDotNet | Test case definitions |

### 1.2 Alternative: Node.js/TypeScript

For teams preferring JavaScript ecosystem:

| Component | Library |
|-----------|---------|
| JSON Schema | AJV (fastest) |
| TCP Client | net (built-in) |
| Testing | Jest |
| CLI | Commander |
| Load Testing | Artillery |

---

## 2. Project Structure

### 2.1 Proposed Directory Layout

```
mbrcpval/
├── src/
│   ├── Core/
│   │   ├── SocketClient.cs           # TCP client with NDJSON framing
│   │   ├── SocketServer.cs           # Mock server for client testing
│   │   ├── MessageParser.cs          # JSON parsing with error handling
│   │   ├── SchemaValidator.cs        # NJsonSchema wrapper
│   │   ├── SequenceValidator.cs      # Message sequence verification
│   │   └── StateMachine.cs           # Connection state machine
│   ├── Testing/
│   │   ├── TestRunner.cs             # Test execution engine
│   │   ├── TestCase.cs               # Test case model
│   │   ├── TestResult.cs             # Test result model
│   │   ├── Assertions.cs             # Custom assertions
│   │   └── Fixtures/
│   │       ├── MockLibrary.cs        # Simulated music library
│   │       └── MockPlayer.cs         # Simulated player state
│   ├── Reporting/
│   │   ├── TextReporter.cs           # Console output
│   │   ├── JsonReporter.cs           # JSON export
│   │   ├── HtmlReporter.cs           # HTML report
│   │   └── JunitReporter.cs          # CI/CD integration
│   ├── CLI/
│   │   ├── Program.cs                # Entry point
│   │   ├── Commands/
│   │   │   ├── ServerCommand.cs      # Validate server
│   │   │   ├── ClientCommand.cs      # Validate client
│   │   │   ├── ProxyCommand.cs       # Proxy mode
│   │   │   ├── MockCommand.cs        # Mock server
│   │   │   └── ReplCommand.cs        # Interactive mode
│   │   └── Options.cs                # CLI options
│   └── Discovery/
│       ├── UdpDiscoveryClient.cs     # Discovery protocol client
│       └── UdpDiscoveryServer.cs     # Mock discovery responder
├── schemas/
│   ├── common/
│   │   ├── message.schema.json
│   │   ├── pagination.schema.json
│   │   └── track.schema.json
│   ├── requests/
│   │   └── *.schema.json             # All request schemas
│   ├── responses/
│   │   └── *.schema.json             # All response schemas
│   └── events/
│       └── *.schema.json             # All event schemas
├── tests/
│   ├── suites/
│   │   ├── handshake.yaml
│   │   ├── player-control.yaml
│   │   ├── now-playing.yaml
│   │   ├── queue.yaml
│   │   ├── library.yaml
│   │   ├── v45-features.yaml
│   │   ├── aria.yaml
│   │   ├── discovery.yaml
│   │   ├── errors.yaml
│   │   └── performance.yaml
│   └── fixtures/
│       ├── mock-library.json
│       ├── mock-track.json
│       └── mock-cover.base64
├── docs/
│   └── (generated documentation)
└── mbrcpval.csproj
```

---

## 3. Core Component Implementations

### 3.1 Socket Client with NDJSON Framing

```csharp
public class MbrcSocketClient : IAsyncDisposable
{
    private readonly TcpClient _tcpClient;
    private readonly NetworkStream _stream;
    private readonly StreamReader _reader;
    private readonly StreamWriter _writer;
    private readonly ILogger _logger;

    private const string LineTerminator = "\r\n";

    public async Task ConnectAsync(string host, int port, CancellationToken ct = default)
    {
        _tcpClient = new TcpClient();
        _tcpClient.NoDelay = true; // Disable Nagle
        _tcpClient.ReceiveTimeout = 30000;
        _tcpClient.SendTimeout = 5000;

        await _tcpClient.ConnectAsync(host, port, ct);
        _stream = _tcpClient.GetStream();
        _reader = new StreamReader(_stream, Encoding.UTF8);
        _writer = new StreamWriter(_stream, new UTF8Encoding(false)) // No BOM
        {
            NewLine = LineTerminator,
            AutoFlush = true
        };
    }

    public async Task SendAsync(SocketMessage message, CancellationToken ct = default)
    {
        var json = JsonConvert.SerializeObject(message);
        _logger.LogDebug("SEND: {Json}", json);
        await _writer.WriteLineAsync(json.AsMemory(), ct);
    }

    public async Task<SocketMessage?> ReceiveAsync(TimeSpan timeout, CancellationToken ct = default)
    {
        using var cts = CancellationTokenSource.CreateLinkedTokenSource(ct);
        cts.CancelAfter(timeout);

        try
        {
            var line = await _reader.ReadLineAsync(cts.Token);
            if (line == null) return null;

            _logger.LogDebug("RECV: {Json}", line);
            return JsonConvert.DeserializeObject<SocketMessage>(line);
        }
        catch (OperationCanceledException)
        {
            return null;
        }
    }

    public async Task<HandshakeResult> PerformHandshakeAsync(
        string clientId = "mbrcpval",
        double protocolVersion = 4.5,
        CancellationToken ct = default)
    {
        // Packet 0: Player identification
        await SendAsync(new SocketMessage(Constants.Player, clientId), ct);

        // Packet 1: Protocol negotiation
        var protocolData = new { protocol_version = protocolVersion, no_broadcast = false };
        await SendAsync(new SocketMessage(Constants.Protocol, protocolData), ct);

        // Receive protocol response
        var protocolResponse = await ReceiveAsync(TimeSpan.FromSeconds(5), ct);
        if (protocolResponse?.Context != Constants.Protocol)
            return HandshakeResult.Failed("No protocol response");

        // Packet 2: Init request
        await SendAsync(new SocketMessage(Constants.Init, null), ct);

        // Receive init burst (6 messages)
        var initMessages = new List<SocketMessage>();
        for (int i = 0; i < 6; i++)
        {
            var msg = await ReceiveAsync(TimeSpan.FromSeconds(2), ct);
            if (msg != null) initMessages.Add(msg);
        }

        return new HandshakeResult
        {
            Success = true,
            NegotiatedVersion = Convert.ToDouble(protocolResponse.Data),
            InitMessages = initMessages
        };
    }

    public async ValueTask DisposeAsync()
    {
        _writer?.Dispose();
        _reader?.Dispose();
        _stream?.Dispose();
        _tcpClient?.Dispose();
    }
}
```

### 3.2 Schema Validator

```csharp
public class SchemaValidator
{
    private readonly Dictionary<string, JsonSchema> _schemas = new();
    private readonly string _schemaDirectory;

    public SchemaValidator(string schemaDirectory)
    {
        _schemaDirectory = schemaDirectory;
        LoadSchemas();
    }

    private void LoadSchemas()
    {
        foreach (var file in Directory.GetFiles(_schemaDirectory, "*.schema.json", SearchOption.AllDirectories))
        {
            var schemaJson = File.ReadAllText(file);
            var schema = JsonSchema.FromJsonAsync(schemaJson).GetAwaiter().GetResult();
            var key = Path.GetFileNameWithoutExtension(file).Replace(".schema", "");
            _schemas[key] = schema;
        }
    }

    public ValidationResult Validate(SocketMessage message)
    {
        // First validate base message structure
        if (!_schemas.TryGetValue("message", out var baseSchema))
            return ValidationResult.Error("Base message schema not found");

        var json = JsonConvert.SerializeObject(message);
        var baseErrors = baseSchema.Validate(json);

        if (baseErrors.Any())
            return new ValidationResult(false, baseErrors.Select(e => e.ToString()).ToList());

        // Then validate context-specific schema
        var contextSchemaKey = $"{message.Context}";
        if (_schemas.TryGetValue(contextSchemaKey, out var contextSchema))
        {
            var contextErrors = contextSchema.Validate(json);
            if (contextErrors.Any())
                return new ValidationResult(false, contextErrors.Select(e => e.ToString()).ToList());
        }

        return ValidationResult.Success();
    }

    public ValidationResult ValidateData(string context, object data)
    {
        var schemaKey = $"{context}-data";
        if (!_schemas.TryGetValue(schemaKey, out var schema))
            return ValidationResult.Success(); // No specific data schema

        var json = JsonConvert.SerializeObject(data);
        var errors = schema.Validate(json);

        return errors.Any()
            ? new ValidationResult(false, errors.Select(e => e.ToString()).ToList())
            : ValidationResult.Success();
    }
}
```

### 3.3 Connection State Machine

```csharp
public class ConnectionStateMachine
{
    public enum State
    {
        Disconnected,
        Connected,
        AwaitingPlayer,
        AwaitingProtocol,
        AwaitingInit,
        Ready
    }

    private State _currentState = State.Disconnected;
    private double _negotiatedProtocol = 0;

    private readonly Dictionary<(State, string), State> _transitions = new()
    {
        // Client-initiated transitions
        { (State.Connected, Constants.Player), State.AwaitingPlayer },
        { (State.AwaitingPlayer, Constants.Protocol), State.AwaitingProtocol },
        { (State.AwaitingProtocol, Constants.Init), State.AwaitingInit },
        // Server responses complete handshake
        { (State.AwaitingInit, "init_complete"), State.Ready },
    };

    private readonly HashSet<string> _validInReady = new()
    {
        Constants.PlayerPlay, Constants.PlayerPause, Constants.PlayerStop,
        Constants.PlayerNext, Constants.PlayerPrevious,
        Constants.PlayerVolume, Constants.PlayerMute,
        Constants.PlayerShuffle, Constants.PlayerRepeat,
        Constants.NowPlayingTrack, Constants.NowPlayingCover,
        Constants.NowPlayingPosition, Constants.NowPlayingRating,
        Constants.NowPlayingList, Constants.NowPlayingLyrics,
        Constants.BrowseGenres, Constants.BrowseArtists,
        Constants.BrowseAlbums, Constants.BrowseTracks,
        Constants.Ping, Constants.Pong,
        // ... all valid commands in Ready state
    };

    public TransitionResult ProcessMessage(SocketMessage message, Direction direction)
    {
        var key = (_currentState, message.Context);

        if (_transitions.TryGetValue(key, out var nextState))
        {
            var previous = _currentState;
            _currentState = nextState;
            return TransitionResult.Valid(previous, nextState, message.Context);
        }

        if (_currentState == State.Ready && _validInReady.Contains(message.Context))
        {
            return TransitionResult.ValidNoChange(_currentState, message.Context);
        }

        return TransitionResult.Invalid(_currentState, message.Context,
            $"Unexpected message '{message.Context}' in state {_currentState}");
    }

    public void SetProtocolVersion(double version) => _negotiatedProtocol = version;
    public double NegotiatedProtocol => _negotiatedProtocol;
    public State CurrentState => _currentState;
    public bool IsReady => _currentState == State.Ready;
}
```

### 3.4 Test Case Definition (YAML)

```yaml
# tests/suites/player-control.yaml
name: Player Control Tests
description: Verify playback control commands
protocol_versions: [4.0, 4.5]

tests:
  - id: PLY-001
    name: Play Command
    description: Verify playerplay starts playback
    prerequisites:
      - state: Ready
      - player: Stopped
      - queue: HasTracks
    steps:
      - send:
          context: playerplay
          data: null
      - expect:
          context: playerstate
          timeout: 2000
          assertions:
            - path: $.data.playerstate
              equals: Playing
    cleanup:
      - send:
          context: playerstop
          data: null

  - id: PLY-002
    name: Pause Command
    description: Verify playerpause pauses playback
    prerequisites:
      - state: Ready
      - player: Playing
    steps:
      - send:
          context: playerpause
          data: null
      - expect:
          context: playerstate
          timeout: 2000
          assertions:
            - path: $.data.playerstate
              equals: Paused

  - id: PLY-007
    name: Volume Set Absolute
    description: Verify volume can be set to specific value
    steps:
      - send:
          context: playervolume
          data: "50"
      - expect:
          context: playervolume
          timeout: 2000
          assertions:
            - path: $.data
              equals: "50"
      - send:
          context: playervolume
          data: "75"
      - expect:
          context: playervolume
          timeout: 2000
          assertions:
            - path: $.data
              equals: "75"

  - id: PLY-008
    name: Volume Relative Increase
    description: Verify volume can be increased relatively
    setup:
      - send:
          context: playervolume
          data: "50"
    steps:
      - send:
          context: playervolume
          data: "+10"
      - expect:
          context: playervolume
          timeout: 2000
          assertions:
            - path: $.data
              equals: "60"

  - id: PLY-010
    name: Volume Boundary Min
    description: Verify volume cannot go below 0
    setup:
      - send:
          context: playervolume
          data: "5"
    steps:
      - send:
          context: playervolume
          data: "-10"
      - expect:
          context: playervolume
          timeout: 2000
          assertions:
            - path: $.data
              equals: "0"

  - id: PLY-011
    name: Volume Boundary Max
    description: Verify volume cannot exceed 100
    setup:
      - send:
          context: playervolume
          data: "95"
    steps:
      - send:
          context: playervolume
          data: "+10"
      - expect:
          context: playervolume
          timeout: 2000
          assertions:
            - path: $.data
              equals: "100"
```

### 3.5 Test Runner

```csharp
public class TestRunner
{
    private readonly MbrcSocketClient _client;
    private readonly SchemaValidator _schemaValidator;
    private readonly ConnectionStateMachine _stateMachine;
    private readonly ILogger _logger;

    public async Task<TestResult> RunTestAsync(TestCase test, CancellationToken ct = default)
    {
        var result = new TestResult(test);
        var stopwatch = Stopwatch.StartNew();

        try
        {
            // Check prerequisites
            foreach (var prereq in test.Prerequisites)
            {
                if (!await CheckPrerequisiteAsync(prereq, ct))
                {
                    result.Status = TestStatus.Skipped;
                    result.Message = $"Prerequisite not met: {prereq}";
                    return result;
                }
            }

            // Run setup steps
            foreach (var setup in test.Setup)
            {
                await ExecuteStepAsync(setup, result, ct);
            }

            // Run test steps
            foreach (var step in test.Steps)
            {
                var stepResult = await ExecuteStepAsync(step, result, ct);
                if (!stepResult.Success)
                {
                    result.Status = TestStatus.Failed;
                    result.FailedStep = step;
                    result.Message = stepResult.Message;
                    break;
                }
            }

            if (result.Status != TestStatus.Failed)
            {
                result.Status = TestStatus.Passed;
            }

            // Run cleanup steps (even on failure)
            foreach (var cleanup in test.Cleanup)
            {
                try
                {
                    await ExecuteStepAsync(cleanup, result, ct);
                }
                catch { /* Ignore cleanup errors */ }
            }
        }
        catch (Exception ex)
        {
            result.Status = TestStatus.Error;
            result.Message = ex.Message;
            result.Exception = ex;
        }

        result.Duration = stopwatch.Elapsed;
        return result;
    }

    private async Task<StepResult> ExecuteStepAsync(TestStep step, TestResult result, CancellationToken ct)
    {
        if (step.Send != null)
        {
            // Validate outgoing message
            var validation = _schemaValidator.Validate(step.Send);
            if (!validation.IsValid)
                return StepResult.Failed($"Invalid message format: {string.Join(", ", validation.Errors)}");

            await _client.SendAsync(step.Send, ct);
            result.MessagesSent.Add(step.Send);
        }

        if (step.Expect != null)
        {
            var timeout = TimeSpan.FromMilliseconds(step.Expect.Timeout);
            var response = await _client.ReceiveAsync(timeout, ct);

            if (response == null)
                return StepResult.Failed($"Timeout waiting for {step.Expect.Context}");

            result.MessagesReceived.Add(response);

            // Validate context
            if (response.Context != step.Expect.Context)
                return StepResult.Failed($"Expected context '{step.Expect.Context}', got '{response.Context}'");

            // Validate schema
            var schemaValidation = _schemaValidator.Validate(response);
            if (!schemaValidation.IsValid)
                return StepResult.Failed($"Response schema invalid: {string.Join(", ", schemaValidation.Errors)}");

            // Run assertions
            foreach (var assertion in step.Expect.Assertions)
            {
                var assertResult = EvaluateAssertion(response, assertion);
                if (!assertResult.Success)
                    return assertResult;
            }
        }

        if (step.Delay.HasValue)
        {
            await Task.Delay(step.Delay.Value, ct);
        }

        return StepResult.Success();
    }

    private StepResult EvaluateAssertion(SocketMessage message, Assertion assertion)
    {
        var json = JObject.FromObject(message);
        var token = json.SelectToken(assertion.Path);

        if (token == null)
            return StepResult.Failed($"Path '{assertion.Path}' not found in message");

        return assertion.Type switch
        {
            AssertionType.Equals =>
                token.ToString() == assertion.Value?.ToString()
                    ? StepResult.Success()
                    : StepResult.Failed($"Expected {assertion.Path}='{assertion.Value}', got '{token}'"),

            AssertionType.NotNull =>
                token.Type != JTokenType.Null
                    ? StepResult.Success()
                    : StepResult.Failed($"Expected {assertion.Path} to be non-null"),

            AssertionType.Contains =>
                token.ToString().Contains(assertion.Value?.ToString() ?? "")
                    ? StepResult.Success()
                    : StepResult.Failed($"Expected {assertion.Path} to contain '{assertion.Value}'"),

            AssertionType.GreaterThan =>
                Convert.ToDouble(token) > Convert.ToDouble(assertion.Value)
                    ? StepResult.Success()
                    : StepResult.Failed($"Expected {assertion.Path} > {assertion.Value}, got {token}"),

            _ => StepResult.Failed($"Unknown assertion type: {assertion.Type}")
        };
    }
}
```

---

## 4. Mock Server Implementation

### 4.1 Mock MBRC Server

```csharp
public class MockMbrcServer : IAsyncDisposable
{
    private readonly TcpListener _listener;
    private readonly MockPlayerState _playerState;
    private readonly MockLibrary _library;
    private readonly ILogger _logger;
    private readonly ConcurrentDictionary<string, MockClientConnection> _clients = new();

    private CancellationTokenSource _cts;

    public MockMbrcServer(int port, MockLibrary library)
    {
        _listener = new TcpListener(IPAddress.Any, port);
        _library = library;
        _playerState = new MockPlayerState();
    }

    public async Task StartAsync()
    {
        _cts = new CancellationTokenSource();
        _listener.Start();
        _logger.LogInformation("Mock server listening on port {Port}", ((IPEndPoint)_listener.LocalEndpoint).Port);

        _ = AcceptClientsAsync(_cts.Token);
    }

    private async Task AcceptClientsAsync(CancellationToken ct)
    {
        while (!ct.IsCancellationRequested)
        {
            try
            {
                var tcpClient = await _listener.AcceptTcpClientAsync(ct);
                var clientId = Guid.NewGuid().ToString();
                var connection = new MockClientConnection(clientId, tcpClient, this);
                _clients[clientId] = connection;
                _ = connection.ProcessAsync(ct);
            }
            catch (OperationCanceledException) { break; }
        }
    }

    public SocketMessage HandleMessage(SocketMessage request, MockClientConnection client)
    {
        return request.Context switch
        {
            Constants.Player => HandlePlayer(request, client),
            Constants.Protocol => HandleProtocol(request, client),
            Constants.Init => HandleInit(request, client),
            Constants.Ping => new SocketMessage(Constants.Pong, null),
            Constants.PlayerPlay => HandlePlay(),
            Constants.PlayerPause => HandlePause(),
            Constants.PlayerStop => HandleStop(),
            Constants.PlayerVolume => HandleVolume(request),
            Constants.NowPlayingTrack => HandleNowPlayingTrack(),
            Constants.BrowseArtists => HandleBrowseArtists(request),
            Constants.BrowseTracks => HandleBrowseTracks(request),
            // ... handle all commands
            _ => null // Unknown command, no response
        };
    }

    private SocketMessage HandlePlayer(SocketMessage request, MockClientConnection client)
    {
        client.ClientId = request.Data?.ToString() ?? "unknown";
        return new SocketMessage(Constants.Player, client.ClientId);
    }

    private SocketMessage HandleProtocol(SocketMessage request, MockClientConnection client)
    {
        var data = request.Data as JObject;
        var requestedVersion = data?["protocol_version"]?.Value<double>() ?? 4.0;

        // Negotiate version
        client.ProtocolVersion = requestedVersion >= 4.5 ? 4.5 : 4.0;
        return new SocketMessage(Constants.Protocol, client.ProtocolVersion);
    }

    private SocketMessage HandleInit(SocketMessage request, MockClientConnection client)
    {
        // Queue the 6 init messages
        client.QueueMessage(new SocketMessage(Constants.NowPlayingTrack, _playerState.CurrentTrack));
        client.QueueMessage(new SocketMessage(Constants.NowPlayingRating, _playerState.CurrentRating));
        client.QueueMessage(new SocketMessage(Constants.NowPlayingLoveStatus, _playerState.CurrentLoveStatus));
        client.QueueMessage(new SocketMessage(Constants.PlayerStatus, _playerState.GetStatus()));
        client.QueueMessage(new SocketMessage(Constants.NowPlayingCover, _playerState.CurrentCover));
        client.QueueMessage(new SocketMessage(Constants.NowPlayingLyrics, _playerState.CurrentLyrics));

        return null; // Init itself has no direct response
    }

    private SocketMessage HandleBrowseTracks(SocketMessage request)
    {
        var data = request.Data as JObject;
        var offset = data?["offset"]?.Value<int>() ?? 0;
        var limit = data?["limit"]?.Value<int>() ?? 100;

        var tracks = _library.GetTracks(offset, limit);
        return new SocketMessage(Constants.BrowseTracks, new
        {
            offset,
            limit,
            total = _library.TotalTracks,
            data = tracks
        });
    }

    // Broadcast to all connected clients
    public async Task BroadcastAsync(SocketMessage message)
    {
        foreach (var client in _clients.Values)
        {
            await client.SendAsync(message);
        }
    }

    public async ValueTask DisposeAsync()
    {
        _cts?.Cancel();
        _listener.Stop();
        foreach (var client in _clients.Values)
        {
            await client.DisposeAsync();
        }
    }
}
```

### 4.2 Mock Library

```csharp
public class MockLibrary
{
    public List<MockTrack> Tracks { get; } = new();
    public List<MockAlbum> Albums { get; } = new();
    public List<MockArtist> Artists { get; } = new();
    public List<MockGenre> Genres { get; } = new();

    public int TotalTracks => Tracks.Count;

    public static MockLibrary LoadFromJson(string path)
    {
        var json = File.ReadAllText(path);
        return JsonConvert.DeserializeObject<MockLibrary>(json);
    }

    public static MockLibrary GenerateRandom(int trackCount)
    {
        var library = new MockLibrary();
        var random = new Random(42); // Fixed seed for reproducibility

        var genres = new[] { "Rock", "Jazz", "Classical", "Electronic", "Pop", "Metal" };
        var artists = Enumerable.Range(1, trackCount / 10)
            .Select(i => new MockArtist { Name = $"Artist {i}" })
            .ToList();

        foreach (var genre in genres)
        {
            library.Genres.Add(new MockGenre { Name = genre, ArtistCount = artists.Count / genres.Length });
        }

        library.Artists.AddRange(artists);

        for (int i = 0; i < trackCount; i++)
        {
            var artist = artists[random.Next(artists.Count)];
            var genre = genres[random.Next(genres.Length)];

            library.Tracks.Add(new MockTrack
            {
                Title = $"Track {i + 1}",
                Artist = artist.Name,
                Album = $"Album {i / 10 + 1}",
                Genre = genre,
                Year = (2000 + random.Next(25)).ToString(),
                Duration = 180000 + random.Next(300000),
                Rating = random.Next(6).ToString(),
                Bitrate = new[] { "128", "192", "256", "320" }[random.Next(4)],
                Format = new[] { "MP3", "FLAC", "AAC", "OGG" }[random.Next(4)],
                Path = $"C:\\Music\\{artist.Name}\\Album {i / 10 + 1}\\Track {i + 1}.mp3",
                TrackNo = (i % 10) + 1,
                Disc = 1,
                PlayCount = random.Next(100),
                SkipCount = random.Next(10),
                Loved = random.Next(10) < 2 ? "L" : ""
            });
        }

        return library;
    }

    public List<object> GetTracks(int offset, int limit)
    {
        return Tracks.Skip(offset).Take(limit).Cast<object>().ToList();
    }

    public List<object> GetArtists(int offset, int limit)
    {
        return Artists.Skip(offset).Take(limit).Cast<object>().ToList();
    }
}
```

---

## 5. CLI Interface Design

### 5.1 Command Structure

```
mbrcpval <command> [options]

Commands:
  server       Validate a server/plugin implementation
  client       Validate a client implementation
  mock         Start a mock server for client testing
  proxy        Run as proxy between client and server
  repl         Interactive protocol explorer
  validate     Validate a single message against schema
  generate     Generate code from schemas

Global Options:
  -v, --verbose    Enable verbose logging
  -q, --quiet      Suppress non-essential output
  --no-color       Disable colored output
  --config <file>  Configuration file path
```

### 5.2 Server Validation Command

```
mbrcpval server [options]

Options:
  -h, --host <host>        Server hostname or IP (default: localhost)
  -p, --port <port>        Server port (default: 3000)
  --protocol <version>     Protocol version to request (4.0 or 4.5)
  -s, --suite <names>      Test suites to run (comma-separated or 'all')
  -t, --tests <ids>        Specific test IDs to run
  --timeout <ms>           Default timeout per test (default: 5000)
  --report <format>        Report format: text, json, html, junit
  -o, --output <path>      Output file path
  --parallel <count>       Run tests in parallel (default: 1)
  --continue-on-failure    Continue running tests after failure
  --aria                   Enable ARiA tests (requires ARiA enabled on server)
  --experimental           Enable experimental feature tests

Examples:
  mbrcpval server -h 192.168.1.10 -p 3000 -s all --report html -o report.html
  mbrcpval server --suite handshake,player --protocol 4.5
  mbrcpval server -t PLY-001,PLY-002,PLY-003 --verbose
```

### 5.3 Mock Server Command

```
mbrcpval mock [options]

Options:
  -p, --port <port>        Listening port (default: 3000)
  --library <file>         Mock library JSON file
  --generate <count>       Generate random library with N tracks
  --behavior <mode>        Server behavior: normal, strict, slow, error
  --delay <ms>             Add delay to all responses
  --log <file>             Log all traffic to file
  --discovery              Enable UDP discovery responder

Examples:
  mbrcpval mock -p 3000 --generate 1000 --log traffic.jsonl
  mbrcpval mock --library test-library.json --behavior strict
```

### 5.4 REPL Command

```
mbrcpval repl [options]

Options:
  -h, --host <host>        Server hostname
  -p, --port <port>        Server port
  --protocol <version>     Protocol version

REPL Commands:
  connect                  Connect and perform handshake
  disconnect               Close connection
  send <context> [data]    Send message
  browse <type>            Browse library (artists, albums, tracks, genres)
  play                     Send playerplay
  pause                    Send playerpause
  volume <value>           Set volume
  record <file>            Start recording session
  stop-record              Stop recording
  replay <file>            Replay recorded session
  help                     Show available commands
  exit                     Exit REPL
```

---

## 6. Fuzzing and Property-Based Testing

### 6.1 FsCheck Integration

```csharp
public class ProtocolFuzzTests
{
    private readonly MockMbrcServer _server;

    [Property(MaxTest = 1000)]
    public Property ServerShouldNotCrashOnAnyMessage(NonNull<string> context, object data)
    {
        return Prop.ForAll(Arb.Default.Int32(), _ =>
        {
            var message = new SocketMessage(context.Get, data);

            try
            {
                var response = _server.HandleMessage(message, new MockClientConnection());
                // Should either return a valid response or null (ignored)
                return response == null || response.Context != null;
            }
            catch (Exception)
            {
                return false; // Server crashed
            }
        });
    }

    [Property]
    public Property VolumeShouldStayInBounds(int initialVolume, int adjustment)
    {
        var clamped = Math.Clamp(initialVolume, 0, 100);
        var adjusted = adjustment >= 0
            ? $"+{Math.Abs(adjustment)}"
            : $"-{Math.Abs(adjustment)}";

        // Set initial
        _server.HandleVolume(new SocketMessage(Constants.PlayerVolume, clamped.ToString()));

        // Apply adjustment
        var response = _server.HandleVolume(new SocketMessage(Constants.PlayerVolume, adjusted));
        var result = int.Parse(response.Data.ToString());

        return (result >= 0 && result <= 100).ToProperty();
    }
}
```

### 6.2 Edge Case Test Generators

```csharp
public static class EdgeCaseGenerators
{
    public static Gen<string> MalformedJson => Gen.OneOf(
        Gen.Constant("{"),
        Gen.Constant("{}{}"),
        Gen.Constant("{\"context\":}"),
        Gen.Constant("{\"context\": \"test\", \"data\":"),
        Gen.Elements("", " ", "\t", "\n", "\r\n"),
        Gen.ArrayOf(Gen.Elements('a', 'z', '0', '9', '{', '}', '"', ':')).Select(cs => new string(cs))
    );

    public static Gen<string> LongStrings => Gen.Choose(1000, 100000)
        .SelectMany(len => Gen.ArrayOf(len, Gen.Elements('a', 'b', 'c')))
        .Select(cs => new string(cs));

    public static Gen<object> WeirdDataPayloads => Gen.OneOf(
        Gen.Constant((object)null),
        Gen.Constant((object)""),
        Gen.Constant((object)new int[0]),
        Gen.Constant((object)new Dictionary<string, object>()),
        LongStrings.Select(s => (object)s),
        Gen.Constant((object)double.NaN),
        Gen.Constant((object)double.PositiveInfinity),
        Gen.Constant((object)double.NegativeInfinity)
    );
}
```

---

## 7. CI/CD Integration

### 7.1 GitHub Actions Workflow

```yaml
name: Protocol Validation

on:
  push:
    paths:
      - 'plugin/**'
      - 'mbrcpval/**'
  pull_request:
    paths:
      - 'plugin/**'
      - 'mbrcpval/**'

jobs:
  schema-validation:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4

      - name: Validate JSON Schemas
        run: |
          npx ajv-cli validate -s schemas/common/message.schema.json \
                               -d tests/fixtures/valid/*.json

  unit-tests:
    runs-on: windows-latest
    steps:
      - uses: actions/checkout@v4

      - name: Setup .NET
        uses: actions/setup-dotnet@v4
        with:
          dotnet-version: '8.0'

      - name: Run Unit Tests
        run: dotnet test mbrcpval.Tests --filter Category=Unit

  integration-tests:
    runs-on: windows-latest
    needs: unit-tests
    steps:
      - uses: actions/checkout@v4

      - name: Build Plugin
        run: dotnet build plugin/mbrc-plugin.csproj -c Release

      - name: Start Mock MusicBee
        run: |
          Start-Process -FilePath "tests/MockMusicBee.exe" -WindowStyle Hidden
          Start-Sleep -Seconds 5
        shell: pwsh

      - name: Run Protocol Tests
        run: |
          dotnet run --project mbrcpval -- server \
            --host localhost \
            --port 3000 \
            --suite all \
            --protocol 4.5 \
            --report junit \
            --output test-results.xml

      - name: Upload Test Results
        uses: actions/upload-artifact@v4
        if: always()
        with:
          name: test-results
          path: test-results.xml

      - name: Publish Test Results
        uses: mikepenz/action-junit-report@v4
        if: always()
        with:
          report_paths: test-results.xml

  fuzz-tests:
    runs-on: windows-latest
    needs: integration-tests
    steps:
      - uses: actions/checkout@v4

      - name: Run Fuzz Tests
        run: dotnet test mbrcpval.Tests --filter Category=Fuzz --timeout 600000
```

---

## 8. Report Templates

### 8.1 HTML Report Template

```html
<!DOCTYPE html>
<html>
<head>
    <title>MBRCPVAL Report - {{timestamp}}</title>
    <style>
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; margin: 40px; }
        .header { border-bottom: 2px solid #333; padding-bottom: 20px; }
        .summary { display: flex; gap: 20px; margin: 20px 0; }
        .stat { background: #f5f5f5; padding: 20px; border-radius: 8px; text-align: center; }
        .stat.passed { background: #d4edda; }
        .stat.failed { background: #f8d7da; }
        .test-list { margin-top: 30px; }
        .test { border: 1px solid #ddd; margin: 10px 0; border-radius: 4px; }
        .test-header { padding: 15px; cursor: pointer; display: flex; justify-content: space-between; }
        .test-header.passed { background: #d4edda; }
        .test-header.failed { background: #f8d7da; }
        .test-header.skipped { background: #fff3cd; }
        .test-details { padding: 15px; border-top: 1px solid #ddd; display: none; }
        .test-details.open { display: block; }
        .message-log { font-family: monospace; font-size: 12px; background: #1e1e1e; color: #d4d4d4; padding: 15px; overflow-x: auto; }
    </style>
</head>
<body>
    <div class="header">
        <h1>MBRCPVAL Protocol Validation Report</h1>
        <p><strong>Target:</strong> {{host}}:{{port}}</p>
        <p><strong>Protocol:</strong> v{{protocolVersion}}</p>
        <p><strong>Plugin Version:</strong> {{pluginVersion}}</p>
        <p><strong>Date:</strong> {{timestamp}}</p>
    </div>

    <div class="summary">
        <div class="stat"><h2>{{totalTests}}</h2><p>Total</p></div>
        <div class="stat passed"><h2>{{passedTests}}</h2><p>Passed</p></div>
        <div class="stat failed"><h2>{{failedTests}}</h2><p>Failed</p></div>
        <div class="stat"><h2>{{skippedTests}}</h2><p>Skipped</p></div>
        <div class="stat"><h2>{{duration}}</h2><p>Duration</p></div>
    </div>

    <div class="test-list">
        <h2>Test Results</h2>
        {{#each tests}}
        <div class="test">
            <div class="test-header {{status}}" onclick="toggleDetails('{{id}}')">
                <span>[{{status}}] {{id}} - {{name}}</span>
                <span>{{duration}}</span>
            </div>
            <div id="details-{{id}}" class="test-details">
                <p><strong>Description:</strong> {{description}}</p>
                {{#if error}}
                <p><strong>Error:</strong> {{error}}</p>
                {{/if}}
                <div class="message-log">
                    {{#each messageLog}}
                    <div>{{direction}}: {{json}}</div>
                    {{/each}}
                </div>
            </div>
        </div>
        {{/each}}
    </div>

    <script>
        function toggleDetails(id) {
            document.getElementById('details-' + id).classList.toggle('open');
        }
    </script>
</body>
</html>
```

---

## 9. Future Considerations

### 9.1 Planned Enhancements

1. **Wireshark Dissector** - Lua plugin for MBRC protocol analysis
2. **Browser-Based GUI** - Electron or web-based interface
3. **Protocol Diff Tool** - Compare behavior between plugin versions
4. **Regression Detection** - Automatic detection of behavior changes
5. **Coverage Metrics** - Track which protocol features are tested

### 9.2 Community Contributions

- Schema contributions for undocumented commands
- Test case contributions for edge cases
- Mock library fixtures for different scenarios
- Language bindings (Python, Go, Rust)

---

## 10. References

1. MBRCPVAL-SPEC.md - Main specification document
2. MBRC-Server-Plugin-Protocol.html - Protocol SDK
3. JSON Schema Draft 2020-12 - https://json-schema.org/specification
4. NJsonSchema - https://github.com/RicoSuter/NJsonSchema
5. FsCheck - https://github.com/fscheck/FsCheck
6. Spectre.Console - https://github.com/spectreconsole/spectre.console

---

*This implementation guide accompanies the MBRCPVAL-SPEC.md specification document.*
