using System.Collections.Generic;
using MusicBeePlugin.AndroidRemote.Settings;
using NUnit.Framework;

namespace MusicBeePlugin.Tests
{
    /// <summary>
    /// Tests for UserSettings defensive code and edge cases.
    /// These BVT tests verify the plugin won't crash with bad/null input data.
    /// </summary>
    [TestFixture]
    public class UserSettingsTests
    {
        [SetUp]
        public void SetUp()
        {
            // Reset settings to known state before each test
            var settings = UserSettings.Instance;
            settings.FilterSelection = FilteringSelection.All;
            settings.BaseIp = null;
            settings.LastOctetMax = 0;
            settings.IpAddressList = new List<string>();
        }

        #region IpAddressList Tests

        [Test]
        public void IpAddressList_DefaultIsNotNull()
        {
            var settings = UserSettings.Instance;
            settings.IpAddressList = null;

            // Getter should return empty list, never null
            Assert.IsNotNull(settings.IpAddressList);
        }

        [Test]
        public void IpAddressList_SetNullReturnsEmptyList()
        {
            var settings = UserSettings.Instance;
            settings.IpAddressList = null;

            Assert.AreEqual(0, settings.IpAddressList.Count);
        }

        [Test]
        public void IpAddressList_SetValidListWorks()
        {
            var settings = UserSettings.Instance;
            var testList = new List<string> { "192.168.1.10", "192.168.1.20" };
            settings.IpAddressList = testList;

            Assert.AreEqual(2, settings.IpAddressList.Count);
            Assert.Contains("192.168.1.10", settings.IpAddressList);
        }

        #endregion

        #region BaseIp and Range Tests

        [Test]
        public void BaseIp_CanBeNull()
        {
            var settings = UserSettings.Instance;
            settings.BaseIp = null;

            // Should not throw
            Assert.IsNull(settings.BaseIp);
        }

        [Test]
        public void BaseIp_CanBeEmpty()
        {
            var settings = UserSettings.Instance;
            settings.BaseIp = string.Empty;

            Assert.AreEqual(string.Empty, settings.BaseIp);
        }

        [Test]
        public void LastOctetMax_ZeroIsValid()
        {
            var settings = UserSettings.Instance;
            settings.LastOctetMax = 0;

            Assert.AreEqual(0u, settings.LastOctetMax);
        }

        [Test]
        public void LastOctetMax_255IsValid()
        {
            var settings = UserSettings.Instance;
            settings.LastOctetMax = 255;

            Assert.AreEqual(255u, settings.LastOctetMax);
        }

        #endregion

        #region FilterSelection Tests

        [Test]
        public void FilterSelection_DefaultIsAll()
        {
            var settings = UserSettings.Instance;
            settings.FilterSelection = FilteringSelection.All;

            Assert.AreEqual(FilteringSelection.All, settings.FilterSelection);
        }

        [Test]
        public void FilterSelection_RangeWithNullBaseIpDoesNotCrash()
        {
            var settings = UserSettings.Instance;
            settings.FilterSelection = FilteringSelection.Range;
            settings.BaseIp = null;
            settings.LastOctetMax = 50;

            // Should not throw when accessed
            Assert.AreEqual(FilteringSelection.Range, settings.FilterSelection);
            Assert.IsNull(settings.BaseIp);
        }

        [Test]
        public void FilterSelection_SpecificWithEmptyListDoesNotCrash()
        {
            var settings = UserSettings.Instance;
            settings.FilterSelection = FilteringSelection.Specific;
            settings.IpAddressList = new List<string>();

            Assert.AreEqual(FilteringSelection.Specific, settings.FilterSelection);
            Assert.AreEqual(0, settings.IpAddressList.Count);
        }

        #endregion

        #region Port Tests

        [Test]
        public void ListeningPort_DefaultIs3000()
        {
            // Port defaults to 3000 when settings file doesn't exist
            var settings = UserSettings.Instance;
            // We can't test the file-based default easily, but we can set and verify
            settings.ListeningPort = 3000;

            Assert.AreEqual(3000u, settings.ListeningPort);
        }

        [Test]
        public void ListeningPort_CanBeSetToHighPort()
        {
            var settings = UserSettings.Instance;
            settings.ListeningPort = 65000;

            Assert.AreEqual(65000u, settings.ListeningPort);
        }

        [Test]
        public void ListeningPort_CanBeSetToLowPort()
        {
            var settings = UserSettings.Instance;
            settings.ListeningPort = 1024;

            Assert.AreEqual(1024u, settings.ListeningPort);
        }

        #endregion

        #region Boolean Settings Tests

        [Test]
        public void DebugLogEnabled_DefaultIsFalse()
        {
            var settings = UserSettings.Instance;
            settings.DebugLogEnabled = false;

            Assert.IsFalse(settings.DebugLogEnabled);
        }

        [Test]
        public void AriaEnabled_DefaultIsFalse()
        {
            var settings = UserSettings.Instance;

            // AriaEnabled defaults to false for security
            settings.AriaEnabled = false;
            Assert.IsFalse(settings.AriaEnabled);
        }

        [Test]
        public void ExperimentalFeaturesEnabled_DefaultIsFalse()
        {
            var settings = UserSettings.Instance;
            settings.ExperimentalFeaturesEnabled = false;

            Assert.IsFalse(settings.ExperimentalFeaturesEnabled);
        }

        [Test]
        public void UpdateFirewall_CanBeToggled()
        {
            var settings = UserSettings.Instance;

            settings.UpdateFirewall = true;
            Assert.IsTrue(settings.UpdateFirewall);

            settings.UpdateFirewall = false;
            Assert.IsFalse(settings.UpdateFirewall);
        }

        #endregion

        #region InstanceId Tests

        [Test]
        public void InstanceId_PropertyAccessDoesNotThrow()
        {
            // InstanceId is generated on first access if null
            // This is set during LoadSettings, we verify the property exists
            var settings = UserSettings.Instance;

            // InstanceId may be null in unit test context (no settings file loaded)
            // but the property should be accessible without throwing
            var instanceId = settings.InstanceId;
            // Just verify it doesn't throw - actual value depends on whether LoadSettings was called
        }

        [Test]
        public void InstanceId_WhenSet_ReturnsValidGuidFormat()
        {
            var settings = UserSettings.Instance;
            var instanceId = settings.InstanceId;

            // If InstanceId is set, it should be a valid GUID format
            if (!string.IsNullOrEmpty(instanceId))
            {
                Assert.IsTrue(System.Guid.TryParse(instanceId, out _),
                    $"InstanceId '{instanceId}' is not a valid GUID format");
            }
        }

        #endregion

        #region Aria Settings Tests

        [Test]
        public void AriaEnabled_CanBeToggled()
        {
            var settings = UserSettings.Instance;

            settings.AriaEnabled = true;
            Assert.IsTrue(settings.AriaEnabled);

            settings.AriaEnabled = false;
            Assert.IsFalse(settings.AriaEnabled);
        }

        #endregion

        #region Experimental Features Tests

        [Test]
        public void ExperimentalFeaturesEnabled_CanBeToggled()
        {
            var settings = UserSettings.Instance;

            settings.ExperimentalFeaturesEnabled = true;
            Assert.IsTrue(settings.ExperimentalFeaturesEnabled);

            settings.ExperimentalFeaturesEnabled = false;
            Assert.IsFalse(settings.ExperimentalFeaturesEnabled);
        }

        #endregion

        #region CurrentVersion Tests

        [Test]
        public void CurrentVersion_CanBeNull()
        {
            var settings = UserSettings.Instance;
            settings.CurrentVersion = null;

            Assert.IsNull(settings.CurrentVersion);
        }

        [Test]
        public void CurrentVersion_CanBeSet()
        {
            var settings = UserSettings.Instance;
            settings.CurrentVersion = "1.5.25.1227";

            Assert.AreEqual("1.5.25.1227", settings.CurrentVersion);
        }

        #endregion
    }
}
