using System;
using MusicBeePlugin.AndroidRemote.Model.Entities;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using NUnit.Framework;

namespace MusicBeePlugin.Tests
{
    /// <summary>
    /// Tests for SocketMessage parsing - the core message format used between
    /// clients (Android app, web clients, etc.) and the MusicBee plugin.
    ///
    /// These tests are critical because:
    /// 1. Older clients may send messages in slightly different formats
    /// 2. JSON parsing edge cases can cause silent failures
    /// 3. BOM characters from different text encodings can corrupt parsing
    /// 4. The protocol must handle various data types (string, object, array, null)
    /// </summary>
    [TestFixture]
    public class MessageParsingTests
    {
        #region Basic Valid Message Parsing

        [Test]
        public void Parse_ValidMessage_ContextAndStringData()
        {
            var json = JObject.Parse("{\"context\":\"player\",\"data\":\"play\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
            Assert.AreEqual("play", msg.Data.ToString());
        }

        [Test]
        public void Parse_ValidMessage_ContextAndObjectData()
        {
            var json = JObject.Parse("{\"context\":\"protocol\",\"data\":{\"protocol_version\":4,\"no_broadcast\":false}}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("protocol", msg.Context);
            Assert.IsInstanceOf<JToken>(msg.Data);

            var data = (JToken)msg.Data;
            Assert.AreEqual(4, (int)data["protocol_version"]);
            Assert.AreEqual(false, (bool)data["no_broadcast"]);
        }

        [Test]
        public void Parse_ValidMessage_ContextAndArrayData()
        {
            var json = JObject.Parse("{\"context\":\"nowplayinglist\",\"data\":[\"track1\",\"track2\",\"track3\"]}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("nowplayinglist", msg.Context);
            Assert.IsInstanceOf<JToken>(msg.Data);
        }

        [Test]
        public void Parse_ValidMessage_ContextAndNumericData()
        {
            var json = JObject.Parse("{\"context\":\"playervolume\",\"data\":75}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("playervolume", msg.Context);
            Assert.AreEqual(75, Convert.ToInt32(msg.Data.ToString()));
        }

        [Test]
        public void Parse_ValidMessage_ContextAndBooleanData()
        {
            var json = JObject.Parse("{\"context\":\"playermute\",\"data\":true}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("playermute", msg.Context);
            Assert.AreEqual("True", msg.Data.ToString());
        }

        #endregion

        #region Null and Empty Data Handling

        [Test]
        public void Parse_NullData_ReturnsEmptyString()
        {
            var json = JObject.Parse("{\"context\":\"player\",\"data\":null}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
            Assert.AreEqual(string.Empty, msg.Data);
        }

        [Test]
        public void Parse_MissingData_ReturnsEmptyString()
        {
            var json = JObject.Parse("{\"context\":\"player\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
            Assert.AreEqual(string.Empty, msg.Data);
        }

        [Test]
        public void Parse_EmptyStringData_ReturnsEmptyString()
        {
            var json = JObject.Parse("{\"context\":\"player\",\"data\":\"\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
            Assert.AreEqual("", msg.Data.ToString());
        }

        #endregion

        #region Missing or Null Context

        [Test]
        public void Parse_MissingContext_ContextIsNull()
        {
            var json = JObject.Parse("{\"data\":\"somedata\"}");
            var msg = new SocketMessage(json);

            Assert.IsNull(msg.Context);
        }

        [Test]
        public void Parse_NullContext_ContextIsNull()
        {
            var json = JObject.Parse("{\"context\":null,\"data\":\"somedata\"}");
            var msg = new SocketMessage(json);

            Assert.IsNull(msg.Context);
        }

        [Test]
        public void Parse_EmptyContext_ContextIsEmpty()
        {
            var json = JObject.Parse("{\"context\":\"\",\"data\":\"somedata\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("", msg.Context);
        }

        #endregion

        #region Null JObject Handling

        [Test]
        public void Parse_NullJObject_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>(() => new SocketMessage((JObject)null));
        }

        #endregion

        #region Malformed JSON Handling

        [Test]
        public void Parse_MalformedJson_ThrowsJsonReaderException()
        {
            Assert.Throws<JsonReaderException>(() => JObject.Parse("{context:player}"));
        }

        [Test]
        public void Parse_IncompleteJson_ThrowsJsonReaderException()
        {
            Assert.Throws<JsonReaderException>(() => JObject.Parse("{\"context\":\"player\""));
        }

        [Test]
        public void Parse_EmptyString_ThrowsJsonReaderException()
        {
            Assert.Throws<JsonReaderException>(() => JObject.Parse(""));
        }

        [Test]
        public void Parse_WhitespaceOnly_ThrowsJsonReaderException()
        {
            Assert.Throws<JsonReaderException>(() => JObject.Parse("   "));
        }

        [Test]
        public void Parse_JsonArray_ThrowsJsonReaderException()
        {
            // A root-level array is not a valid message format
            Assert.Throws<JsonReaderException>(() => JObject.Parse("[{\"context\":\"player\"}]"));
        }

        #endregion

        #region BOM Character Handling

        [Test]
        public void Parse_WithUtf8Bom_ParsesCorrectly()
        {
            // UTF-8 BOM: \uFEFF
            var jsonWithBom = "\uFEFF{\"context\":\"player\",\"data\":\"play\"}";
            var cleanJson = jsonWithBom.TrimStart('\uFEFF');
            var json = JObject.Parse(cleanJson);
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
        }

        [Test]
        public void Parse_WithZeroWidthSpace_ParsesCorrectly()
        {
            // Zero-width space: \u200B
            var jsonWithZws = "\u200B{\"context\":\"player\",\"data\":\"play\"}";
            var cleanJson = jsonWithZws.TrimStart('\u200B');
            var json = JObject.Parse(cleanJson);
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
        }

        [Test]
        public void Parse_WithUtf16BeBom_ParsesCorrectly()
        {
            // UTF-16 BE BOM: \uFFFE
            var jsonWithBom = "\uFFFE{\"context\":\"player\",\"data\":\"play\"}";
            var cleanJson = jsonWithBom.TrimStart('\uFFFE');
            var json = JObject.Parse(cleanJson);
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
        }

        [Test]
        public void Parse_WithMultipleBomChars_ParsesCorrectly()
        {
            // Multiple BOM chars that older clients might send
            var jsonWithBoms = "\uFEFF\u200B\uFFFE{\"context\":\"player\",\"data\":\"play\"}";
            var cleanJson = jsonWithBoms.TrimStart('\uFEFF', '\u200B', '\uFFFE');
            var json = JObject.Parse(cleanJson);
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
        }

        #endregion

        #region Data Type Edge Cases

        [Test]
        public void Parse_DataContainsCurlyBraces_InString()
        {
            // String data that looks like JSON but isn't
            var json = JObject.Parse("{\"context\":\"message\",\"data\":\"Hello {world}\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("message", msg.Context);
            // This is the current behavior - string with braces triggers JToken storage
            Assert.IsNotNull(msg.Data);
        }

        [Test]
        public void Parse_NestedObjectData_PreservesStructure()
        {
            var json = JObject.Parse("{\"context\":\"trackinfo\",\"data\":{\"artist\":\"Test\",\"album\":{\"name\":\"Album\",\"year\":2024}}}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("trackinfo", msg.Context);
            var data = (JToken)msg.Data;
            Assert.AreEqual("Test", (string)data["artist"]);
            Assert.AreEqual("Album", (string)data["album"]["name"]);
        }

        [Test]
        public void Parse_FloatData_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"playerprogress\",\"data\":0.75}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("playerprogress", msg.Context);
            Assert.AreEqual(0.75, Convert.ToDouble(msg.Data.ToString()));
        }

        [Test]
        public void Parse_NegativeNumberData_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"seekoffset\",\"data\":-30}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("seekoffset", msg.Context);
            Assert.AreEqual(-30, Convert.ToInt32(msg.Data.ToString()));
        }

        #endregion

        #region Protocol Version Edge Cases (Older Client Compatibility)

        [Test]
        public void Parse_ProtocolVersionAsInteger_ParsesCorrectly()
        {
            // Some older clients send protocol_version as integer
            var json = JObject.Parse("{\"context\":\"protocol\",\"data\":{\"protocol_version\":4}}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("protocol", msg.Context);
            var data = (JToken)msg.Data;
            Assert.AreEqual(4, (int)data["protocol_version"]);
        }

        [Test]
        public void Parse_ProtocolVersionAsDouble_ParsesCorrectly()
        {
            // Newer clients send protocol_version as double
            var json = JObject.Parse("{\"context\":\"protocol\",\"data\":{\"protocol_version\":4.5}}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("protocol", msg.Context);
            var data = (JToken)msg.Data;
            Assert.AreEqual(4.5, (double)data["protocol_version"]);
        }

        [Test]
        public void Parse_ProtocolWithNoBroadcast_True_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"protocol\",\"data\":{\"protocol_version\":4,\"no_broadcast\":true}}");
            var msg = new SocketMessage(json);

            var data = (JToken)msg.Data;
            Assert.AreEqual(true, (bool)data["no_broadcast"]);
        }

        [Test]
        public void Parse_ProtocolWithNoBroadcast_False_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"protocol\",\"data\":{\"protocol_version\":4,\"no_broadcast\":false}}");
            var msg = new SocketMessage(json);

            var data = (JToken)msg.Data;
            Assert.AreEqual(false, (bool)data["no_broadcast"]);
        }

        [Test]
        public void Parse_ProtocolWithMissingNoBroadcast_DataIsValid()
        {
            // Older clients may not send no_broadcast at all
            var json = JObject.Parse("{\"context\":\"protocol\",\"data\":{\"protocol_version\":4}}");
            var msg = new SocketMessage(json);

            var data = (JToken)msg.Data;
            Assert.IsNull(data["no_broadcast"]);
        }

        #endregion

        #region Multi-Message Parsing Simulation

        [Test]
        public void Parse_MultipleMessages_SplitByCRLF()
        {
            // Simulates how ProtocolHandler splits messages
            var multiMessage = "{\"context\":\"player\",\"data\":\"play\"}\r\n{\"context\":\"playervolume\",\"data\":80}";
            var messages = multiMessage.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

            Assert.AreEqual(2, messages.Length);

            var msg1 = new SocketMessage(JObject.Parse(messages[0]));
            var msg2 = new SocketMessage(JObject.Parse(messages[1]));

            Assert.AreEqual("player", msg1.Context);
            Assert.AreEqual("playervolume", msg2.Context);
        }

        [Test]
        public void Parse_MessageWithTrailingCRLF_ParsesCorrectly()
        {
            var messageWithCrlf = "{\"context\":\"player\",\"data\":\"play\"}\r\n";
            var messages = messageWithCrlf.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

            Assert.AreEqual(1, messages.Length);

            var msg = new SocketMessage(JObject.Parse(messages[0]));
            Assert.AreEqual("player", msg.Context);
        }

        [Test]
        public void Parse_MessageWithLeadingCRLF_ParsesCorrectly()
        {
            var messageWithCrlf = "\r\n{\"context\":\"player\",\"data\":\"play\"}";
            var messages = messageWithCrlf.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

            Assert.AreEqual(1, messages.Length);

            var msg = new SocketMessage(JObject.Parse(messages[0]));
            Assert.AreEqual("player", msg.Context);
        }

        [Test]
        public void Parse_MultipleMessagesWithBom_AllParseCorrectly()
        {
            // Simulates real-world scenario where BOM appears after CRLF
            var multiMessage = "\uFEFF{\"context\":\"player\"}\r\n\uFEFF{\"context\":\"volume\"}";
            var messages = multiMessage.Split(new[] { "\r\n" }, StringSplitOptions.RemoveEmptyEntries);

            Assert.AreEqual(2, messages.Length);

            var msg1 = new SocketMessage(JObject.Parse(messages[0].TrimStart('\uFEFF', '\u200B', '\uFFFE')));
            var msg2 = new SocketMessage(JObject.Parse(messages[1].TrimStart('\uFEFF', '\u200B', '\uFFFE')));

            Assert.AreEqual("player", msg1.Context);
            Assert.AreEqual("volume", msg2.Context);
        }

        #endregion

        #region Serialization Round-Trip Tests

        [Test]
        public void Serialize_SimpleMessage_ProducesValidJson()
        {
            var msg = new SocketMessage("player", "play");
            var json = msg.ToJsonString();

            Assert.IsTrue(json.Contains("\"context\":\"player\""));
            Assert.IsTrue(json.Contains("\"data\":\"play\""));

            // Verify it can be parsed back
            var parsed = new SocketMessage(JObject.Parse(json));
            Assert.AreEqual("player", parsed.Context);
        }

        [Test]
        public void Serialize_MessageWithObjectData_ProducesValidJson()
        {
            var data = new { volume = 75, muted = false };
            var msg = new SocketMessage("playerstate", data);
            var json = msg.ToJsonString();

            // Verify it can be parsed back
            var parsed = JObject.Parse(json);
            Assert.AreEqual("playerstate", (string)parsed["context"]);
            Assert.AreEqual(75, (int)parsed["data"]["volume"]);
        }

        [Test]
        public void ToString_ReturnsJsonString()
        {
            var msg = new SocketMessage("test", "data");
            var str = msg.ToString();
            var json = msg.ToJsonString();

            Assert.AreEqual(json, str);
        }

        #endregion

        #region Known Context Values

        [Test]
        public void Parse_VerifyConnectionContext_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"verifyconnection\",\"data\":\"\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("verifyconnection", msg.Context);
        }

        [Test]
        public void Parse_PlayerContext_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"player\",\"data\":\"MusicBee\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
        }

        [Test]
        public void Parse_NowPlayingTrackContext_WithComplexData()
        {
            var json = JObject.Parse("{\"context\":\"nowplayingtrack\",\"data\":{\"artist\":\"Test Artist\",\"title\":\"Test Song\",\"album\":\"Test Album\"}}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("nowplayingtrack", msg.Context);
            var data = (JToken)msg.Data;
            Assert.AreEqual("Test Artist", (string)data["artist"]);
            Assert.AreEqual("Test Song", (string)data["title"]);
        }

        #endregion

        #region Unicode and Special Characters

        [Test]
        public void Parse_UnicodeInData_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"track\",\"data\":{\"artist\":\"日本語アーティスト\",\"title\":\"Ñoño\"}}");
            var msg = new SocketMessage(json);

            var data = (JToken)msg.Data;
            Assert.AreEqual("日本語アーティスト", (string)data["artist"]);
            Assert.AreEqual("Ñoño", (string)data["title"]);
        }

        [Test]
        public void Parse_EmojiInData_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"message\",\"data\":\"Hello 🎵🎶\"}");
            var msg = new SocketMessage(json);

            Assert.IsTrue(msg.Data.ToString().Contains("🎵"));
        }

        [Test]
        public void Parse_EscapedCharactersInData_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"path\",\"data\":\"C:\\\\Music\\\\Song.mp3\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("C:\\Music\\Song.mp3", msg.Data.ToString());
        }

        [Test]
        public void Parse_NewlinesInStringData_ParsesCorrectly()
        {
            var json = JObject.Parse("{\"context\":\"lyrics\",\"data\":\"Line 1\\nLine 2\\nLine 3\"}");
            var msg = new SocketMessage(json);

            Assert.IsTrue(msg.Data.ToString().Contains("\n"));
        }

        #endregion

        #region Default Constructor Tests

        [Test]
        public void DefaultConstructor_ContextIsNull()
        {
            var msg = new SocketMessage();

            Assert.IsNull(msg.Context);
            Assert.IsNull(msg.Data);
        }

        [Test]
        public void StringConstructor_SetsContextAndData()
        {
            var msg = new SocketMessage("test", "data");

            Assert.AreEqual("test", msg.Context);
            Assert.AreEqual("data", msg.Data);
        }

        [Test]
        public void StringConstructor_NullData_DefaultsToNull()
        {
            var msg = new SocketMessage("test");

            Assert.AreEqual("test", msg.Context);
            Assert.IsNull(msg.Data);
        }

        [Test]
        public void NewLineTerminated_DefaultsFalse()
        {
            var msg = new SocketMessage("test", "data");

            Assert.IsFalse(msg.NewLineTerminated);
        }

        #endregion

        #region Case Sensitivity Tests

        [Test]
        public void Parse_UppercaseContext_PreservesCase()
        {
            var json = JObject.Parse("{\"context\":\"PLAYER\",\"data\":\"play\"}");
            var msg = new SocketMessage(json);

            // Context comparison in ProtocolHandler is case-sensitive
            Assert.AreEqual("PLAYER", msg.Context);
            Assert.AreNotEqual("player", msg.Context);
        }

        [Test]
        public void Parse_MixedCaseKeys_ParsesCorrectly()
        {
            // JSON keys are case-sensitive - "Context" is different from "context"
            var json = JObject.Parse("{\"context\":\"player\",\"data\":\"play\"}");
            var msg = new SocketMessage(json);

            Assert.AreEqual("player", msg.Context);
        }

        #endregion
    }
}
