using System.Collections.Generic;
using System.Net;
using MusicBeePlugin.AndroidRemote.Settings;
using MusicBeePlugin.Tools;
using NUnit.Framework;

namespace MusicBeePlugin.Tests
{
    /// <summary>
    /// Comprehensive tests for IpFilterHelper.
    /// These tests ensure bulletproof IP filtering with all edge cases covered.
    /// </summary>
    [TestFixture]
    public class IpFilterHelperTests
    {
        #region TryParseIpOctets Tests

        [Test]
        public void TryParseIpOctets_ValidIp_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.TryParseIpOctets("192.168.1.100", out var octets));
            Assert.AreEqual(4, octets.Length);
            Assert.AreEqual(192, octets[0]);
            Assert.AreEqual(168, octets[1]);
            Assert.AreEqual(1, octets[2]);
            Assert.AreEqual(100, octets[3]);
        }

        [Test]
        public void TryParseIpOctets_AllZeros_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.TryParseIpOctets("0.0.0.0", out var octets));
            Assert.AreEqual(0, octets[0]);
            Assert.AreEqual(0, octets[3]);
        }

        [Test]
        public void TryParseIpOctets_AllMax_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.TryParseIpOctets("255.255.255.255", out var octets));
            Assert.AreEqual(255, octets[0]);
            Assert.AreEqual(255, octets[3]);
        }

        [Test]
        public void TryParseIpOctets_Null_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets(null, out _));
        }

        [Test]
        public void TryParseIpOctets_Empty_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("", out _));
        }

        [Test]
        public void TryParseIpOctets_Whitespace_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("   ", out _));
        }

        [Test]
        public void TryParseIpOctets_TooFewOctets_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("192.168.1", out _));
        }

        [Test]
        public void TryParseIpOctets_TooManyOctets_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("192.168.1.100.50", out _));
        }

        [Test]
        public void TryParseIpOctets_NegativeOctet_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("192.168.-1.100", out _));
        }

        [Test]
        public void TryParseIpOctets_OctetOver255_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("192.168.1.256", out _));
        }

        [Test]
        public void TryParseIpOctets_NonNumeric_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("192.168.abc.100", out _));
        }

        [Test]
        public void TryParseIpOctets_LeadingDot_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets(".192.168.1.100", out _));
        }

        [Test]
        public void TryParseIpOctets_TrailingDot_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("192.168.1.100.", out _));
        }

        [Test]
        public void TryParseIpOctets_DoubleDot_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.TryParseIpOctets("192..168.1.100", out _));
        }

        #endregion

        #region IsValidIpAddress Tests

        [Test]
        public void IsValidIpAddress_ValidIp_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsValidIpAddress("192.168.1.100"));
        }

        [Test]
        public void IsValidIpAddress_Null_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidIpAddress(null));
        }

        [Test]
        public void IsValidIpAddress_Empty_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidIpAddress(""));
        }

        [Test]
        public void IsValidIpAddress_InvalidFormat_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidIpAddress("not.an.ip"));
        }

        [Test]
        public void IsValidIpAddress_IPv6_ReturnsFalse()
        {
            // We only support IPv4
            Assert.IsFalse(IpFilterHelper.IsValidIpAddress("::1"));
        }

        #endregion

        #region IsInRange Tests - Core Functionality

        [Test]
        public void IsInRange_IpInRange_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsInRange("192.168.10.50", "192.168.10.10", 60));
        }

        [Test]
        public void IsInRange_IpAtRangeStart_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsInRange("192.168.10.10", "192.168.10.10", 50));
        }

        [Test]
        public void IsInRange_IpAtRangeEnd_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsInRange("192.168.10.50", "192.168.10.10", 50));
        }

        [Test]
        public void IsInRange_IpBelowRange_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.5", "192.168.10.10", 50));
        }

        [Test]
        public void IsInRange_IpAboveRange_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.60", "192.168.10.10", 50));
        }

        #endregion

        #region IsInRange Tests - Different Subnet

        [Test]
        public void IsInRange_DifferentFirstOctet_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("10.168.10.50", "192.168.10.10", 60));
        }

        [Test]
        public void IsInRange_DifferentSecondOctet_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.169.10.50", "192.168.10.10", 60));
        }

        [Test]
        public void IsInRange_DifferentThirdOctet_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.20.50", "192.168.10.10", 60));
        }

        #endregion

        #region IsInRange Tests - Invalid Inputs

        [Test]
        public void IsInRange_NullIpString_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange(null, "192.168.10.10", 50));
        }

        [Test]
        public void IsInRange_EmptyIpString_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("", "192.168.10.10", 50));
        }

        [Test]
        public void IsInRange_NullBaseIp_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.50", null, 50));
        }

        [Test]
        public void IsInRange_EmptyBaseIp_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.50", "", 50));
        }

        [Test]
        public void IsInRange_InvalidIpFormat_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("invalid", "192.168.10.10", 50));
        }

        [Test]
        public void IsInRange_InvalidBaseIpFormat_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.50", "invalid", 50));
        }

        #endregion

        #region IsInRange Tests - Boundary LastOctetMax

        [Test]
        public void IsInRange_LastOctetMaxZero_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.50", "192.168.10.10", 0));
        }

        [Test]
        public void IsInRange_LastOctetMax255_ReturnsFalse()
        {
            // 255 is broadcast, not valid for hosts
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.50", "192.168.10.10", 255));
        }

        [Test]
        public void IsInRange_LastOctetMax254_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsInRange("192.168.10.50", "192.168.10.10", 254));
        }

        [Test]
        public void IsInRange_LastOctetMax1_ValidRange_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsInRange("192.168.10.1", "192.168.10.1", 1));
        }

        [Test]
        public void IsInRange_LastOctetMaxOver255_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInRange("192.168.10.50", "192.168.10.10", 500));
        }

        #endregion

        #region IsInSpecificList Tests

        [Test]
        public void IsInSpecificList_IpInList_ReturnsTrue()
        {
            var list = new List<string> { "192.168.1.10", "192.168.1.20", "192.168.1.30" };
            Assert.IsTrue(IpFilterHelper.IsInSpecificList("192.168.1.20", list));
        }

        [Test]
        public void IsInSpecificList_IpNotInList_ReturnsFalse()
        {
            var list = new List<string> { "192.168.1.10", "192.168.1.20" };
            Assert.IsFalse(IpFilterHelper.IsInSpecificList("192.168.1.30", list));
        }

        [Test]
        public void IsInSpecificList_EmptyList_ReturnsFalse()
        {
            var list = new List<string>();
            Assert.IsFalse(IpFilterHelper.IsInSpecificList("192.168.1.10", list));
        }

        [Test]
        public void IsInSpecificList_NullList_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsInSpecificList("192.168.1.10", null));
        }

        [Test]
        public void IsInSpecificList_NullIp_ReturnsFalse()
        {
            var list = new List<string> { "192.168.1.10" };
            Assert.IsFalse(IpFilterHelper.IsInSpecificList(null, list));
        }

        [Test]
        public void IsInSpecificList_EmptyIp_ReturnsFalse()
        {
            var list = new List<string> { "192.168.1.10" };
            Assert.IsFalse(IpFilterHelper.IsInSpecificList("", list));
        }

        [Test]
        public void IsInSpecificList_SingleItemMatch_ReturnsTrue()
        {
            var list = new List<string> { "192.168.1.10" };
            Assert.IsTrue(IpFilterHelper.IsInSpecificList("192.168.1.10", list));
        }

        [Test]
        public void IsInSpecificList_CaseSensitive_ExactMatch()
        {
            // IPs should be compared exactly
            var list = new List<string> { "192.168.1.10" };
            Assert.IsTrue(IpFilterHelper.IsInSpecificList("192.168.1.10", list));
        }

        #endregion

        #region IsValidRange Tests

        [Test]
        public void IsValidRange_ValidRange_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsValidRange("192.168.1.10", 50));
        }

        [Test]
        public void IsValidRange_MaxEqualsBase_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsValidRange("192.168.1.50", 50));
        }

        [Test]
        public void IsValidRange_MaxBelowBase_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidRange("192.168.1.50", 40));
        }

        [Test]
        public void IsValidRange_NullBaseIp_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidRange(null, 50));
        }

        [Test]
        public void IsValidRange_EmptyBaseIp_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidRange("", 50));
        }

        [Test]
        public void IsValidRange_InvalidBaseIp_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidRange("invalid", 50));
        }

        [Test]
        public void IsValidRange_MaxZero_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidRange("192.168.1.0", 0));
        }

        [Test]
        public void IsValidRange_Max255_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsValidRange("192.168.1.10", 255));
        }

        #endregion

        #region IsIpAllowed Tests - FilteringSelection.All

        [Test]
        public void IsIpAllowed_AllMode_AnyIp_ReturnsTrue()
        {
            var ip = IPAddress.Parse("192.168.1.100");
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.All, null, null, 0));
        }

        [Test]
        public void IsIpAllowed_AllMode_IgnoresOtherParams_ReturnsTrue()
        {
            var ip = IPAddress.Parse("10.0.0.1");
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.All,
                new List<string> { "192.168.1.1" }, "192.168.1.1", 50));
        }

        #endregion

        #region IsIpAllowed Tests - FilteringSelection.Specific

        [Test]
        public void IsIpAllowed_SpecificMode_IpInList_ReturnsTrue()
        {
            var ip = IPAddress.Parse("192.168.1.20");
            var list = new List<string> { "192.168.1.10", "192.168.1.20" };
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Specific, list, null, 0));
        }

        [Test]
        public void IsIpAllowed_SpecificMode_IpNotInList_ReturnsFalse()
        {
            var ip = IPAddress.Parse("192.168.1.30");
            var list = new List<string> { "192.168.1.10", "192.168.1.20" };
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Specific, list, null, 0));
        }

        [Test]
        public void IsIpAllowed_SpecificMode_NullList_ReturnsFalse()
        {
            var ip = IPAddress.Parse("192.168.1.10");
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Specific, null, null, 0));
        }

        #endregion

        #region IsIpAllowed Tests - FilteringSelection.Range

        [Test]
        public void IsIpAllowed_RangeMode_IpInRange_ReturnsTrue()
        {
            var ip = IPAddress.Parse("192.168.10.50");
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Range, null, "192.168.10.10", 60));
        }

        [Test]
        public void IsIpAllowed_RangeMode_IpOutOfRange_ReturnsFalse()
        {
            var ip = IPAddress.Parse("192.168.10.100");
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Range, null, "192.168.10.10", 60));
        }

        [Test]
        public void IsIpAllowed_RangeMode_NullBaseIp_ReturnsFalse()
        {
            var ip = IPAddress.Parse("192.168.10.50");
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Range, null, null, 60));
        }

        #endregion

        #region IsIpAllowed Tests - Loopback Always Allowed

        [Test]
        public void IsIpAllowed_Loopback_AllMode_ReturnsTrue()
        {
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(IPAddress.Loopback, FilteringSelection.All, null, null, 0));
        }

        [Test]
        public void IsIpAllowed_Loopback_SpecificMode_EmptyList_ReturnsTrue()
        {
            // Loopback should be allowed even with empty specific list
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(IPAddress.Loopback, FilteringSelection.Specific,
                new List<string>(), null, 0));
        }

        [Test]
        public void IsIpAllowed_Loopback_RangeMode_InvalidRange_ReturnsTrue()
        {
            // Loopback should be allowed even with invalid range
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(IPAddress.Loopback, FilteringSelection.Range,
                null, null, 0));
        }

        [Test]
        public void IsIpAllowed_127001_SpecificMode_NotInList_ReturnsTrue()
        {
            // 127.0.0.1 is loopback and should always be allowed
            var ip = IPAddress.Parse("127.0.0.1");
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Specific,
                new List<string> { "192.168.1.1" }, null, 0));
        }

        #endregion

        #region IsIpAllowed Tests - Null IP

        [Test]
        public void IsIpAllowed_NullIp_AllMode_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(null, FilteringSelection.All, null, null, 0));
        }

        [Test]
        public void IsIpAllowed_NullIp_SpecificMode_ReturnsFalse()
        {
            var list = new List<string> { "192.168.1.10" };
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(null, FilteringSelection.Specific, list, null, 0));
        }

        [Test]
        public void IsIpAllowed_NullIp_RangeMode_ReturnsFalse()
        {
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(null, FilteringSelection.Range, null, "192.168.1.1", 50));
        }

        #endregion

        #region Real World Scenarios

        [Test]
        public void RealWorld_HomeNetwork_TypeicalRange()
        {
            // Typical home network: 192.168.1.1 - 192.168.1.254
            var ip = IPAddress.Parse("192.168.1.150");
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Range, null, "192.168.1.1", 254));
        }

        [Test]
        public void RealWorld_OfficeSubnet_RestrictedRange()
        {
            // Office restricts to workstations 10.0.1.100 - 10.0.1.150
            var allowed = IPAddress.Parse("10.0.1.120");
            var denied = IPAddress.Parse("10.0.1.200");

            Assert.IsTrue(IpFilterHelper.IsIpAllowed(allowed, FilteringSelection.Range, null, "10.0.1.100", 150));
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(denied, FilteringSelection.Range, null, "10.0.1.100", 150));
        }

        [Test]
        public void RealWorld_WhitelistedDevices()
        {
            // Only specific devices allowed
            var whitelist = new List<string>
            {
                "192.168.1.50",  // Phone
                "192.168.1.51",  // Tablet
                "192.168.1.52"   // Laptop
            };

            var phone = IPAddress.Parse("192.168.1.50");
            var unknown = IPAddress.Parse("192.168.1.99");

            Assert.IsTrue(IpFilterHelper.IsIpAllowed(phone, FilteringSelection.Specific, whitelist, null, 0));
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(unknown, FilteringSelection.Specific, whitelist, null, 0));
        }

        [Test]
        public void RealWorld_CorruptedSettings_Graceful()
        {
            // Corrupted/missing settings should fail safely
            var ip = IPAddress.Parse("192.168.1.100");

            // Empty everything - Range mode should deny
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Range, null, "", 0));

            // Empty everything - Specific mode should deny
            Assert.IsFalse(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.Specific, null, null, 0));

            // All mode still allows
            Assert.IsTrue(IpFilterHelper.IsIpAllowed(ip, FilteringSelection.All, null, null, 0));
        }

        #endregion
    }
}
