@echo Off
setlocal

set config=%1
if "%config%" == "" (
   set config=Release
)

set version=
if not "%PackageVersion%" == "" (
   set version=-Version %PackageVersion%
)

REM Find MSBuild using vswhere
for /f "usebackq tokens=*" %%i in (`"%ProgramFiles(x86)%\Microsoft Visual Studio\Installer\vswhere.exe" -latest -requires Microsoft.Component.MSBuild -find MSBuild\**\Bin\MSBuild.exe`) do set MSBUILD=%%i

if "%MSBUILD%" == "" (
    echo ERROR: MSBuild not found. Please install Visual Studio with C++ build tools.
    pause
    exit /b 1
)

echo Using MSBuild: %MSBUILD%
echo Configuration: %config%
echo.

REM Remove Previous output
if exist "%cd%\build" rmdir "%cd%\build" /s /q

REM Ensure deploy folder exists
if not exist "%cd%\deploy" mkdir "%cd%\deploy"

REM Generate build epoch and update AssemblyInfo.cs
for /f %%i in ('powershell -NoProfile -Command "$epoch = [int][double]::Parse((Get-Date -UFormat '%%s')); $epoch.ToString('X8')"') do set BUILD_EPOCH=%%i
echo Build Epoch: %BUILD_EPOCH%
powershell -NoProfile -ExecutionPolicy Bypass -File update-epoch.ps1 -Epoch %BUILD_EPOCH%

REM Package restore (MSBuild will restore packages automatically)
"%MSBUILD%" MBRC.sln /t:Restore /p:Configuration="%config%" /v:M

REM Build solution (AfterBuild targets copy to deploy folder)
"%MSBUILD%" MBRC.sln /p:Configuration="%config%";Platform="Any CPU" /v:M /fl /flp:LogFile=msbuild.log;Verbosity=Normal /nr:false

if %ERRORLEVEL% neq 0 (
    echo.
    echo BUILD FAILED
    pause
    exit /b 1
)

REM Build mbrcpval as single-file exe
echo.
echo ============================================
echo Building mbrcpval validator...
echo ============================================
dotnet publish "%cd%\mbrcpval\mbrcpval.csproj" -c Release -r win-x64 --self-contained true -p:PublishSingleFile=true -p:IncludeNativeLibrariesForSelfExtract=true -o "%cd%\mbrcpval\bin\publish" --nologo -v q

if %ERRORLEVEL% neq 0 (
    echo WARNING: mbrcpval build failed, continuing...
) else (
    echo mbrcpval.exe published to mbrcpval\bin\publish
)

REM Verify deploy folder contents
echo.
echo ============================================
echo Deploy folder contents:
echo ============================================
dir /b "%cd%\deploy"
echo.

echo Build complete!

REM Run BVT tests unless /notest is specified
if "%2" == "/notest" goto :skip_tests
if "%1" == "/notest" goto :skip_tests

echo.
echo ============================================
echo Running BVT Tests...
echo ============================================

REM Run tests and save results to log file
set "TEST_LOG=%cd%\deploy\test-results.log"

echo Test Run: %date% %time% > "%TEST_LOG%"
echo Configuration: %config% >> "%TEST_LOG%"
echo ---------------------------------------- >> "%TEST_LOG%"

REM Build and run tests (remove --no-restore to ensure test project builds)
dotnet test "%cd%\mbrc-plugin.Tests\mbrc-plugin.Tests.csproj" --configuration %config% --verbosity normal --logger "console;verbosity=detailed" --results-directory "%cd%\deploy" >> "%TEST_LOG%" 2>&1

set TEST_RESULT=%ERRORLEVEL%

echo ---------------------------------------- >> "%TEST_LOG%"

REM Extract and display summary
echo. >> "%TEST_LOG%"
echo ============ TEST SUMMARY ============ >> "%TEST_LOG%"
findstr /C:"Passed!" /C:"Failed!" /C:"Total tests:" /C:"Passed:" /C:"Failed:" /C:"Skipped:" "%TEST_LOG%" > "%TEMP%\test-summary.tmp" 2>nul
if exist "%TEMP%\test-summary.tmp" (
    type "%TEMP%\test-summary.tmp" >> "%TEST_LOG%"
    del "%TEMP%\test-summary.tmp" 2>nul
)
echo Test run completed with exit code: %TEST_RESULT% >> "%TEST_LOG%"
echo ======================================= >> "%TEST_LOG%"

if %TEST_RESULT% neq 0 (
    echo TESTS FAILED
    echo.
    echo ============ TEST SUMMARY ============
    findstr /C:"Passed!" /C:"Failed!" /C:"Total tests:" /C:"Passed:" /C:"Failed:" /C:"Skipped:" "%TEST_LOG%"
    echo Test run completed with exit code: %TEST_RESULT%
    echo =======================================
    pause
    exit /b 1
)

echo.
echo ============ TEST SUMMARY ============
findstr /C:"Passed!" /C:"Failed!" /C:"Total tests:" /C:"Passed:" /C:"Failed:" /C:"Skipped:" "%TEST_LOG%"
echo Test run completed with exit code: %TEST_RESULT%
echo =======================================
echo.
echo Full results: %TEST_LOG%

:skip_tests

REM Package binaries and source
echo.
echo ============================================
echo Creating packages...
echo ============================================
powershell -NoProfile -ExecutionPolicy Bypass -File package.ps1
powershell -NoProfile -ExecutionPolicy Bypass -File package-source.ps1

echo.
echo ============================================
echo BUILD AND PACKAGE COMPLETE
echo ============================================
