# Protocol v4.5 - Library Event Subscriptions

## Overview

Protocol v4.5 introduces **Library Event Subscriptions**, enabling clients to receive real-time notifications when the MusicBee library changes. This is a new server-push capability that complements the existing request-response pattern.

## MusicBee Plugin API Events

MusicBee provides event callbacks via the `ReceiveNotification` method. The plugin subscribes to these events and forwards them to connected clients that have opted in.

### Available Events

| NotificationType | Fired When | Data Available |
|------------------|------------|----------------|
| `TagsChanging` | Before tags are modified | File path |
| `TagsChanged` | After tags are modified | File path |
| `FileAddedToLibrary` | New file added | File path, metadata |
| `FileDeleted` | File removed | File path |
| `PlayCountersChanged` | Play count updated | File path |
| `RatingChanged` | Rating modified | File path, new rating |

### Plugin Implementation

```csharp
public void ReceiveNotification(string sourceFileUrl, NotificationType type)
{
    switch (type)
    {
        case NotificationType.TagsChanged:
            BroadcastLibraryEvent("librarytagchanged", sourceFileUrl);
            break;

        case NotificationType.FileAddedToLibrary:
            BroadcastLibraryEvent("libraryfileadded", sourceFileUrl);
            break;

        case NotificationType.FileDeleted:
            BroadcastLibraryEvent("libraryfiledeleted", sourceFileUrl);
            break;

        case NotificationType.PlayCountersChanged:
            BroadcastLibraryEvent("libraryplaycountchanged", sourceFileUrl);
            break;

        case NotificationType.RatingChanged:
            BroadcastLibraryEvent("libraryratingchanged", sourceFileUrl);
            break;
    }
}
```

## Protocol Messages

### Client Subscribes (Client → Server)

```json
{
  "context": "librarysubscribe",
  "data": {
    "events": ["tagchanged", "fileadded", "filedeleted", "playcountchanged", "ratingchanged"],
    "include_metadata": true
  }
}
```

| Field | Type | Description |
|-------|------|-------------|
| `events` | string[] | Event types to subscribe to |
| `include_metadata` | bool | Include full track metadata in notifications |

### Server Confirms (Server → Client)

```json
{
  "context": "librarysubscribe",
  "data": {
    "success": true,
    "subscribed": ["tagchanged", "fileadded", "filedeleted", "playcountchanged", "ratingchanged"]
  }
}
```

### Client Unsubscribes (Client → Server)

```json
{
  "context": "libraryunsubscribe",
  "data": {}
}
```

## Event Notifications (Server → Client)

### Tag Changed

```json
{
  "context": "librarytagchanged",
  "data": {
    "path": "C:\\Music\\Artist\\Album\\Track.mp3",
    "timestamp": "2025-12-25T10:30:00Z",
    "changes": {
      "artist": "New Artist",
      "title": "New Title",
      "album": "New Album",
      "genre": "Rock"
    }
  }
}
```

### File Added

```json
{
  "context": "libraryfileadded",
  "data": {
    "path": "C:\\Music\\Artist\\Album\\NewTrack.mp3",
    "timestamp": "2025-12-25T10:30:00Z",
    "metadata": {
      "artist": "Artist Name",
      "album": "Album Title",
      "title": "Track Title",
      "year": "2025",
      "genre": "Electronic",
      "duration": 234000,
      "rating": "0"
    }
  }
}
```

### File Deleted

```json
{
  "context": "libraryfiledeleted",
  "data": {
    "path": "C:\\Music\\Artist\\Album\\DeletedTrack.mp3",
    "timestamp": "2025-12-25T10:30:00Z"
  }
}
```

### Play Count Changed

```json
{
  "context": "libraryplaycountchanged",
  "data": {
    "path": "C:\\Music\\Artist\\Album\\Track.mp3",
    "timestamp": "2025-12-25T10:30:00Z",
    "playcount": 15,
    "lastplayed": "2025-12-25T10:30:00Z"
  }
}
```

### Rating Changed

```json
{
  "context": "libraryratingchanged",
  "data": {
    "path": "C:\\Music\\Artist\\Album\\Track.mp3",
    "timestamp": "2025-12-25T10:30:00Z",
    "rating": "4"
  }
}
```

## Implementation Requirements

### Plugin Side

1. **Subscription State**: Track which clients have subscribed to which events
2. **Event Handler**: Subscribe to MusicBee's `ReceiveNotification` for library events
3. **Metadata Retrieval**: When `include_metadata: true`, fetch full track info
4. **Broadcast Logic**: Only send to clients that subscribed to that event type
5. **Cleanup**: Remove subscription state when client disconnects

### Client Connection Tracking

```csharp
public class ClientSubscription
{
    public string ClientId { get; set; }
    public HashSet<string> SubscribedEvents { get; set; }
    public bool IncludeMetadata { get; set; }
}
```

### Batching Consideration

For bulk operations (e.g., user edits 100 tracks), consider batching:

```json
{
  "context": "librarytagchanged",
  "data": {
    "batch": true,
    "count": 100,
    "items": [
      { "path": "...", "changes": {...} },
      { "path": "...", "changes": {...} }
    ]
  }
}
```

## Backwards Compatibility

- Clients that don't send `librarysubscribe` receive no library events (opt-in)
- Existing v4.0 clients continue to work unchanged
- Protocol negotiation already handles version differences

## Constants

Add to `Constants.cs`:

```csharp
public const string LibrarySubscribe = "librarysubscribe";
public const string LibraryUnsubscribe = "libraryunsubscribe";
public const string LibraryTagChanged = "librarytagchanged";
public const string LibraryFileAdded = "libraryfileadded";
public const string LibraryFileDeleted = "libraryfiledeleted";
public const string LibraryPlayCountChanged = "libraryplaycountchanged";
public const string LibraryRatingChanged = "libraryratingchanged";
```
