# Compare-Codebases.ps1
# Compares mbrc-plugin-main-server (current v4.5) with mbrc-plugin-master (reference)
# Run from: mbrc-plugin-main-server\plugin\Documentation\
# Usage: .\Compare-Codebases.ps1
#        .\Compare-Codebases.ps1 -Detailed
#        .\Compare-Codebases.ps1 -NoFile (console only, no file output)

param(
    [string]$CurrentPath = "C:\Users\scott\source\repos\MBX\ReferenceCode\mbrc-plugin-main-server\plugin",
    [string]$ReferencePath = "C:\Users\scott\source\repos\MBX\ReferenceCode\mbrc-plugin-master\plugin",
    [switch]$Detailed,
    [switch]$NoFile
)

# Generate dated output filename
$dateStamp = Get-Date -Format "yyyy-MM-dd_HHmmss"
$scriptDir = Split-Path -Parent $MyInvocation.MyCommand.Path
$outputFile = Join-Path $scriptDir "comparison_$dateStamp.txt"

# Collect output for file
$fileOutput = New-Object System.Collections.ArrayList

# Helper function to write to both console (with color) and file (plain)
function Write-Both {
    param(
        [string]$Message,
        [string]$Color = "White",
        [switch]$NoNewline
    )

    if ($NoNewline) {
        Write-Host $Message -ForegroundColor $Color -NoNewline
    } else {
        Write-Host $Message -ForegroundColor $Color
    }

    # Add to file output (strip color)
    if (-not $NoNewline) {
        [void]$fileOutput.Add($Message)
    }
}

Write-Both ""
Write-Both ("=" * 60) -Color Cyan
Write-Both "  MBRC PLUGIN CODEBASE COMPARISON" -Color Cyan
Write-Both "  Generated: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')" -Color Gray
Write-Both ("=" * 60) -Color Cyan

# Validate paths
if (-not (Test-Path $CurrentPath)) {
    Write-Both "ERROR: Current path not found: $CurrentPath" -Color Red
    exit 1
}
if (-not (Test-Path $ReferencePath)) {
    Write-Both "WARNING: Reference path not found: $ReferencePath" -Color Yellow
    Write-Both "  Comparison will only show current codebase stats." -Color Yellow
    $ReferencePath = $null
}

# Get current codebase stats
Write-Both ""
Write-Both "[CURRENT] mbrc-plugin-main-server/plugin (v4.5)" -Color Green
Write-Both "  Path: $CurrentPath" -Color Gray

$currentFiles = Get-ChildItem -Path $CurrentPath -Filter "*.cs" -Recurse -ErrorAction SilentlyContinue
$currentStats = $currentFiles | Measure-Object -Property Length -Sum

Write-Both "  Total .cs files: $($currentStats.Count)"
Write-Both "  Total size: $([math]::Round($currentStats.Sum / 1KB, 1)) KB"

# Get reference codebase stats if available
if ($ReferencePath) {
    Write-Both ""
    Write-Both "[REFERENCE] mbrc-plugin-master/plugin (Original)" -Color Yellow
    Write-Both "  Path: $ReferencePath" -Color Gray

    $refFiles = Get-ChildItem -Path $ReferencePath -Filter "*.cs" -Recurse -ErrorAction SilentlyContinue
    $refStats = $refFiles | Measure-Object -Property Length -Sum

    Write-Both "  Total .cs files: $($refStats.Count)"
    Write-Both "  Total size: $([math]::Round($refStats.Sum / 1KB, 1)) KB"

    # Delta
    Write-Both ""
    Write-Both "[DELTA]" -Color Magenta
    $fileDelta = $currentStats.Count - $refStats.Count
    $sizeDelta = $currentStats.Sum - $refStats.Sum
    Write-Both "  File count: $(if($fileDelta -ge 0){'+'}else{''})$fileDelta"
    Write-Both "  Size: $(if($sizeDelta -ge 0){'+'}else{''})$([math]::Round($sizeDelta / 1KB, 1)) KB"
}

# Plugin.cs comparison
Write-Both ""
Write-Both "[PLUGIN.CS ANALYSIS]" -Color Cyan
$pluginFile = Get-Item -Path "$CurrentPath\Plugin.cs" -ErrorAction SilentlyContinue
if ($pluginFile) {
    $pluginLines = (Get-Content $pluginFile.FullName).Count
    Write-Both "  Size: $([math]::Round($pluginFile.Length / 1KB, 1)) KB"
    Write-Both "  Lines: $pluginLines"
    Write-Both "  Target: < 20 KB, < 500 lines"

    $sizeStatus = if ($pluginFile.Length -lt 20KB) { "PASS" } else { "NEEDS WORK" }
    $lineStatus = if ($pluginLines -lt 500) { "PASS" } else { "NEEDS WORK" }
    $sizeColor = if ($sizeStatus -eq "PASS") { "Green" } else { "Yellow" }
    $lineColor = if ($lineStatus -eq "PASS") { "Green" } else { "Yellow" }
    Write-Both "  Size status: $sizeStatus" -Color $sizeColor
    Write-Both "  Line status: $lineStatus" -Color $lineColor
}

# Top 15 largest files
Write-Both ""
Write-Both "[TOP 15 LARGEST FILES]" -Color Cyan
$currentFiles | Sort-Object Length -Descending | Select-Object -First 15 | ForEach-Object {
    $size = [math]::Round($_.Length / 1KB, 1)
    $relativePath = $_.FullName.Replace("$CurrentPath\", "")
    $status = if ($_.Length -gt 50KB) { " [LARGE]" } else { "" }
    Write-Both ("  {0,6} KB - {1}{2}" -f $size, $relativePath, $status)
}

# Services analysis
$servicesPath = "$CurrentPath\AndroidRemote\Services"
if (Test-Path $servicesPath) {
    Write-Both ""
    Write-Both "[EXTRACTED SERVICES]" -Color Green
    $services = Get-ChildItem -Path $servicesPath -Filter "*.cs"
    $servicesStats = $services | Measure-Object -Property Length -Sum

    Write-Both "  Total services: $($services.Count)"
    Write-Both "  Total size: $([math]::Round($servicesStats.Sum / 1KB, 1)) KB"
    Write-Both ""

    $services | Sort-Object Length -Descending | ForEach-Object {
        $size = [math]::Round($_.Length / 1KB, 1)
        Write-Both ("  {0,6} KB - {1}" -f $size, $_.Name)
    }
}

# Folder breakdown
if ($Detailed) {
    Write-Both ""
    Write-Both "[FOLDER BREAKDOWN]" -Color Cyan
    Get-ChildItem -Path $CurrentPath -Directory -Recurse | ForEach-Object {
        $folderFiles = Get-ChildItem -Path $_.FullName -Filter "*.cs" -File -ErrorAction SilentlyContinue
        if ($folderFiles) {
            $folderStats = $folderFiles | Measure-Object -Property Length -Sum
            $relativePath = $_.FullName.Replace("$CurrentPath\", "")
            Write-Both ("  {0,6} KB ({1,3} files) - {2}" -f [math]::Round($folderStats.Sum / 1KB, 1), $folderStats.Count, $relativePath)
        }
    }
}

# Risk status check
Write-Both ""
Write-Both "[RISK STATUS SUMMARY]" -Color Cyan
$riskFile = "$CurrentPath\Documentation\RISK_REMEDIATION_BACKLOG.md"
if (Test-Path $riskFile) {
    $riskContent = Get-Content $riskFile -Raw

    $completed = ([regex]::Matches($riskContent, '\*\*Status\*\*:\s*(COMPLETED|NOT AN ISSUE|ADDRESSED)').Count)
    $inProgress = ([regex]::Matches($riskContent, '\*\*Status\*\*:\s*IN PROGRESS').Count)
    $notStarted = ([regex]::Matches($riskContent, '\*\*Status\*\*:\s*NOT STARTED').Count)
    $lowPriority = ([regex]::Matches($riskContent, '\*\*Status\*\*:\s*LOW PRIORITY').Count)

    Write-Both "  Completed/Resolved: $completed" -Color Green
    Write-Both "  In Progress: $inProgress" -Color Yellow
    Write-Both "  Not Started: $notStarted" -Color Red
    Write-Both "  Low Priority: $lowPriority" -Color Gray
} else {
    Write-Both "  Risk file not found: $riskFile" -Color Yellow
}

Write-Both ""
Write-Both ("=" * 60) -Color Cyan
Write-Both "  Comparison complete. Run with -Detailed for folder breakdown." -Color Gray
Write-Both ("=" * 60) -Color Cyan
Write-Both ""

# Write to file unless -NoFile specified
if (-not $NoFile) {
    $fileOutput | Out-File -FilePath $outputFile -Encoding UTF8
    Write-Host "Output saved to: $outputFile" -ForegroundColor Green
}
