using System;
using System.Drawing;
using System.Windows.Forms;

namespace MusicBeePlugin.Clouseau.UI.Theme
{
    /// <summary>
    /// Centralized dark theme colors and utilities for mbClouseau UI.
    /// </summary>
    public static class DarkTheme
    {
        // Primary colors
        public static readonly Color Background = Color.FromArgb(30, 30, 30);
        public static readonly Color Panel = Color.FromArgb(45, 45, 48);
        public static readonly Color Border = Color.FromArgb(67, 67, 70);
        public static readonly Color Text = Color.FromArgb(241, 241, 241);
        public static readonly Color TextDim = Color.FromArgb(153, 153, 153);
        public static readonly Color Accent = Color.FromArgb(0, 122, 204);
        public static readonly Color Selection = Color.FromArgb(51, 51, 51);
        public static readonly Color Hover = Color.FromArgb(62, 62, 64);

        /// <summary>
        /// Recursively applies dark theme to a control and its children.
        /// </summary>
        public static void Apply(Control control)
        {
            control.BackColor = Background;
            control.ForeColor = Text;

            if (control is ListView lv)
            {
                lv.BackColor = Panel;
                lv.ForeColor = Text;
            }
            else if (control is TextBox tb)
            {
                tb.BackColor = Panel;
                tb.ForeColor = Text;
                tb.BorderStyle = BorderStyle.FixedSingle;
            }
            else if (control is Button btn)
            {
                btn.BackColor = Panel;
                btn.ForeColor = Text;
                btn.FlatStyle = FlatStyle.Flat;
                btn.FlatAppearance.BorderColor = Border;
                btn.FlatAppearance.MouseOverBackColor = Hover;
            }
            else if (control is ComboBox cb)
            {
                cb.BackColor = Panel;
                cb.ForeColor = Text;
                cb.FlatStyle = FlatStyle.Flat;
            }
            else if (control is Label lbl)
            {
                lbl.ForeColor = Text;
            }
            else if (control is CheckBox chk)
            {
                chk.ForeColor = Text;
            }
            else if (control is TreeView tv)
            {
                tv.BackColor = Panel;
                tv.ForeColor = Text;
            }
            else if (control is GroupBox gb)
            {
                gb.BackColor = Panel;
                gb.ForeColor = Text;
            }

            foreach (Control child in control.Controls)
            {
                Apply(child);
            }
        }

        /// <summary>
        /// Creates a dark-themed button.
        /// </summary>
        public static Button CreateButton(string text, EventHandler onClick)
        {
            var button = new Button
            {
                Text = text,
                Height = 30,
                Margin = new Padding(5),
                BackColor = Panel,
                ForeColor = Text,
                FlatStyle = FlatStyle.Flat
            };
            button.FlatAppearance.BorderColor = Border;
            button.FlatAppearance.MouseOverBackColor = Hover;
            if (onClick != null)
                button.Click += onClick;
            return button;
        }

        /// <summary>
        /// Creates a dark-themed GroupBox for info panels.
        /// </summary>
        public static GroupBox CreateInfoPanel(string title)
        {
            return new GroupBox
            {
                Text = title,
                Dock = DockStyle.Fill,
                Margin = new Padding(5),
                BackColor = Panel,
                ForeColor = Text,
                Font = new Font("Segoe UI", 9F, FontStyle.Bold)
            };
        }

        /// <summary>
        /// Formats a file size in bytes to human-readable format.
        /// </summary>
        public static string FormatFileSize(long bytes)
        {
            string[] sizes = { "B", "KB", "MB", "GB" };
            double len = bytes;
            int order = 0;
            while (len >= 1024 && order < sizes.Length - 1)
            {
                order++;
                len = len / 1024;
            }
            return $"{len:0.##} {sizes[order]}";
        }
    }

    /// <summary>
    /// Custom renderer for dark-themed menus.
    /// </summary>
    internal class DarkMenuRenderer : ToolStripProfessionalRenderer
    {
        public DarkMenuRenderer() : base(new DarkColorTable()) { }

        protected override void OnRenderMenuItemBackground(ToolStripItemRenderEventArgs e)
        {
            var item = e.Item;
            var g = e.Graphics;
            var rect = new Rectangle(Point.Empty, item.Size);

            if (item.Selected || item.Pressed)
            {
                using (var brush = new SolidBrush(DarkTheme.Hover))
                {
                    g.FillRectangle(brush, rect);
                }
            }
            else
            {
                using (var brush = new SolidBrush(DarkTheme.Panel))
                {
                    g.FillRectangle(brush, rect);
                }
            }
        }

        protected override void OnRenderToolStripBackground(ToolStripRenderEventArgs e)
        {
            using (var brush = new SolidBrush(DarkTheme.Panel))
            {
                e.Graphics.FillRectangle(brush, e.AffectedBounds);
            }
        }

        protected override void OnRenderToolStripBorder(ToolStripRenderEventArgs e)
        {
            using (var pen = new Pen(DarkTheme.Border))
            {
                e.Graphics.DrawRectangle(pen, 0, 0, e.AffectedBounds.Width - 1, e.AffectedBounds.Height - 1);
            }
        }

        protected override void OnRenderSeparator(ToolStripSeparatorRenderEventArgs e)
        {
            var g = e.Graphics;
            var bounds = new Rectangle(Point.Empty, e.Item.Size);

            using (var pen = new Pen(DarkTheme.Border))
            {
                var y = bounds.Height / 2;
                g.DrawLine(pen, 4, y, bounds.Width - 4, y);
            }
        }
    }

    /// <summary>
    /// Custom color table for dark theme.
    /// </summary>
    internal class DarkColorTable : ProfessionalColorTable
    {
        public override Color MenuItemSelected => DarkTheme.Hover;
        public override Color MenuItemSelectedGradientBegin => DarkTheme.Hover;
        public override Color MenuItemSelectedGradientEnd => DarkTheme.Hover;
        public override Color MenuItemBorder => DarkTheme.Border;
        public override Color MenuBorder => DarkTheme.Border;
        public override Color MenuStripGradientBegin => DarkTheme.Panel;
        public override Color MenuStripGradientEnd => DarkTheme.Panel;
        public override Color ToolStripDropDownBackground => DarkTheme.Panel;
        public override Color ImageMarginGradientBegin => DarkTheme.Panel;
        public override Color ImageMarginGradientMiddle => DarkTheme.Panel;
        public override Color ImageMarginGradientEnd => DarkTheme.Panel;
        public override Color SeparatorDark => DarkTheme.Border;
        public override Color SeparatorLight => DarkTheme.Border;
    }

    /// <summary>
    /// Custom renderer for dark-themed toolstrips.
    /// </summary>
    internal class DarkToolStripRenderer : ToolStripProfessionalRenderer
    {
        public DarkToolStripRenderer() : base(new DarkColorTable()) { }

        protected override void OnRenderButtonBackground(ToolStripItemRenderEventArgs e)
        {
            var btn = e.Item as ToolStripButton;
            var g = e.Graphics;
            var bounds = new Rectangle(Point.Empty, e.Item.Size);

            if (btn != null && (btn.Selected || btn.Pressed))
            {
                using (var brush = new SolidBrush(DarkTheme.Hover))
                {
                    g.FillRectangle(brush, bounds);
                }
            }
        }

        protected override void OnRenderToolStripBackground(ToolStripRenderEventArgs e)
        {
            using (var brush = new SolidBrush(DarkTheme.Panel))
            {
                e.Graphics.FillRectangle(brush, e.AffectedBounds);
            }
        }

        protected override void OnRenderToolStripBorder(ToolStripRenderEventArgs e)
        {
            // No border for toolstrip
        }

        protected override void OnRenderSeparator(ToolStripSeparatorRenderEventArgs e)
        {
            var g = e.Graphics;
            var bounds = new Rectangle(Point.Empty, e.Item.Size);

            using (var pen = new Pen(DarkTheme.Border))
            {
                if (e.Vertical)
                {
                    var x = bounds.Width / 2;
                    g.DrawLine(pen, x, 4, x, bounds.Height - 4);
                }
                else
                {
                    var y = bounds.Height / 2;
                    g.DrawLine(pen, 4, y, bounds.Width - 4, y);
                }
            }
        }
    }
}
