using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Reflection;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Introspection;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Spy tab for reflection-based exploration of MusicBee UI and assemblies.
    /// </summary>
    public class SpyTab : DashboardTabBase
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();
        private readonly ReflectionExplorer _reflectionExplorer;

        private TreeView _spyTreeView;
        private ListView _spyMetadataList;
        private ListView _spyEventsList;
        private TextBox _spySearchBox;
        private SpyToolbar _spyToolbar;
        private SplitContainer _spyMainSplit;
        private SplitContainer _spyRightSplit;
        private ToolStripButton _spyOverlayButton;

        /// <summary>
        /// Creates a new Spy tab.
        /// </summary>
        public SpyTab(ReflectionExplorer reflectionExplorer)
        {
            _reflectionExplorer = reflectionExplorer ?? new ReflectionExplorer();
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            BackColor = DarkBackground;

            // Main split container (left: tree, right: details)
            _spyMainSplit = new SplitContainer
            {
                Dock = DockStyle.Fill,
                Orientation = Orientation.Vertical,
                BackColor = DarkBackground,
                Panel1MinSize = 100,
                Panel2MinSize = 100
            };

            // Left panel: Toolbar + Tree View
            var leftPanel = new Panel { Dock = DockStyle.Fill, BackColor = DarkBackground };

            // Toolbar
            var toolbar = new ToolStrip
            {
                BackColor = DarkPanel,
                ForeColor = DarkText,
                GripStyle = ToolStripGripStyle.Hidden,
                Renderer = new DarkToolStripRenderer()
            };

            var refreshBtn = new ToolStripButton("Refresh", null, OnSpyRefresh) { ForeColor = DarkText };
            toolbar.Items.Add(refreshBtn);

            var expandBtn = new ToolStripButton("Expand All", null, OnSpyExpandAll) { ForeColor = DarkText };
            toolbar.Items.Add(expandBtn);

            var collapseBtn = new ToolStripButton("Collapse", null, OnSpyCollapseAll) { ForeColor = DarkText };
            toolbar.Items.Add(collapseBtn);

            toolbar.Items.Add(new ToolStripSeparator());

            // Spy Overlay button
            _spyOverlayButton = new ToolStripButton("Spy Overlay", null, OnSpyOverlayToggle)
            {
                ForeColor = DarkText,
                ToolTipText = "Start real-time UI element highlighting"
            };
            toolbar.Items.Add(_spyOverlayButton);

            toolbar.Items.Add(new ToolStripSeparator());

            var viewLabel = new ToolStripLabel("View:") { ForeColor = DarkText };
            toolbar.Items.Add(viewLabel);

            var viewCombo = new ToolStripComboBox { ForeColor = DarkText, Width = 120 };
            viewCombo.ComboBox.BackColor = DarkPanel;
            viewCombo.ComboBox.ForeColor = DarkText;
            viewCombo.ComboBox.FlatStyle = FlatStyle.Flat;
            viewCombo.Items.AddRange(new[] { "UI Controls", "Assemblies", "Forms", "Menus" });
            viewCombo.SelectedIndex = 0;
            viewCombo.SelectedIndexChanged += OnSpyViewChanged;
            toolbar.Items.Add(viewCombo);

            leftPanel.Controls.Add(toolbar);

            // Search box
            _spySearchBox = new TextBox
            {
                Dock = DockStyle.Top,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                BorderStyle = BorderStyle.FixedSingle,
                Font = new Font("Segoe UI", 9F)
            };
            _spySearchBox.TextChanged += OnSpySearchChanged;
            _spySearchBox.GotFocus += (s, e) => { if (_spySearchBox.Text == "Search...") _spySearchBox.Text = ""; };
            _spySearchBox.LostFocus += (s, e) => { if (string.IsNullOrEmpty(_spySearchBox.Text)) _spySearchBox.Text = "Search..."; };
            _spySearchBox.Text = "Search...";
            _spySearchBox.ForeColor = DarkTextDim;

            // Tree view
            _spyTreeView = new TreeView
            {
                Dock = DockStyle.Fill,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                BorderStyle = BorderStyle.None,
                Font = new Font("Consolas", 9F),
                ShowLines = true,
                ShowPlusMinus = true,
                ShowRootLines = true,
                FullRowSelect = true
            };
            _spyTreeView.AfterSelect += OnSpyTreeNodeSelected;
            _spyTreeView.NodeMouseDoubleClick += OnSpyTreeNodeDoubleClick;

            leftPanel.Controls.Add(_spyTreeView);
            leftPanel.Controls.Add(_spySearchBox);
            leftPanel.Controls.Add(toolbar);

            _spyMainSplit.Panel1.Controls.Add(leftPanel);

            // Right panel: Metadata + Events (vertical split)
            _spyRightSplit = new SplitContainer
            {
                Dock = DockStyle.Fill,
                Orientation = Orientation.Horizontal,
                BackColor = DarkBackground,
                Panel1MinSize = 100,
                Panel2MinSize = 100,
                SplitterWidth = 8
            };

            // Top-right: Metadata/Properties
            var metadataPanel = new Panel { Dock = DockStyle.Fill, BackColor = DarkBackground };

            var metadataLabel = new Label
            {
                Text = "Properties & Fields",
                Dock = DockStyle.Top,
                Height = 40,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                Font = new Font("Segoe UI", 9F, FontStyle.Bold),
                TextAlign = ContentAlignment.MiddleLeft,
                Padding = new Padding(8, 0, 0, 0)
            };

            _spyMetadataList = new ListView
            {
                Dock = DockStyle.Fill,
                View = View.Details,
                FullRowSelect = true,
                GridLines = true,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                BorderStyle = BorderStyle.None,
                Font = new Font("Consolas", 9F)
            };
            _spyMetadataList.Columns.Add("Name", 90);
            _spyMetadataList.Columns.Add("Type", 85);
            _spyMetadataList.Columns.Add("Value", 140);
            _spyMetadataList.Columns.Add("Access", 70);

            metadataPanel.Controls.Add(_spyMetadataList);
            metadataPanel.Controls.Add(metadataLabel);

            _spyRightSplit.Panel1.Controls.Add(metadataPanel);

            // Bottom-right: Events + Methods
            var eventsPanel = new Panel { Dock = DockStyle.Fill, BackColor = DarkBackground };

            var eventsToolbar = new Panel
            {
                Dock = DockStyle.Top,
                Height = 45,
                BackColor = DarkPanel
            };

            var eventsLabel = new Label
            {
                Text = "Event Handlers & Methods",
                Location = new Point(8, 0),
                Size = new Size(200, 45),
                TextAlign = ContentAlignment.MiddleLeft,
                ForeColor = DarkText,
                Font = new Font("Segoe UI", 9F, FontStyle.Bold)
            };

            var invokeBtn = DarkTheme.CreateButton("Invoke", OnSpyInvokeMethod);
            invokeBtn.Location = new Point(210, 8);
            invokeBtn.Size = new Size(90, 30);
            invokeBtn.BackColor = Color.FromArgb(70, 130, 180);

            eventsToolbar.Controls.Add(eventsLabel);
            eventsToolbar.Controls.Add(invokeBtn);

            _spyEventsList = new ListView
            {
                Dock = DockStyle.Fill,
                View = View.Details,
                FullRowSelect = true,
                GridLines = true,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                BorderStyle = BorderStyle.None,
                Font = new Font("Consolas", 9F)
            };
            _spyEventsList.Columns.Add("Event/Method", 100);
            _spyEventsList.Columns.Add("Handler", 115);
            _spyEventsList.Columns.Add("Target Type", 100);
            _spyEventsList.Columns.Add("Visibility", 80);

            eventsPanel.Controls.Add(_spyEventsList);
            eventsPanel.Controls.Add(eventsToolbar);

            _spyRightSplit.Panel2.Controls.Add(eventsPanel);

            _spyMainSplit.Panel2.Controls.Add(_spyRightSplit);

            Controls.Add(_spyMainSplit);

            // Set splitter distances on load when dimensions are known
            this.Load += (s, e) =>
            {
                if (_spyMainSplit.Width > 0)
                    _spyMainSplit.SplitterDistance = _spyMainSplit.Width / 3;
                if (_spyRightSplit.Height > 0)
                    _spyRightSplit.SplitterDistance = _spyRightSplit.Height / 2;
            };

            // Initial population
            PopulateSpyTree("UI Controls");
        }

        /// <inheritdoc/>
        public override void RefreshData()
        {
            PopulateSpyTree("UI Controls");
        }

        private void PopulateSpyTree(string viewMode)
        {
            _spyTreeView.Nodes.Clear();

            try
            {
                switch (viewMode)
                {
                    case "UI Controls":
                        PopulateUIControlsTree();
                        break;
                    case "Assemblies":
                        PopulateAssembliesTree();
                        break;
                    case "Forms":
                        PopulateFormsTree();
                        break;
                    case "Menus":
                        PopulateMenusTree();
                        break;
                }
            }
            catch (Exception ex)
            {
                var errorNode = _spyTreeView.Nodes.Add($"Error: {ex.Message}");
                errorNode.ForeColor = Color.Red;
            }
        }

        private void PopulateUIControlsTree()
        {
            var mainForm = _reflectionExplorer.FindMainForm();
            if (mainForm == null)
            {
                _spyTreeView.Nodes.Add("Could not find main form");
                return;
            }

            var rootNode = _spyTreeView.Nodes.Add($"MainForm ({mainForm.GetType().Name})");
            rootNode.Tag = mainForm;
            rootNode.ForeColor = Color.FromArgb(100, 200, 255);

            PopulateControlNode(rootNode, mainForm);

            rootNode.Expand();
        }

        private void PopulateControlNode(TreeNode parentNode, Control control)
        {
            foreach (Control child in control.Controls)
            {
                var name = string.IsNullOrEmpty(child.Name) ? $"[{child.GetType().Name}]" : child.Name;
                var nodeText = $"{name} ({child.GetType().Name})";
                if (!child.Visible) nodeText += " [HIDDEN]";

                var node = parentNode.Nodes.Add(nodeText);
                node.Tag = child;

                // Color code by type
                if (child is MenuStrip) node.ForeColor = Color.FromArgb(255, 200, 100);
                else if (child is ToolStrip) node.ForeColor = Color.FromArgb(200, 150, 255);
                else if (child is Panel || child is SplitContainer) node.ForeColor = Color.FromArgb(150, 200, 150);
                else if (!child.Visible) node.ForeColor = DarkTextDim;
                else node.ForeColor = DarkText;

                // Recurse
                if (child.Controls.Count > 0)
                {
                    PopulateControlNode(node, child);
                }
            }
        }

        private void PopulateAssembliesTree()
        {
            var categories = new[] { "musicbee", "plugins", "ui", "system" };

            foreach (var category in categories)
            {
                var categoryNode = _spyTreeView.Nodes.Add(category.ToUpper());
                categoryNode.ForeColor = Color.FromArgb(100, 200, 255);

                var assemblies = _reflectionExplorer.GetAssembliesByCategory(category);
                foreach (var asm in assemblies.Take(50))
                {
                    var asmNode = categoryNode.Nodes.Add($"{asm.Name} v{asm.Version}");
                    asmNode.Tag = asm.Assembly;
                    asmNode.ForeColor = DarkText;

                    asmNode.Nodes.Add("Loading...");
                }
            }
        }

        private void PopulateFormsTree()
        {
            var forms = _reflectionExplorer.GetOpenForms();

            foreach (var form in forms)
            {
                var node = _spyTreeView.Nodes.Add($"{form.Text} ({form.Type})");
                node.Tag = form.Form;
                node.ForeColor = Color.FromArgb(100, 200, 255);

                node.Nodes.Add($"{form.ControlCount} controls...");
            }
        }

        private void PopulateMenusTree()
        {
            var mainForm = _reflectionExplorer.FindMainForm();
            if (mainForm == null)
            {
                _spyTreeView.Nodes.Add("Could not find main form");
                return;
            }

            // Collect all MenuStrips - check MainMenuStrip property and Controls
            var menuStrips = new List<MenuStrip>();

            if (mainForm.MainMenuStrip != null)
            {
                menuStrips.Add(mainForm.MainMenuStrip);
            }

            foreach (Control control in mainForm.Controls)
            {
                if (control is MenuStrip ms && !menuStrips.Contains(ms))
                {
                    menuStrips.Add(ms);
                }
            }

            if (menuStrips.Count == 0)
            {
                _spyTreeView.Nodes.Add("No menus found on main form");
                return;
            }

            foreach (var menuStrip in menuStrips)
            {
                var menuNode = _spyTreeView.Nodes.Add($"Menu: {menuStrip.Name}");
                menuNode.Tag = menuStrip;
                menuNode.ForeColor = Color.FromArgb(255, 200, 100);

                foreach (var item in _reflectionExplorer.GetMenuItems(menuStrip))
                {
                    var itemNode = menuNode.Nodes.Add(item.Path);
                    itemNode.Tag = item;

                    if (item.ClickHandler != null)
                    {
                        itemNode.ForeColor = Color.FromArgb(100, 255, 100);
                        itemNode.Text += $" -> {item.ClickHandler.Name}()";
                    }
                    else
                    {
                        itemNode.ForeColor = DarkTextDim;
                    }
                }

                menuNode.Expand();
            }
        }

        #region Event Handlers

        private void OnSpyOverlayToggle(object sender, EventArgs e)
        {
            try
            {
                if (_spyToolbar == null || _spyToolbar.IsDisposed)
                {
                    var mainForm = _reflectionExplorer?.FindMainForm();
                    IntPtr targetHandle = mainForm?.Handle ?? IntPtr.Zero;

                    _spyToolbar = new SpyToolbar(targetHandle);
                    _spyToolbar.ToolbarClosed += OnSpyToolbarClosed;
                    _spyToolbar.Show();

                    _spyOverlayButton.Text = "Stop Overlay";
                    _spyOverlayButton.ForeColor = Color.FromArgb(0, 174, 255);

                    _spyToolbar.OverlayForm.StartSpyMode();

                    Logger.Info("Spy overlay started");
                }
                else
                {
                    _spyToolbar.OverlayForm.StopSpyMode();
                    _spyToolbar.Close();
                    _spyToolbar = null;

                    _spyOverlayButton.Text = "Spy Overlay";
                    _spyOverlayButton.ForeColor = DarkText;

                    Logger.Info("Spy overlay stopped");
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error toggling spy overlay");
                MessageBox.Show($"Error toggling spy overlay:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void OnSpyToolbarClosed(object sender, EventArgs e)
        {
            _spyToolbar = null;
            _spyOverlayButton.Text = "Spy Overlay";
            _spyOverlayButton.ForeColor = DarkText;
        }

        private void OnSpyRefresh(object sender, EventArgs e)
        {
            var viewCombo = ((ToolStrip)((ToolStripButton)sender).Owner).Items
                .OfType<ToolStripComboBox>().FirstOrDefault();
            var viewMode = viewCombo?.SelectedItem?.ToString() ?? "UI Controls";
            PopulateSpyTree(viewMode);
        }

        private void OnSpyExpandAll(object sender, EventArgs e)
        {
            _spyTreeView.ExpandAll();
        }

        private void OnSpyCollapseAll(object sender, EventArgs e)
        {
            _spyTreeView.CollapseAll();
        }

        private void OnSpyViewChanged(object sender, EventArgs e)
        {
            var combo = sender as ToolStripComboBox;
            if (combo?.SelectedItem != null)
            {
                PopulateSpyTree(combo.SelectedItem.ToString());
            }
        }

        private void OnSpySearchChanged(object sender, EventArgs e)
        {
            var searchText = _spySearchBox.Text?.ToLower();
            if (string.IsNullOrEmpty(searchText) || searchText == "search...") return;

            foreach (TreeNode node in GetAllNodes(_spyTreeView.Nodes))
            {
                if (node.Text.ToLower().Contains(searchText))
                {
                    _spyTreeView.SelectedNode = node;
                    node.EnsureVisible();
                    break;
                }
            }
        }

        private IEnumerable<TreeNode> GetAllNodes(TreeNodeCollection nodes)
        {
            foreach (TreeNode node in nodes)
            {
                yield return node;
                foreach (var child in GetAllNodes(node.Nodes))
                {
                    yield return child;
                }
            }
        }

        private void OnSpyTreeNodeSelected(object sender, TreeViewEventArgs e)
        {
            _spyMetadataList.Items.Clear();
            _spyEventsList.Items.Clear();

            var tag = e.Node?.Tag;
            if (tag == null) return;

            if (tag is Control control)
            {
                PopulateControlMetadata(control);
                PopulateControlEvents(control);
            }
            else if (tag is Assembly assembly)
            {
                PopulateAssemblyTypes(assembly, e.Node);
            }
            else if (tag is MenuItemInfo menuItem)
            {
                PopulateMenuItemInfo(menuItem);
            }
        }

        private void OnSpyTreeNodeDoubleClick(object sender, TreeNodeMouseClickEventArgs e)
        {
            var tag = e.Node?.Tag;
            if (tag is Assembly assembly)
            {
                e.Node.Nodes.Clear();
                PopulateAssemblyTypes(assembly, e.Node);
                e.Node.Expand();
            }
        }

        private void PopulateControlMetadata(Control control)
        {
            AddMetadataItem("Name", "string", control.Name, "public");
            AddMetadataItem("Type", "Type", control.GetType().FullName, "public");
            AddMetadataItem("Visible", "bool", control.Visible.ToString(), "public");
            AddMetadataItem("Enabled", "bool", control.Enabled.ToString(), "public");
            AddMetadataItem("Bounds", "Rectangle", control.Bounds.ToString(), "public");
            AddMetadataItem("Dock", "DockStyle", control.Dock.ToString(), "public");
            AddMetadataItem("Anchor", "AnchorStyles", control.Anchor.ToString(), "public");

            foreach (var field in _reflectionExplorer.GetPrivateFields(control).Take(30))
            {
                var access = field.IsPrivate ? "private" : (field.IsStatic ? "static" : "internal");
                var item = AddMetadataItem(field.Name, field.FieldType?.Name ?? "?", field.ValueString, access);
                if (field.IsPrivate) item.ForeColor = DarkTextDim;
            }
        }

        private void PopulateControlEvents(Control control)
        {
            foreach (var handler in _reflectionExplorer.GetEventHandlers(control))
            {
                var visibility = handler.HandlerMethod?.IsPrivate == true ? "private" : "public";
                var item = new ListViewItem(new[]
                {
                    handler.EventName,
                    handler.HandlerMethod?.Name ?? "?",
                    handler.TargetType?.Name ?? "?",
                    visibility
                });
                item.Tag = handler;
                item.ForeColor = handler.HandlerMethod?.IsPrivate == true ? Color.FromArgb(255, 200, 100) : DarkText;
                item.BackColor = DarkPanel;
                _spyEventsList.Items.Add(item);
            }

            var controlType = control.GetType();
            var methods = controlType.GetMethods(BindingFlags.NonPublic | BindingFlags.Instance)
                .Where(m => !m.IsSpecialName && m.GetParameters().Length == 0)
                .Take(20);

            foreach (var method in methods)
            {
                var item = new ListViewItem(new[]
                {
                    "[Method]",
                    method.Name + "()",
                    method.DeclaringType?.Name ?? "?",
                    method.IsPrivate ? "private" : "internal"
                });
                item.Tag = new MethodInvokeInfo { Target = control, Method = method };
                item.ForeColor = Color.FromArgb(150, 150, 255);
                item.BackColor = DarkPanel;
                _spyEventsList.Items.Add(item);
            }
        }

        private void PopulateAssemblyTypes(Assembly assembly, TreeNode parentNode)
        {
            try
            {
                var types = _reflectionExplorer.GetTypes(assembly).Take(100);

                foreach (var typeInfo in types)
                {
                    var typeNode = parentNode.Nodes.Add(typeInfo.FullName);
                    typeNode.Tag = typeInfo.Type;
                    typeNode.ForeColor = typeInfo.IsPublic ? DarkText : DarkTextDim;
                }

                _spyMetadataList.Items.Clear();
                AddMetadataItem("Assembly", "string", assembly.GetName().Name, "public");
                AddMetadataItem("Version", "string", assembly.GetName().Version?.ToString() ?? "?", "public");
                AddMetadataItem("Type Count", "int", types.Count().ToString(), "public");
            }
            catch (Exception ex)
            {
                AddMetadataItem("Error", "string", ex.Message, "error");
            }
        }

        private void PopulateMenuItemInfo(MenuItemInfo menuItem)
        {
            AddMetadataItem("Path", "string", menuItem.Path, "public");
            AddMetadataItem("Text", "string", menuItem.Text, "public");
            AddMetadataItem("Name", "string", menuItem.Name, "public");
            AddMetadataItem("Enabled", "bool", menuItem.Enabled.ToString(), "public");
            AddMetadataItem("Shortcut", "string", menuItem.ShortcutKeys ?? "None", "public");

            if (menuItem.ClickHandler != null)
            {
                AddMetadataItem("Handler", "MethodInfo", menuItem.ClickHandler.Name, "private");
                AddMetadataItem("Target", "Type", menuItem.HandlerTarget?.FullName ?? "?", "private");

                var item = new ListViewItem(new[]
                {
                    "Click",
                    menuItem.ClickHandler.Name + "()",
                    menuItem.HandlerTarget?.Name ?? "?",
                    menuItem.ClickHandler.IsPrivate ? "private" : "public"
                });
                item.Tag = menuItem;
                item.ForeColor = Color.FromArgb(100, 255, 100);
                item.BackColor = DarkPanel;
                _spyEventsList.Items.Add(item);
            }
        }

        private ListViewItem AddMetadataItem(string name, string type, string value, string access)
        {
            var item = new ListViewItem(new[] { name, type, value, access });
            item.ForeColor = DarkText;
            item.BackColor = DarkPanel;
            _spyMetadataList.Items.Add(item);
            return item;
        }

        private void OnSpyInvokeMethod(object sender, EventArgs e)
        {
            if (_spyEventsList.SelectedItems.Count == 0)
            {
                MessageBox.Show("Select a method or event handler to invoke.", "No Selection",
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
                return;
            }

            var tag = _spyEventsList.SelectedItems[0].Tag;

            try
            {
                if (tag is EventHandlerInfo handlerInfo && handlerInfo.Target != null && handlerInfo.HandlerMethod != null)
                {
                    var result = MessageBox.Show(
                        $"Invoke method: {handlerInfo.HandlerMethod.Name}?\n\n" +
                        $"Target: {handlerInfo.TargetType?.Name}\n" +
                        $"This may trigger UI actions.",
                        "Confirm Invoke",
                        MessageBoxButtons.YesNo,
                        MessageBoxIcon.Question);

                    if (result == DialogResult.Yes)
                    {
                        var parameters = handlerInfo.HandlerMethod.GetParameters();
                        object[] args = parameters.Length == 2
                            ? new object[] { null, EventArgs.Empty }
                            : null;

                        handlerInfo.HandlerMethod.Invoke(handlerInfo.Target, args);
                        Logger.Info($"Invoked: {handlerInfo.HandlerMethod.Name}");
                    }
                }
                else if (tag is MethodInvokeInfo methodInfo)
                {
                    var result = MessageBox.Show(
                        $"Invoke method: {methodInfo.Method.Name}()?\n\n" +
                        $"Target: {methodInfo.Target?.GetType().Name}\n" +
                        $"This may trigger UI actions.",
                        "Confirm Invoke",
                        MessageBoxButtons.YesNo,
                        MessageBoxIcon.Question);

                    if (result == DialogResult.Yes)
                    {
                        methodInfo.Method.Invoke(methodInfo.Target, null);
                        Logger.Info($"Invoked: {methodInfo.Method.Name}");
                    }
                }
                else if (tag is MenuItemInfo menuItem && menuItem.ClickHandler != null)
                {
                    var result = MessageBox.Show(
                        $"Invoke menu handler: {menuItem.ClickHandler.Name}?\n\n" +
                        $"Menu: {menuItem.Path}\n" +
                        $"This may trigger UI actions.",
                        "Confirm Invoke",
                        MessageBoxButtons.YesNo,
                        MessageBoxIcon.Question);

                    if (result == DialogResult.Yes)
                    {
                        var parameters = menuItem.ClickHandler.GetParameters();
                        object[] args = parameters.Length == 2
                            ? new object[] { null, EventArgs.Empty }
                            : null;

                        var mainForm = _reflectionExplorer.FindMainForm();
                        if (mainForm != null && menuItem.HandlerTarget == mainForm.GetType())
                        {
                            menuItem.ClickHandler.Invoke(mainForm, args);
                            Logger.Info($"Invoked menu: {menuItem.Path}");
                        }
                        else
                        {
                            MessageBox.Show("Could not determine target instance for this handler.",
                                "Invoke Failed", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error invoking method");
                MessageBox.Show($"Error invoking method:\n{ex.Message}", "Invoke Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        #endregion

        /// <summary>
        /// Helper class for storing method invocation info.
        /// </summary>
        private class MethodInvokeInfo
        {
            public object Target { get; set; }
            public MethodInfo Method { get; set; }
        }
    }
}
