using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Introspection;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Plugins tab displaying discovered MusicBee plugins.
    /// </summary>
    public class PluginsTab : DashboardTabBase
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();
        private readonly PluginDiscovery _pluginDiscovery;
        private readonly string _pluginsPath;
        private ListView _pluginsListView;

        /// <summary>
        /// Creates a new Plugins tab.
        /// </summary>
        public PluginsTab(PluginDiscovery pluginDiscovery, string pluginsPath)
        {
            _pluginDiscovery = pluginDiscovery;
            _pluginsPath = pluginsPath;
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            var panel = new Panel
            {
                Dock = DockStyle.Fill,
                BackColor = DarkBackground
            };

            // Toolbar
            var toolbar = new ToolStrip
            {
                BackColor = DarkPanel,
                ForeColor = DarkText,
                GripStyle = ToolStripGripStyle.Hidden,
                Renderer = new DarkToolStripRenderer()
            };

            var refreshBtn = new ToolStripButton("Refresh", null, (s, e) => RefreshData())
            {
                ForeColor = DarkText
            };
            toolbar.Items.Add(refreshBtn);

            toolbar.Items.Add(new ToolStripSeparator());

            var exportBtn = new ToolStripButton("Export", null, OnExportPlugins)
            {
                ForeColor = DarkText
            };
            toolbar.Items.Add(exportBtn);

            toolbar.Items.Add(new ToolStripSeparator());

            var pathLabel = new ToolStripLabel($"Path: {_pluginsPath}")
            {
                ForeColor = DarkTextDim
            };
            toolbar.Items.Add(pathLabel);

            // Plugins list
            _pluginsListView = new ListView
            {
                Dock = DockStyle.Fill,
                View = View.Details,
                FullRowSelect = true,
                GridLines = true,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                BorderStyle = BorderStyle.None
            };

            _pluginsListView.Columns.Add("Plugin Name", 200);
            _pluginsListView.Columns.Add("Version", 80);
            _pluginsListView.Columns.Add("Type", 100);
            _pluginsListView.Columns.Add("Author", 120);
            _pluginsListView.Columns.Add("File", 150);
            _pluginsListView.Columns.Add("Size", 80);
            _pluginsListView.Columns.Add("Modified", 130);

            // Context menu
            var contextMenu = new ContextMenuStrip
            {
                BackColor = DarkPanel,
                ForeColor = DarkText,
                Renderer = new DarkMenuRenderer()
            };
            var openFolderItem = new ToolStripMenuItem("Open Plugins Folder", null, (s, e) =>
            {
                try
                {
                    Process.Start("explorer.exe", _pluginsPath);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error opening plugins folder");
                }
            })
            { ForeColor = DarkText };
            contextMenu.Items.Add(openFolderItem);

            var copyInfoItem = new ToolStripMenuItem("Copy Plugin Info", null, OnCopyPluginInfo)
            { ForeColor = DarkText };
            contextMenu.Items.Add(copyInfoItem);

            _pluginsListView.ContextMenuStrip = contextMenu;

            panel.Controls.Add(_pluginsListView);
            panel.Controls.Add(toolbar);

            Controls.Add(panel);
        }

        /// <inheritdoc/>
        public override void RefreshData()
        {
            try
            {
                _pluginsListView.Items.Clear();

                var plugins = _pluginDiscovery.DiscoverPlugins();

                foreach (var plugin in plugins.OrderBy(p => p.PluginName ?? p.FileName))
                {
                    var item = new ListViewItem(new[]
                    {
                        plugin.PluginName ?? plugin.FileName,
                        plugin.IsValidPlugin ? $"{plugin.VersionMajor}.{plugin.VersionMinor}.{plugin.Revision}" : "N/A",
                        plugin.PluginType ?? (plugin.IsValidPlugin ? "Unknown" : "Not a Plugin"),
                        plugin.PluginAuthor ?? "",
                        plugin.FileName,
                        FormatFileSize(plugin.FileSizeBytes),
                        plugin.LastModified.ToString("yyyy-MM-dd HH:mm")
                    });

                    if (!plugin.IsValidPlugin)
                    {
                        item.ForeColor = DarkTextDim;
                    }
                    else
                    {
                        item.ForeColor = DarkText;
                    }

                    item.BackColor = DarkPanel;
                    item.Tag = plugin;
                    _pluginsListView.Items.Add(item);
                }

                // Auto-resize columns to fit content
                foreach (ColumnHeader column in _pluginsListView.Columns)
                {
                    column.Width = -2;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error refreshing plugins");
                MessageBox.Show($"Error refreshing plugins:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void OnExportPlugins(object sender, EventArgs e)
        {
            using (var dialog = new SaveFileDialog())
            {
                dialog.Filter = "Text Files (*.txt)|*.txt";
                dialog.DefaultExt = "txt";
                dialog.FileName = $"clouseau_plugins_{DateTime.Now:yyyyMMdd_HHmmss}";

                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    try
                    {
                        var content = _pluginDiscovery.GetDiscoverySummary();
                        File.WriteAllText(dialog.FileName, content);

                        MessageBox.Show($"Plugins exported to:\n{dialog.FileName}", "Export Complete",
                            MessageBoxButtons.OK, MessageBoxIcon.Information);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Error exporting plugins");
                        MessageBox.Show($"Error exporting plugins:\n{ex.Message}", "Error",
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        private void OnCopyPluginInfo(object sender, EventArgs e)
        {
            if (_pluginsListView.SelectedItems.Count > 0)
            {
                var plugin = _pluginsListView.SelectedItems[0].Tag as DiscoveredPlugin;
                if (plugin != null)
                {
                    var info = plugin.ToDisplayString();
                    Clipboard.SetText(info);
                    Logger.Debug($"Copied plugin info for {plugin.PluginName ?? plugin.FileName}");
                }
            }
        }
    }
}
