using System;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Core;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Overview tab displaying session info, event statistics, and quick actions.
    /// </summary>
    public class OverviewTab : DashboardTabBase
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();
        private readonly Plugin.MusicBeeApiInterface _mbApi;
        private readonly StateManager _stateManager;
        private readonly DateTime _sessionStartTime;
        private readonly Action _onCreateDump;
        private readonly Action _onRefreshAll;
        private readonly Action _onOpenUIGallery;

        private Label _sessionInfoLabel;
        private Label _eventStatsLabel;
        private Label _musicBeeInfoLabel;
        private Image _logoImage;

        /// <summary>
        /// Creates a new Overview tab.
        /// </summary>
        public OverviewTab(
            Plugin.MusicBeeApiInterface mbApi,
            StateManager stateManager,
            DateTime sessionStartTime,
            Action onCreateDump,
            Action onRefreshAll,
            Action onOpenUIGallery)
        {
            _mbApi = mbApi;
            _stateManager = stateManager;
            _sessionStartTime = sessionStartTime;
            _onCreateDump = onCreateDump;
            _onRefreshAll = onRefreshAll;
            _onOpenUIGallery = onOpenUIGallery;
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            // Container for header + main content
            var container = new Panel
            {
                Dock = DockStyle.Fill,
                BackColor = DarkBackground
            };

            // Header panel with logo and version
            var headerPanel = new Panel
            {
                Dock = DockStyle.Top,
                Height = 80,
                BackColor = DarkPanel,
                Padding = new Padding(15, 10, 15, 10)
            };

            // Try to load the Clouseau icon from embedded resource
            var iconBox = new PictureBox
            {
                Size = new Size(60, 60),
                Location = new Point(15, 10),
                SizeMode = PictureBoxSizeMode.Zoom,
                BackColor = Color.Transparent
            };
            try
            {
                var assembly = System.Reflection.Assembly.GetExecutingAssembly();
                using (var stream = assembly.GetManifestResourceStream("HALRAD_mb_Clouseau.png"))
                {
                    if (stream != null)
                    {
                        _logoImage = Image.FromStream(stream);
                        iconBox.Image = _logoImage;
                    }
                }
            }
            catch { /* Icon is optional */ }
            headerPanel.Controls.Add(iconBox);

            var titleLabel = new Label
            {
                Text = "mbClouseau",
                Location = new Point(85, 12),
                AutoSize = true,
                Font = new Font("Segoe UI", 18F, FontStyle.Bold),
                ForeColor = Color.FromArgb(100, 180, 255)
            };
            headerPanel.Controls.Add(titleLabel);

            var taglineLabel = new Label
            {
                Text = "MusicBee Debug & Trace Utility",
                Location = new Point(585, 40),
                AutoSize = true,
                Font = new Font("Segoe UI", 9F, FontStyle.Italic),
                ForeColor = Color.FromArgb(180, 180, 180)
            };
            headerPanel.Controls.Add(taglineLabel);

            var versionLabel = new Label
            {
                Text = $"v{typeof(OverviewTab).Assembly.GetName().Version}",
                AutoSize = true,
                Font = new Font("Segoe UI", 9F),
                ForeColor = DarkText,
                Anchor = AnchorStyles.Top | AnchorStyles.Right
            };
            versionLabel.Location = new Point(headerPanel.Width - versionLabel.PreferredWidth - 20, 48);
            headerPanel.Controls.Add(versionLabel);

            var mainPanel = new TableLayoutPanel
            {
                Dock = DockStyle.Fill,
                ColumnCount = 2,
                RowCount = 2,
                BackColor = DarkBackground,
                Padding = new Padding(10)
            };

            mainPanel.ColumnStyles.Add(new ColumnStyle(SizeType.Percent, 50F));
            mainPanel.ColumnStyles.Add(new ColumnStyle(SizeType.Percent, 50F));
            mainPanel.RowStyles.Add(new RowStyle(SizeType.Percent, 50F));
            mainPanel.RowStyles.Add(new RowStyle(SizeType.Percent, 50F));

            // Session Info Panel (top-left)
            var sessionPanel = DarkTheme.CreateInfoPanel("Session Information");
            _sessionInfoLabel = new Label
            {
                Dock = DockStyle.Fill,
                Font = new Font("Consolas", 10F),
                ForeColor = DarkText,
                Text = "Loading...",
                Padding = new Padding(10)
            };
            sessionPanel.Controls.Add(_sessionInfoLabel);
            mainPanel.Controls.Add(sessionPanel, 0, 0);

            // Event Stats Panel (top-right)
            var eventPanel = DarkTheme.CreateInfoPanel("Event Statistics");
            _eventStatsLabel = new Label
            {
                Dock = DockStyle.Fill,
                Font = new Font("Consolas", 10F),
                ForeColor = DarkText,
                Text = "Loading...",
                Padding = new Padding(10)
            };
            eventPanel.Controls.Add(_eventStatsLabel);
            mainPanel.Controls.Add(eventPanel, 1, 0);

            // MusicBee Info Panel (bottom-left)
            var mbPanel = DarkTheme.CreateInfoPanel("MusicBee Information");
            _musicBeeInfoLabel = new Label
            {
                Dock = DockStyle.Fill,
                Font = new Font("Consolas", 10F),
                ForeColor = DarkText,
                Text = "Loading...",
                Padding = new Padding(10)
            };
            mbPanel.Controls.Add(_musicBeeInfoLabel);
            mainPanel.Controls.Add(mbPanel, 0, 1);

            // Actions Panel (bottom-right)
            var actionsPanel = DarkTheme.CreateInfoPanel("Quick Actions");
            var actionsContent = new FlowLayoutPanel
            {
                Dock = DockStyle.Fill,
                FlowDirection = FlowDirection.TopDown,
                Padding = new Padding(10),
                BackColor = DarkPanel
            };

            var dumpButton = DarkTheme.CreateButton("Create State Dump", (s, e) => _onCreateDump?.Invoke());
            dumpButton.Width = 280;
            dumpButton.Height = 45;
            dumpButton.Margin = new Padding(5, 10, 5, 5);
            actionsContent.Controls.Add(dumpButton);

            var refreshButton = DarkTheme.CreateButton("Refresh All", (s, e) => _onRefreshAll?.Invoke());
            refreshButton.Width = 280;
            refreshButton.Height = 45;
            refreshButton.Margin = new Padding(5, 10, 5, 5);
            actionsContent.Controls.Add(refreshButton);

            var exportButton = DarkTheme.CreateButton("Export Overview", OnExportOverview);
            exportButton.Width = 280;
            exportButton.Height = 45;
            exportButton.Margin = new Padding(5, 10, 5, 5);
            actionsContent.Controls.Add(exportButton);

            var uiGalleryButton = DarkTheme.CreateButton("UI Gallery", (s, e) => _onOpenUIGallery?.Invoke());
            uiGalleryButton.Width = 280;
            uiGalleryButton.Height = 45;
            uiGalleryButton.Margin = new Padding(5, 10, 5, 5);
            actionsContent.Controls.Add(uiGalleryButton);

            var openLogsButton = DarkTheme.CreateButton("Open Logs Folder", OnOpenLogsFolder);
            openLogsButton.Width = 280;
            openLogsButton.Height = 45;
            openLogsButton.Margin = new Padding(5, 10, 5, 5);
            actionsContent.Controls.Add(openLogsButton);

            actionsPanel.Controls.Add(actionsContent);
            mainPanel.Controls.Add(actionsPanel, 1, 1);

            // Add header and main panel to container (order matters for docking)
            container.Controls.Add(mainPanel);
            container.Controls.Add(headerPanel);
            Controls.Add(container);
        }

        /// <inheritdoc/>
        public override void OnTimerTick()
        {
            UpdateDisplay();
        }

        /// <inheritdoc/>
        public override void RefreshData()
        {
            UpdateDisplay();
        }

        private void UpdateDisplay()
        {
            if (_sessionInfoLabel == null) return;

            try
            {
                var duration = DateTime.Now - _sessionStartTime;
                var mbUptime = _stateManager?.GetMusicBeeUptime();

                var sessionText = new StringBuilder();
                sessionText.AppendLine($"Session Start:    {_sessionStartTime:yyyy-MM-dd HH:mm:ss}");
                if (mbUptime.HasValue)
                {
                    sessionText.AppendLine($"MusicBee Uptime:  {mbUptime.Value:hh\\:mm\\:ss}");
                }
                if (_stateManager != null)
                {
                    sessionText.AppendLine($"Process ID:       {_stateManager.MusicBeeProcessId}");
                    var persistedState = _stateManager.GetPersistedState();
                    sessionText.AppendLine($"Total Sessions:   {persistedState.TotalSessions}");
                }

                _sessionInfoLabel.Text = sessionText.ToString();

                // Event statistics
                var eventText = new StringBuilder();
                eventText.AppendLine($"Events/min:       {_stateManager?.GetEventsPerMinute():F1}");
                eventText.AppendLine();

                if (_stateManager?.EventCounts != null)
                {
                    eventText.AppendLine("Events by Type:");
                    foreach (var kvp in _stateManager.EventCounts.OrderByDescending(x => x.Value).Take(8))
                    {
                        eventText.AppendLine($"  {kvp.Key,-28} {kvp.Value,5}");
                    }
                }

                _eventStatsLabel.Text = eventText.ToString();

                // MusicBee information
                var mbText = new StringBuilder();
                mbText.AppendLine($"API Revision:     {_mbApi.ApiRevision}");
                mbText.AppendLine($"Version:          {_mbApi.MusicBeeVersion}");
                mbText.AppendLine($"Play State:       {_mbApi.Player_GetPlayState()}");
                mbText.AppendLine($"Volume:           {_mbApi.Player_GetVolume():P0}");
                mbText.AppendLine($"Shuffle:          {(_mbApi.Player_GetShuffle() ? "On" : "Off")}");
                mbText.AppendLine($"Repeat:           {_mbApi.Player_GetRepeat()}");
                mbText.AppendLine();

                var currentTrack = _mbApi.NowPlaying_GetFileTag(Plugin.MetaDataType.TrackTitle);
                var currentArtist = _mbApi.NowPlaying_GetFileTag(Plugin.MetaDataType.Artist);
                if (!string.IsNullOrEmpty(currentTrack))
                {
                    mbText.AppendLine($"Now Playing:");
                    mbText.AppendLine($"  {currentArtist}");
                    mbText.AppendLine($"  {currentTrack}");
                }

                _musicBeeInfoLabel.Text = mbText.ToString();
            }
            catch (Exception ex)
            {
                Logger.Trace($"Error updating overview display: {ex.Message}");
            }
        }

        private void OnOpenLogsFolder(object sender, EventArgs e)
        {
            try
            {
                var logsPath = Path.Combine(
                    Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData),
                    "MusicBee", "mb_clouseau", "logs");

                if (!Directory.Exists(logsPath))
                {
                    Directory.CreateDirectory(logsPath);
                }

                System.Diagnostics.Process.Start("explorer.exe", logsPath);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error opening logs folder");
                MessageBox.Show($"Error opening logs folder:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void OnExportOverview(object sender, EventArgs e)
        {
            using (var dialog = new SaveFileDialog())
            {
                dialog.Filter = "Text Files (*.txt)|*.txt";
                dialog.DefaultExt = "txt";
                dialog.FileName = $"clouseau_overview_{DateTime.Now:yyyyMMdd_HHmmss}";

                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    try
                    {
                        var sb = new StringBuilder();
                        sb.AppendLine("mbClouseau Overview Export");
                        sb.AppendLine($"Generated: {DateTime.Now:yyyy-MM-dd HH:mm:ss}");
                        sb.AppendLine(new string('=', 60));
                        sb.AppendLine();
                        sb.AppendLine("=== Session Information ===");
                        sb.AppendLine(_sessionInfoLabel?.Text ?? "N/A");
                        sb.AppendLine();
                        sb.AppendLine("=== Event Statistics ===");
                        sb.AppendLine(_eventStatsLabel?.Text ?? "N/A");
                        sb.AppendLine();
                        sb.AppendLine("=== MusicBee Information ===");
                        sb.AppendLine(_musicBeeInfoLabel?.Text ?? "N/A");

                        File.WriteAllText(dialog.FileName, sb.ToString());
                        MessageBox.Show($"Overview exported to:\n{dialog.FileName}", "Export Complete",
                            MessageBoxButtons.OK, MessageBoxIcon.Information);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Error exporting overview");
                        MessageBox.Show($"Error exporting overview:\n{ex.Message}", "Error",
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        /// <summary>
        /// Releases resources used by the OverviewTab.
        /// </summary>
        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                _logoImage?.Dispose();
                _logoImage = null;
            }
            base.Dispose(disposing);
        }
    }
}
