using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Text;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Core;
using MusicBeePlugin.Clouseau.Metrics;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Metrics tab displaying real-time system and process metrics.
    /// </summary>
    public class MetricsTab : DashboardTabBase
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();
        private readonly MetricsCollector _metricsCollector;
        private readonly Plugin.MusicBeeApiInterface _mbApi;
        private readonly DateTime _sessionStartTime;
        private readonly Func<int> _getEventCount;
        private Label _metricsLabel;
        private Button _exportButton;

        /// <summary>
        /// Creates a new Metrics tab.
        /// </summary>
        public MetricsTab(
            MetricsCollector metricsCollector,
            Plugin.MusicBeeApiInterface mbApi,
            DateTime sessionStartTime,
            Func<int> getEventCount)
        {
            _metricsCollector = metricsCollector;
            _mbApi = mbApi;
            _sessionStartTime = sessionStartTime;
            _getEventCount = getEventCount;
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            var panel = new Panel
            {
                Dock = DockStyle.Fill,
                Padding = new Padding(15),
                BackColor = DarkBackground
            };

            // Toolbar at top
            var toolbar = new Panel
            {
                Dock = DockStyle.Top,
                Height = 55,
                BackColor = DarkPanel,
                Padding = new Padding(10, 12, 10, 10)
            };

            _exportButton = DarkTheme.CreateButton("Export Metrics", OnExportMetrics);
            _exportButton.Dock = DockStyle.Left;
            _exportButton.Width = 120;
            toolbar.Controls.Add(_exportButton);

            var forceGcButton = DarkTheme.CreateButton("Force GC", OnForceGC);
            forceGcButton.Dock = DockStyle.Left;
            forceGcButton.Width = 100;
            toolbar.Controls.Add(forceGcButton);

            _metricsLabel = new Label
            {
                AutoSize = false,
                Dock = DockStyle.Fill,
                Font = new Font("Consolas", 10F),
                ForeColor = DarkText,
                Text = "Metrics collection in progress...",
                TextAlign = ContentAlignment.TopLeft,
                Padding = new Padding(10)
            };

            // Dock order: Fill first, then Top
            panel.Controls.Add(_metricsLabel);
            panel.Controls.Add(toolbar);

            Controls.Add(panel);
        }

        /// <inheritdoc/>
        public override void OnTimerTick()
        {
            UpdateDisplay();
        }

        /// <inheritdoc/>
        public override void RefreshData()
        {
            UpdateDisplay();
        }

        private void UpdateDisplay()
        {
            if (_metricsLabel == null) return;

            try
            {
                var sb = new StringBuilder();
                sb.AppendLine("=== System Metrics ===");
                sb.AppendLine();

                if (_metricsCollector != null)
                {
                    var snapshot = _metricsCollector.GetCurrentSnapshot();
                    if (snapshot != null)
                    {
                        sb.AppendLine($"Collection Count:     {_metricsCollector.CollectionCount}");
                        sb.AppendLine($"Collector Uptime:     {_metricsCollector.Uptime:hh\\:mm\\:ss}");
                        sb.AppendLine();

                        sb.AppendLine("--- Process ---");
                        sb.AppendLine($"Working Set:          {snapshot.MBWorkingSetMB:N0} MB");
                        sb.AppendLine($"Private Bytes:        {snapshot.MBPrivateBytesMB:N0} MB");
                        sb.AppendLine($"Managed Heap:         {snapshot.ManagedHeapMB:N1} MB");
                        sb.AppendLine($"Thread Count:         {snapshot.MBThreadCount}");
                        sb.AppendLine($"Handle Count:         {snapshot.MBHandleCount}");
                        sb.AppendLine();

                        sb.AppendLine("--- CLR / GC ---");
                        sb.AppendLine($"Gen 0 Collections:    {snapshot.GCGen0Collections}");
                        sb.AppendLine($"Gen 1 Collections:    {snapshot.GCGen1Collections}");
                        sb.AppendLine($"Gen 2 Collections:    {snapshot.GCGen2Collections}");
                        sb.AppendLine($"GC Time %:            {snapshot.GCTimePercent:F1}%");
                        sb.AppendLine();

                        sb.AppendLine("--- System ---");
                        sb.AppendLine($"CPU Usage:            {snapshot.CpuPercent:F1}%");
                        sb.AppendLine($"Memory Used:          {snapshot.MemoryPercent:F1}%");
                        sb.AppendLine($"Available Memory:     {snapshot.AvailableMemoryMB:N0} MB");
                        sb.AppendLine();

                        // Alerts
                        if (snapshot.HandleLeakSuspected || snapshot.MemoryLeakSuspected ||
                            snapshot.GCPressureHigh || snapshot.LowMemoryWarning)
                        {
                            sb.AppendLine("--- ALERTS ---");
                            if (snapshot.HandleLeakSuspected)
                                sb.AppendLine("[!] Handle leak suspected");
                            if (snapshot.MemoryLeakSuspected)
                                sb.AppendLine("[!] Memory leak suspected");
                            if (snapshot.GCPressureHigh)
                                sb.AppendLine("[!] High GC pressure");
                            if (snapshot.LowMemoryWarning)
                                sb.AppendLine("[!] Low system memory");
                        }
                    }
                    else
                    {
                        sb.AppendLine("Waiting for first metrics collection...");
                    }
                }
                else
                {
                    // Fallback to basic process metrics
                    var process = Process.GetCurrentProcess();
                    var workingSet = process.WorkingSet64 / (1024 * 1024);
                    var privateBytes = process.PrivateMemorySize64 / (1024 * 1024);
                    var threads = process.Threads.Count;
                    var handles = process.HandleCount;
                    var gcTotal = GC.GetTotalMemory(false) / (1024 * 1024);

                    var duration = DateTime.Now - _sessionStartTime;

                    sb.AppendLine($"Session Duration:     {duration:hh\\:mm\\:ss}");
                    sb.AppendLine($"Events Logged:        {_getEventCount?.Invoke() ?? 0}");
                    sb.AppendLine();
                    sb.AppendLine("--- Process Metrics ---");
                    sb.AppendLine($"Working Set:          {workingSet:N0} MB");
                    sb.AppendLine($"Private Bytes:        {privateBytes:N0} MB");
                    sb.AppendLine($"Managed Heap:         {gcTotal:N0} MB");
                    sb.AppendLine($"Thread Count:         {threads}");
                    sb.AppendLine($"Handle Count:         {handles}");
                    sb.AppendLine();
                    sb.AppendLine("--- GC Statistics ---");
                    sb.AppendLine($"Gen 0 Collections:    {GC.CollectionCount(0)}");
                    sb.AppendLine($"Gen 1 Collections:    {GC.CollectionCount(1)}");
                    sb.AppendLine($"Gen 2 Collections:    {GC.CollectionCount(2)}");
                    sb.AppendLine();
                    sb.AppendLine("--- MusicBee ---");
                    sb.AppendLine($"API Revision:         {_mbApi.ApiRevision}");
                    sb.AppendLine($"MusicBee Version:     {_mbApi.MusicBeeVersion}");
                    sb.AppendLine($"Play State:           {_mbApi.Player_GetPlayState()}");
                    sb.AppendLine($"Volume:               {_mbApi.Player_GetVolume():P0}");
                }

                _metricsLabel.Text = sb.ToString();
            }
            catch (Exception ex)
            {
                _metricsLabel.Text = $"Error updating metrics: {ex.Message}";
                Logger.Error(ex, "Error updating metrics display");
            }
        }

        private void OnExportMetrics(object sender, EventArgs e)
        {
            using (var dialog = new SaveFileDialog())
            {
                dialog.Filter = "Text Files (*.txt)|*.txt|JSON Files (*.json)|*.json";
                dialog.DefaultExt = "txt";
                dialog.FileName = $"clouseau_metrics_{DateTime.Now:yyyyMMdd_HHmmss}";

                if (dialog.ShowDialog() == DialogResult.OK)
                {
                    try
                    {
                        if (dialog.FileName.EndsWith(".json") && _metricsCollector != null)
                        {
                            var snapshot = _metricsCollector.GetCurrentSnapshot();
                            var json = Newtonsoft.Json.JsonConvert.SerializeObject(snapshot,
                                Newtonsoft.Json.Formatting.Indented);
                            File.WriteAllText(dialog.FileName, json);
                        }
                        else
                        {
                            var content = _metricsCollector?.GetDiagnosticSummary() ??
                                _metricsLabel?.Text ?? "No metrics available";
                            File.WriteAllText(dialog.FileName, content);
                        }

                        MessageBox.Show($"Metrics exported to:\n{dialog.FileName}", "Export Complete",
                            MessageBoxButtons.OK, MessageBoxIcon.Information);
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Error exporting metrics");
                        MessageBox.Show($"Error exporting metrics:\n{ex.Message}", "Error",
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }

        private void OnForceGC(object sender, EventArgs e)
        {
            try
            {
                Cursor = Cursors.WaitCursor;
                string report;

                if (_metricsCollector != null)
                {
                    report = _metricsCollector.ForceGCAndReport();
                }
                else
                {
                    var before = GC.GetTotalMemory(false);
                    GC.Collect();
                    GC.WaitForPendingFinalizers();
                    GC.Collect();
                    var after = GC.GetTotalMemory(true);
                    var freed = (before - after) / (1024 * 1024);
                    report = $"Forced GC complete.\nFreed: {freed}MB";
                }

                Cursor = Cursors.Default;
                UpdateDisplay();
                MessageBox.Show(report, "GC Complete", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
            catch (Exception ex)
            {
                Cursor = Cursors.Default;
                Logger.Error(ex, "Error forcing GC");
                MessageBox.Show($"Error: {ex.Message}", "Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }
    }
}
