using System;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Introspection;
using MusicBeePlugin.Clouseau.Logging;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Invoke tab for launching the Method Invoker and displaying assembly info.
    /// </summary>
    public class InvokeTab : DashboardTabBase
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();
        private readonly EventLogger _eventLogger;
        private ListView _assemblyListView;

        /// <summary>
        /// Creates a new Invoke tab.
        /// </summary>
        public InvokeTab(EventLogger eventLogger)
        {
            _eventLogger = eventLogger;
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            // Main panel with padding
            var mainPanel = new Panel
            {
                Dock = DockStyle.Fill,
                BackColor = DarkBackground,
                Padding = new Padding(10)
            };

            // Instructions panel with button - compact layout
            var instructionsPanel = new Panel
            {
                Dock = DockStyle.Top,
                Height = 90,
                BackColor = DarkPanel
            };

            var titleLabel = new Label
            {
                Text = "Method Invoker",
                Font = new Font("Segoe UI", 10F, FontStyle.Bold),
                ForeColor = DarkText,
                Location = new Point(10, 10),
                AutoSize = true
            };
            instructionsPanel.Controls.Add(titleLabel);

            var openInvokerButton = new Button
            {
                Text = "Open",
                Location = new Point(250, 8),
                Size = new Size(80, 28),
                BackColor = DarkAccent,
                ForeColor = Color.White,
                FlatStyle = FlatStyle.Flat,
                Font = new Font("Segoe UI", 8F, FontStyle.Bold),
                TextAlign = ContentAlignment.MiddleCenter
            };
            openInvokerButton.FlatAppearance.BorderSize = 0;
            openInvokerButton.Click += OnOpenMethodInvoker;
            instructionsPanel.Controls.Add(openInvokerButton);

            // Assembly list
            _assemblyListView = new ListView
            {
                Dock = DockStyle.Fill,
                View = View.Details,
                FullRowSelect = true,
                GridLines = true,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                BorderStyle = BorderStyle.None,
                Font = new Font("Consolas", 9F)
            };
            _assemblyListView.Columns.Add("Assembly", 250);
            _assemblyListView.Columns.Add("Version", 100);
            _assemblyListView.Columns.Add("Types", 80);
            _assemblyListView.Columns.Add("Category", 100);

            PopulateAssemblyList();

            // Dock order: Fill first, Top last (so Top takes space first)
            mainPanel.Controls.Add(_assemblyListView);
            mainPanel.Controls.Add(instructionsPanel);

            Controls.Add(mainPanel);
        }

        private void PopulateAssemblyList()
        {
            try
            {
                var reflectionExplorer = new ReflectionExplorer();
                var categories = new[] { "musicbee", "plugins" };

                foreach (var category in categories)
                {
                    var assemblies = reflectionExplorer.GetAssembliesByCategory(category);
                    foreach (var asm in assemblies.Take(20))
                    {
                        var item = new ListViewItem(new[]
                        {
                            asm.Name,
                            asm.Version ?? "?",
                            asm.TypeCount.ToString(),
                            category.ToUpper()
                        });
                        item.ForeColor = category == "musicbee" ? Color.FromArgb(100, 200, 255) : DarkText;
                        item.BackColor = DarkPanel;
                        _assemblyListView.Items.Add(item);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error populating quick assembly list");
            }
        }

        private void OnOpenMethodInvoker(object sender, EventArgs e)
        {
            try
            {
                var invokerForm = new MethodInvokerForm(_eventLogger);
                invokerForm.Show();
                Logger.Info("Method Invoker form opened");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error opening Method Invoker form");
                MessageBox.Show($"Error opening Method Invoker:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <inheritdoc/>
        public override void RefreshData()
        {
            _assemblyListView.Items.Clear();
            PopulateAssemblyList();
        }
    }
}
