using System;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Introspection;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Handlers tab for Event and Menu inspection.
    /// </summary>
    public class HandlersTab : DashboardTabBase
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();
        private readonly ReflectionExplorer _reflectionExplorer;
        private Label _statsLabel;

        /// <summary>
        /// Creates a new Handlers tab.
        /// </summary>
        public HandlersTab(ReflectionExplorer reflectionExplorer)
        {
            _reflectionExplorer = reflectionExplorer;
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            // Main panel with padding
            var mainPanel = new Panel
            {
                Dock = DockStyle.Fill,
                BackColor = DarkBackground,
                Padding = new Padding(10)
            };

            // Instructions panel with buttons
            var instructionsPanel = new Panel
            {
                Dock = DockStyle.Top,
                Height = 130,
                BackColor = DarkPanel
            };

            var titleLabel = new Label
            {
                Text = "Event Handlers & Menus",
                Font = new Font("Segoe UI", 11F, FontStyle.Bold),
                ForeColor = DarkText,
                Location = new Point(15, 20),
                AutoSize = true
            };
            instructionsPanel.Controls.Add(titleLabel);

            var descLabel = new Label
            {
                Text = "Inspect event handlers attached to controls and explore the MusicBee menu system.",
                Font = new Font("Segoe UI", 8F),
                ForeColor = DarkTextDim,
                Location = new Point(15, 50),
                AutoSize = true,
                MaximumSize = new Size(800, 0)
            };
            instructionsPanel.Controls.Add(descLabel);

            var openEventInspectorButton = new Button
            {
                Text = "Event",
                Location = new Point(15, 85),
                Size = new Size(80, 32),
                BackColor = Color.FromArgb(70, 130, 180),
                ForeColor = Color.White,
                FlatStyle = FlatStyle.Flat,
                Font = new Font("Segoe UI", 8F, FontStyle.Bold)
            };
            openEventInspectorButton.FlatAppearance.BorderSize = 0;
            openEventInspectorButton.Click += OnOpenEventInspector;
            instructionsPanel.Controls.Add(openEventInspectorButton);

            var openMenuInspectorButton = new Button
            {
                Text = "Menu",
                Location = new Point(105, 85),
                Size = new Size(80, 32),
                BackColor = Color.FromArgb(130, 100, 180),
                ForeColor = Color.White,
                FlatStyle = FlatStyle.Flat,
                Font = new Font("Segoe UI", 8F, FontStyle.Bold),
                Margin = new Padding(5)
            };
            openMenuInspectorButton.FlatAppearance.BorderSize = 0;
            openMenuInspectorButton.Click += OnOpenMenuInspector;
            instructionsPanel.Controls.Add(openMenuInspectorButton);

            // Quick stats panel
            var statsPanel = new GroupBox
            {
                Text = "Handler Statistics",
                Dock = DockStyle.Fill,
                ForeColor = DarkText,
                BackColor = DarkPanel,
                Font = new Font("Segoe UI", 9F, FontStyle.Bold),
                Padding = new Padding(10),
                Margin = new Padding(0, 10, 0, 0)
            };

            _statsLabel = new Label
            {
                Dock = DockStyle.Fill,
                Font = new Font("Consolas", 10F),
                ForeColor = DarkText,
                Text = "Loading statistics...",
                Padding = new Padding(10)
            };
            statsPanel.Controls.Add(_statsLabel);

            PopulateStatistics();

            // Dock order matters: Fill first, then Top
            mainPanel.Controls.Add(statsPanel);
            mainPanel.Controls.Add(instructionsPanel);

            Controls.Add(mainPanel);
        }

        private void PopulateStatistics()
        {
            try
            {
                var eventInspector = new EventInspector();
                var menuWalker = new MenuWalker();
                var mainForm = _reflectionExplorer.FindMainForm();

                if (mainForm != null)
                {
                    var eventStats = eventInspector.GetEventStatistics(mainForm);
                    var menuStats = menuWalker.GetMenuStatistics(mainForm);

                    var sb = new StringBuilder();
                    sb.AppendLine("=== Event Handler Statistics ===");
                    sb.AppendLine($"Controls with handlers: {eventStats.TotalControlsWithHandlers}");
                    sb.AppendLine($"Total event handlers: {eventStats.TotalHandlers}");
                    sb.AppendLine();

                    if (eventStats.HandlersByEvent.Count > 0)
                    {
                        sb.AppendLine("Top Events:");
                        foreach (var kvp in eventStats.HandlersByEvent.OrderByDescending(x => x.Value).Take(5))
                        {
                            sb.AppendLine($"  {kvp.Key}: {kvp.Value}");
                        }
                    }

                    sb.AppendLine();
                    sb.AppendLine("=== Menu Statistics ===");
                    sb.AppendLine($"Total menu items: {menuStats.TotalMenuItems}");
                    sb.AppendLine($"Items with handlers: {menuStats.MenuItemsWithHandlers}");
                    sb.AppendLine($"Items with shortcuts: {menuStats.MenuItemsWithShortcuts}");
                    sb.AppendLine($"Context menus: {menuStats.ContextMenuCount}");
                    sb.AppendLine($"Context menu items: {menuStats.TotalContextMenuItems}");

                    _statsLabel.Text = sb.ToString();
                }
                else
                {
                    _statsLabel.Text = "Could not find MusicBee main form.\nOpen Event Inspector or Menu Inspector to refresh.";
                }
            }
            catch (Exception ex)
            {
                _statsLabel.Text = $"Error loading statistics: {ex.Message}";
                Logger.Error(ex, "Error loading handler statistics");
            }
        }

        private void OnOpenEventInspector(object sender, EventArgs e)
        {
            try
            {
                var eventInspectorForm = new EventInspectorForm();
                eventInspectorForm.Show();
                Logger.Info("Event Inspector form opened");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error opening Event Inspector form");
                MessageBox.Show($"Error opening Event Inspector:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void OnOpenMenuInspector(object sender, EventArgs e)
        {
            try
            {
                var menuInspectorForm = new MenuInspectorForm(null, Settings, DataFolder);
                menuInspectorForm.Show();
                Logger.Info("Menu Inspector form opened");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error opening Menu Inspector form");
                MessageBox.Show($"Error opening Menu Inspector:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <inheritdoc/>
        public override void RefreshData()
        {
            PopulateStatistics();
        }
    }
}
