using System;
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Introspection;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Dumps tab for managing state dump files.
    /// </summary>
    public class DumpsTab : DashboardTabBase
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();
        private readonly StateDumper _stateDumper;
        private readonly string _dataFolder;
        private ListView _dumpsListView;

        /// <summary>
        /// Creates a new Dumps tab.
        /// </summary>
        public DumpsTab(StateDumper stateDumper, string dataFolder)
        {
            _stateDumper = stateDumper;
            _dataFolder = dataFolder;
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            var panel = new Panel
            {
                Dock = DockStyle.Fill,
                BackColor = DarkBackground
            };

            // Toolbar
            var toolbar = new ToolStrip
            {
                BackColor = DarkPanel,
                ForeColor = DarkText,
                GripStyle = ToolStripGripStyle.Hidden,
                Renderer = new DarkToolStripRenderer()
            };

            var createDumpBtn = new ToolStripButton("Create New Dump", null, OnCreateDump)
            { ForeColor = DarkText };
            toolbar.Items.Add(createDumpBtn);

            var refreshBtn = new ToolStripButton("Refresh List", null, (s, e) => RefreshData())
            { ForeColor = DarkText };
            toolbar.Items.Add(refreshBtn);

            toolbar.Items.Add(new ToolStripSeparator());

            var openFolderBtn = new ToolStripButton("Open Dumps Folder", null, (s, e) =>
            {
                try
                {
                    var dumpsPath = Path.Combine(_dataFolder, "dumps");
                    if (!Directory.Exists(dumpsPath))
                    {
                        Directory.CreateDirectory(dumpsPath);
                    }
                    Process.Start("explorer.exe", dumpsPath);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error opening dumps folder");
                }
            })
            { ForeColor = DarkText };
            toolbar.Items.Add(openFolderBtn);

            // Dumps list
            _dumpsListView = new ListView
            {
                Dock = DockStyle.Fill,
                View = View.Details,
                FullRowSelect = true,
                GridLines = true,
                BackColor = DarkPanel,
                ForeColor = DarkText,
                BorderStyle = BorderStyle.None
            };

            _dumpsListView.Columns.Add("File Name", 280);
            _dumpsListView.Columns.Add("Created", 150);
            _dumpsListView.Columns.Add("Size", 100);

            // Double-click to open
            _dumpsListView.DoubleClick += OnDumpDoubleClick;

            // Context menu
            var contextMenu = new ContextMenuStrip
            {
                BackColor = DarkPanel,
                ForeColor = DarkText,
                Renderer = new DarkMenuRenderer()
            };
            var openItem = new ToolStripMenuItem("Open in Editor", null, OnOpenDump)
            { ForeColor = DarkText };
            contextMenu.Items.Add(openItem);

            var deleteItem = new ToolStripMenuItem("Delete", null, OnDeleteDump)
            { ForeColor = DarkText };
            contextMenu.Items.Add(deleteItem);

            _dumpsListView.ContextMenuStrip = contextMenu;

            // Dock order: Fill first, Top last
            panel.Controls.Add(_dumpsListView);
            panel.Controls.Add(toolbar);

            Controls.Add(panel);
        }

        /// <inheritdoc/>
        public override void RefreshData()
        {
            try
            {
                _dumpsListView.Items.Clear();

                var dumps = _stateDumper.GetExistingDumps();

                foreach (var dump in dumps)
                {
                    var item = new ListViewItem(new[]
                    {
                        dump.FileName,
                        dump.CreatedTime.ToString("yyyy-MM-dd HH:mm:ss"),
                        FormatFileSize(dump.FileSizeBytes)
                    });

                    item.ForeColor = DarkText;
                    item.BackColor = DarkPanel;
                    item.Tag = dump;
                    _dumpsListView.Items.Add(item);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error refreshing dumps");
            }
        }

        /// <summary>
        /// Creates a new state dump.
        /// </summary>
        public void CreateDump()
        {
            OnCreateDump(this, EventArgs.Empty);
        }

        private void OnCreateDump(object sender, EventArgs e)
        {
            try
            {
                var form = FindForm();
                if (form != null) form.Cursor = Cursors.WaitCursor;

                var path = _stateDumper.DumpState();
                RefreshData();

                if (form != null) form.Cursor = Cursors.Default;

                var result = MessageBox.Show(
                    $"State dump created:\n{path}\n\nOpen in editor?",
                    "Dump Created",
                    MessageBoxButtons.YesNo,
                    MessageBoxIcon.Information);

                if (result == DialogResult.Yes)
                {
                    Process.Start("notepad.exe", path);
                }
            }
            catch (Exception ex)
            {
                var form = FindForm();
                if (form != null) form.Cursor = Cursors.Default;

                Logger.Error(ex, "Error creating state dump");
                MessageBox.Show($"Error creating dump:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void OnDumpDoubleClick(object sender, EventArgs e)
        {
            OnOpenDump(sender, e);
        }

        private void OnOpenDump(object sender, EventArgs e)
        {
            if (_dumpsListView.SelectedItems.Count == 0) return;

            var dump = _dumpsListView.SelectedItems[0].Tag as DumpFileInfo;
            if (dump != null && File.Exists(dump.FilePath))
            {
                try
                {
                    Process.Start("notepad.exe", dump.FilePath);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Error opening dump file");
                }
            }
        }

        private void OnDeleteDump(object sender, EventArgs e)
        {
            if (_dumpsListView.SelectedItems.Count == 0) return;

            var dump = _dumpsListView.SelectedItems[0].Tag as DumpFileInfo;
            if (dump != null)
            {
                var result = MessageBox.Show(
                    $"Delete dump file?\n{dump.FileName}",
                    "Confirm Delete",
                    MessageBoxButtons.YesNo,
                    MessageBoxIcon.Question);

                if (result == DialogResult.Yes)
                {
                    try
                    {
                        File.Delete(dump.FilePath);
                        RefreshData();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Error deleting dump file");
                        MessageBox.Show($"Error deleting file:\n{ex.Message}", "Error",
                            MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }
            }
        }
    }
}
