using System;
using System.Drawing;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Core;
using MusicBeePlugin.Clouseau.UI.Theme;

namespace MusicBeePlugin.Clouseau.UI.Tabs
{
    /// <summary>
    /// Base class for Dashboard tab UserControls.
    /// Provides common dependencies and theme access.
    /// </summary>
    public abstract class DashboardTabBase : UserControl
    {
        /// <summary>
        /// MusicBee API interface.
        /// </summary>
        protected Plugin.MusicBeeApiInterface MbApi { get; }

        /// <summary>
        /// Clouseau settings.
        /// </summary>
        protected ClouseauSettings Settings { get; }

        /// <summary>
        /// Session state manager.
        /// </summary>
        protected StateManager StateManager { get; }

        /// <summary>
        /// Data folder for dumps and exports.
        /// </summary>
        protected string DataFolder { get; }

        // Theme color shortcuts
        protected static Color DarkBackground => DarkTheme.Background;
        protected static Color DarkPanel => DarkTheme.Panel;
        protected static Color DarkBorder => DarkTheme.Border;
        protected static Color DarkText => DarkTheme.Text;
        protected static Color DarkTextDim => DarkTheme.TextDim;
        protected static Color DarkAccent => DarkTheme.Accent;
        protected static Color DarkSelection => DarkTheme.Selection;
        protected static Color DarkHover => DarkTheme.Hover;

        /// <summary>
        /// Creates a new tab with no dependencies (for design-time).
        /// </summary>
        protected DashboardTabBase()
        {
            // Default styling
            BackColor = DarkBackground;
            ForeColor = DarkText;
            Dock = DockStyle.Fill;
        }

        /// <summary>
        /// Creates a new tab with common dependencies.
        /// </summary>
        protected DashboardTabBase(
            Plugin.MusicBeeApiInterface mbApi,
            ClouseauSettings settings,
            StateManager stateManager,
            string dataFolder)
        {
            MbApi = mbApi;
            Settings = settings;
            StateManager = stateManager;
            DataFolder = dataFolder;

            // Default styling
            BackColor = DarkBackground;
            ForeColor = DarkText;
            Dock = DockStyle.Fill;
        }

        /// <summary>
        /// Called to refresh the tab's data.
        /// Override in derived classes.
        /// </summary>
        public virtual void RefreshData()
        {
            // Default: no-op
        }

        /// <summary>
        /// Called on each status timer tick (1 second).
        /// Override in tabs that need periodic updates (Overview, Metrics).
        /// </summary>
        public virtual void OnTimerTick()
        {
            // Default: no-op
        }

        /// <summary>
        /// Called when the tab becomes visible/active.
        /// </summary>
        public virtual void OnTabActivated()
        {
            // Default: no-op
        }

        /// <summary>
        /// Applies dark theme to this control and children.
        /// </summary>
        protected void ApplyDarkTheme()
        {
            DarkTheme.Apply(this);
        }

        /// <summary>
        /// Creates a dark-themed button.
        /// </summary>
        protected Button CreateDarkButton(string text, EventHandler onClick)
        {
            return DarkTheme.CreateButton(text, onClick);
        }

        /// <summary>
        /// Creates a dark-themed info panel (GroupBox).
        /// </summary>
        protected GroupBox CreateInfoPanel(string title)
        {
            return DarkTheme.CreateInfoPanel(title);
        }

        /// <summary>
        /// Formats file size for display.
        /// </summary>
        protected static string FormatFileSize(long bytes)
        {
            return DarkTheme.FormatFileSize(bytes);
        }
    }
}
