using System;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using MusicBeePlugin.Clouseau.Core;
using MusicBeePlugin.Clouseau.Introspection;
using MusicBeePlugin.Clouseau.Logging;
using MusicBeePlugin.Clouseau.Metrics;
using MusicBeePlugin.Clouseau.UI.Tabs;
using MusicBeePlugin.Clouseau.UI.Theme;
using NLog;

namespace MusicBeePlugin.Clouseau.UI
{
    /// <summary>
    /// Full dashboard window for mbClouseau.
    /// Provides comprehensive view of events, metrics, plugins, and dumps.
    /// Dark theme consistent with MusicBee.
    /// </summary>
    public class DashboardForm : Form
    {
        private static readonly Logger Logger = NLog.LogManager.GetCurrentClassLogger();

        // Theme color shortcuts
        private static Color DarkBackground => DarkTheme.Background;
        private static Color DarkPanel => DarkTheme.Panel;
        private static Color DarkText => DarkTheme.Text;

        // UI Components
        private MenuStrip _menuStrip;
        private TabControl _tabControl;
        private StatusStrip _statusStrip;
        private ToolStripStatusLabel _sessionDurationLabel;
        private ToolStripStatusLabel _eventCountLabel;
        private Timer _statusTimer;

        // Tab pages
        private TabPage _overviewTab;
        private TabPage _eventsTab;
        private TabPage _metricsTab;
        private TabPage _pluginsTab;
        private TabPage _dumpsTab;
        private TabPage _spyTab;
        private TabPage _invokeTab;
        private TabPage _handlersTab;

        // Components on tabs
        private OverviewTab _overviewTabControl;
        private EventsTab _eventsTabControl;
        private InvokeTab _invokeTabControl;
        private HandlersTab _handlersTabControl;
        private MetricsTab _metricsTabControl;
        private PluginsTab _pluginsTabControl;
        private DumpsTab _dumpsTabControl;
        private SpyTab _spyTabControl;
        private ReflectionExplorer _reflectionExplorer;

        // Dependencies
        private readonly Plugin.MusicBeeApiInterface _mbApi;
        private readonly string _dataFolder;
        private readonly string _pluginsPath;
        private readonly ClouseauSettings _settings;
        private readonly StateManager _stateManager;
        private readonly MetricsCollector _metricsCollector;
        private readonly EventLogger _eventLogger;
        private readonly PluginDiscovery _pluginDiscovery;
        private readonly StateDumper _stateDumper;

        // State
        private readonly DateTime _sessionStartTime;

        /// <summary>
        /// Creates a new DashboardForm with full dependency injection.
        /// </summary>
        /// <param name="mbApi">MusicBee API interface.</param>
        /// <param name="dataFolder">Data folder for dumps.</param>
        /// <param name="pluginsPath">Path to MusicBee Plugins folder.</param>
        /// <param name="settings">Clouseau settings.</param>
        /// <param name="stateManager">State manager for session tracking.</param>
        /// <param name="metricsCollector">Metrics collector for system metrics.</param>
        /// <param name="eventLogger">Event logger for log entries.</param>
        /// <param name="pluginDiscovery">Plugin discovery for plugin enumeration.</param>
        public DashboardForm(
            Plugin.MusicBeeApiInterface mbApi,
            string dataFolder,
            string pluginsPath,
            ClouseauSettings settings = null,
            StateManager stateManager = null,
            MetricsCollector metricsCollector = null,
            EventLogger eventLogger = null,
            PluginDiscovery pluginDiscovery = null)
        {
            _mbApi = mbApi;
            _dataFolder = dataFolder;
            _pluginsPath = pluginsPath;
            _settings = settings ?? new ClouseauSettings();
            _stateManager = stateManager;
            _metricsCollector = metricsCollector;
            _eventLogger = eventLogger;
            _sessionStartTime = stateManager?.SessionStartTime ?? DateTime.Now;

            // Initialize helpers
            _stateDumper = new StateDumper(mbApi, Path.Combine(dataFolder, "dumps"), pluginsPath);
            _pluginDiscovery = pluginDiscovery ?? new PluginDiscovery(pluginsPath);

            // Subscribe to event logger if available
            if (_eventLogger != null)
            {
                _eventLogger.LogEntryAdded += OnEventLoggerEntryAdded;
            }

            // Form properties
            Text = "mbClouseau Dashboard";
            MinimumSize = new Size(1000, 800);
            Icon = null; // No icon in MVP

            // Load saved window position/size
            LoadWindowSettings();

            // Apply dark theme to form
            BackColor = DarkBackground;
            ForeColor = DarkText;

            // Initialize components
            InitializeMenu();
            InitializeTabs();
            InitializeStatusBar();

            // Add controls in order
            Controls.Add(_tabControl);
            Controls.Add(_menuStrip);
            Controls.Add(_statusStrip);

            MainMenuStrip = _menuStrip;

            // Status update timer (1 second for real-time updates)
            _statusTimer = new Timer { Interval = 1000 };
            _statusTimer.Tick += OnStatusTimerTick;
            _statusTimer.Start();

            // Load initial data
            _pluginsTabControl?.RefreshData();
            _dumpsTabControl?.RefreshData();
            _overviewTabControl?.RefreshData();
            _metricsTabControl?.RefreshData();
        }

        /// <summary>
        /// Backward-compatible constructor.
        /// </summary>
        public DashboardForm(Plugin.MusicBeeApiInterface mbApi, string dataFolder, string pluginsPath)
            : this(mbApi, dataFolder, pluginsPath, null, null, null, null, null)
        {
        }

        /// <summary>
        /// Handles new log entries from EventLogger.
        /// </summary>
        private void OnEventLoggerEntryAdded(object sender, LogEntry entry)
        {
            if (entry != null)
            {
                AddLogEntry(entry);
            }
        }

        /// <summary>
        /// Initializes the menu bar with dark theme.
        /// </summary>
        private void InitializeMenu()
        {
            _menuStrip = new MenuStrip
            {
                BackColor = DarkPanel,
                ForeColor = DarkText,
                Renderer = new DarkMenuRenderer()
            };

            // File menu
            var fileMenu = new ToolStripMenuItem("File") { ForeColor = DarkText };

            var exportItem = new ToolStripMenuItem("Export Log...", null, OnExportLog) { ForeColor = DarkText };
            exportItem.ShortcutKeys = Keys.Control | Keys.E;
            fileMenu.DropDownItems.Add(exportItem);

            var createDumpItem = new ToolStripMenuItem("Create State Dump", null, OnCreateDump) { ForeColor = DarkText };
            createDumpItem.ShortcutKeys = Keys.Control | Keys.D;
            fileMenu.DropDownItems.Add(createDumpItem);

            fileMenu.DropDownItems.Add(new ToolStripSeparator());

            var closeItem = new ToolStripMenuItem("Close", null, (s, e) => Close()) { ForeColor = DarkText };
            closeItem.ShortcutKeys = Keys.Alt | Keys.F4;
            fileMenu.DropDownItems.Add(closeItem);

            _menuStrip.Items.Add(fileMenu);

            // View menu
            var viewMenu = new ToolStripMenuItem("View") { ForeColor = DarkText };

            var refreshItem = new ToolStripMenuItem("Refresh All", null, OnRefreshAll) { ForeColor = DarkText };
            refreshItem.ShortcutKeys = Keys.F5;
            viewMenu.DropDownItems.Add(refreshItem);

            var refreshPluginsItem = new ToolStripMenuItem("Refresh Plugins", null, (s, e) => _pluginsTabControl?.RefreshData()) { ForeColor = DarkText };
            viewMenu.DropDownItems.Add(refreshPluginsItem);

            var refreshDumpsItem = new ToolStripMenuItem("Refresh Dumps", null, (s, e) => _dumpsTabControl?.RefreshData()) { ForeColor = DarkText };
            viewMenu.DropDownItems.Add(refreshDumpsItem);

            viewMenu.DropDownItems.Add(new ToolStripSeparator());

            var clearLogItem = new ToolStripMenuItem("Clear Log", null, OnClearLog) { ForeColor = DarkText };
            viewMenu.DropDownItems.Add(clearLogItem);

            viewMenu.DropDownItems.Add(new ToolStripSeparator());

            var uiGalleryItem = new ToolStripMenuItem("UI Gallery...", null, OnOpenUIGallery) { ForeColor = DarkText };
            uiGalleryItem.ShortcutKeys = Keys.Control | Keys.G;
            viewMenu.DropDownItems.Add(uiGalleryItem);

            var methodInvokerItem = new ToolStripMenuItem("Method Invoker...", null, OnOpenMethodInvoker) { ForeColor = DarkText };
            methodInvokerItem.ShortcutKeys = Keys.Control | Keys.I;
            viewMenu.DropDownItems.Add(methodInvokerItem);

            _menuStrip.Items.Add(viewMenu);

            // Help menu
            var helpMenu = new ToolStripMenuItem("Help") { ForeColor = DarkText };

            var checkUpdatesItem = new ToolStripMenuItem("Check for Updates", null, (s, args) =>
            {
                try { System.Diagnostics.Process.Start("https://halrad.com/mbClouseau/download.html"); }
                catch { }
            }) { ForeColor = DarkText };
            helpMenu.DropDownItems.Add(checkUpdatesItem);
            helpMenu.DropDownItems.Add(new ToolStripSeparator());

            var aboutItem = new ToolStripMenuItem("About mbClouseau", null, OnAbout) { ForeColor = DarkText };
            helpMenu.DropDownItems.Add(aboutItem);

            _menuStrip.Items.Add(helpMenu);
        }

        /// <summary>
        /// Initializes the tab control and pages with dark theme.
        /// </summary>
        private void InitializeTabs()
        {
            _tabControl = new TabControl
            {
                Dock = DockStyle.Fill,
                BackColor = DarkBackground,
                ForeColor = DarkText
            };

            // Create all tabs
            _overviewTab = CreateOverviewTab();
            _eventsTab = CreateEventsTab();
            _metricsTab = CreateMetricsTab();
            _pluginsTab = CreatePluginsTab();
            _dumpsTab = CreateDumpsTab();
            _spyTab = CreateSpyTab();
            _invokeTab = CreateInvokeTab();
            _handlersTab = CreateHandlersTab();

            _tabControl.TabPages.Add(_overviewTab);
            _tabControl.TabPages.Add(_eventsTab);
            _tabControl.TabPages.Add(_metricsTab);
            _tabControl.TabPages.Add(_pluginsTab);
            _tabControl.TabPages.Add(_dumpsTab);
            _tabControl.TabPages.Add(_spyTab);
            _tabControl.TabPages.Add(_invokeTab);
            _tabControl.TabPages.Add(_handlersTab);

            // Apply dark theme to tab control
            _tabControl.DrawMode = TabDrawMode.OwnerDrawFixed;
            _tabControl.DrawItem += OnDrawTabItem;
        }

        /// <summary>
        /// Custom draw for dark-themed tabs.
        /// </summary>
        private void OnDrawTabItem(object sender, DrawItemEventArgs e)
        {
            var tabControl = sender as TabControl;
            var tabPage = tabControl.TabPages[e.Index];
            var tabRect = tabControl.GetTabRect(e.Index);

            var isSelected = e.Index == tabControl.SelectedIndex;
            var backgroundColor = isSelected ? DarkTheme.Accent : DarkPanel;
            var textColor = isSelected ? Color.White : DarkText;

            using (var brush = new SolidBrush(backgroundColor))
            {
                e.Graphics.FillRectangle(brush, tabRect);
            }

            using (var brush = new SolidBrush(textColor))
            {
                var flags = TextFormatFlags.HorizontalCenter | TextFormatFlags.VerticalCenter;
                TextRenderer.DrawText(e.Graphics, tabPage.Text, e.Font, tabRect, textColor, flags);
            }
        }

        /// <summary>
        /// Creates the Overview tab with session info, uptime, and event counts.
        /// </summary>
        private TabPage CreateOverviewTab()
        {
            var tab = new TabPage("Overview")
            {
                BackColor = DarkBackground,
                ForeColor = DarkText
            };

            _overviewTabControl = new OverviewTab(
                _mbApi,
                _stateManager,
                _sessionStartTime,
                () => OnCreateDump(this, EventArgs.Empty),
                () => OnRefreshAll(this, EventArgs.Empty),
                () => OnOpenUIGallery(this, EventArgs.Empty))
            {
                Dock = DockStyle.Fill
            };
            tab.Controls.Add(_overviewTabControl);
            return tab;
        }

        /// <summary>
        /// Creates the Events tab with full log viewer.
        /// </summary>
        private TabPage CreateEventsTab()
        {
            var tab = new TabPage("Events")
            {
                BackColor = DarkBackground
            };

            _eventsTabControl = new EventsTab(5000);
            tab.Controls.Add(_eventsTabControl);

            return tab;
        }

        /// <summary>
        /// Creates the Metrics tab with real-time metrics display.
        /// </summary>
        private TabPage CreateMetricsTab()
        {
            var tab = new TabPage("Metrics")
            {
                BackColor = DarkBackground
            };

            _metricsTabControl = new MetricsTab(
                _metricsCollector,
                _mbApi,
                _sessionStartTime,
                () => _eventsTabControl?.EntryCount ?? 0);
            tab.Controls.Add(_metricsTabControl);

            return tab;
        }

        /// <summary>
        /// Creates the Plugins tab.
        /// </summary>
        private TabPage CreatePluginsTab()
        {
            var tab = new TabPage("Plugins")
            {
                BackColor = DarkBackground
            };

            _pluginsTabControl = new PluginsTab(_pluginDiscovery, _pluginsPath);
            tab.Controls.Add(_pluginsTabControl);

            return tab;
        }

        /// <summary>
        /// Creates the Dumps tab.
        /// </summary>
        private TabPage CreateDumpsTab()
        {
            var tab = new TabPage("Dumps")
            {
                BackColor = DarkBackground
            };

            _dumpsTabControl = new DumpsTab(_stateDumper, _dataFolder);
            tab.Controls.Add(_dumpsTabControl);

            return tab;
        }

        /// <summary>
        /// Creates the Spy tab with UI Inspector (three-pane layout).
        /// </summary>
        private TabPage CreateSpyTab()
        {
            var tab = new TabPage("Spy")
            {
                BackColor = DarkBackground
            };

            _reflectionExplorer = new ReflectionExplorer();
            _spyTabControl = new SpyTab(_reflectionExplorer)
            {
                Dock = DockStyle.Fill
            };
            tab.Controls.Add(_spyTabControl);
            return tab;
        }

        /// <summary>
        /// Creates the Invoke tab with Method Invoker functionality.
        /// </summary>
        private TabPage CreateInvokeTab()
        {
            var tab = new TabPage("Invoke")
            {
                BackColor = DarkBackground
            };

            _invokeTabControl = new InvokeTab(_eventLogger);
            tab.Controls.Add(_invokeTabControl);

            return tab;
        }

        /// <summary>
        /// Opens the Method Invoker form (menu handler).
        /// </summary>
        private void OnOpenMethodInvoker(object sender, EventArgs e)
        {
            try
            {
                var invokerForm = new MethodInvokerForm(_eventLogger);
                invokerForm.Show();
                Logger.Info("Method Invoker form opened");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error opening Method Invoker form");
                MessageBox.Show($"Error opening Method Invoker:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Creates the Handlers tab with Event and Menu inspection.
        /// </summary>
        private TabPage CreateHandlersTab()
        {
            var tab = new TabPage("Handlers")
            {
                BackColor = DarkBackground
            };

            _handlersTabControl = new HandlersTab(_reflectionExplorer);
            tab.Controls.Add(_handlersTabControl);

            return tab;
        }

        /// <summary>
        /// Initializes the status bar with dark theme.
        /// </summary>
        private void InitializeStatusBar()
        {
            _statusStrip = new StatusStrip
            {
                BackColor = DarkPanel,
                ForeColor = DarkText,
                SizingGrip = true
            };

            _sessionDurationLabel = new ToolStripStatusLabel
            {
                Text = "Session: 00:00:00",
                ForeColor = DarkText
            };

            _eventCountLabel = new ToolStripStatusLabel
            {
                Text = "Events: 0",
                ForeColor = DarkText
            };

            var springLabel = new ToolStripStatusLabel
            {
                Spring = true,
                Text = ""
            };

            var versionLabel = new ToolStripStatusLabel
            {
                Text = $"mbClouseau v{typeof(DashboardForm).Assembly.GetName().Version}",
                ForeColor = DarkTheme.TextDim
            };

            _statusStrip.Items.Add(_sessionDurationLabel);
            _statusStrip.Items.Add(new ToolStripSeparator());
            _statusStrip.Items.Add(_eventCountLabel);
            _statusStrip.Items.Add(springLabel);
            _statusStrip.Items.Add(versionLabel);
        }

        /// <summary>
        /// Updates the status bar periodically.
        /// </summary>
        private void OnStatusTimerTick(object sender, EventArgs e)
        {
            var duration = DateTime.Now - _sessionStartTime;
            _sessionDurationLabel.Text = $"Session: {duration:hh\\:mm\\:ss}";
            _eventCountLabel.Text = $"Events: {_eventsTabControl?.EntryCount ?? 0}";

            // Update displays
            _overviewTabControl?.OnTimerTick();
            _metricsTabControl?.OnTimerTick();
        }

        /// <summary>
        /// Adds a log entry to the dashboard log viewer.
        /// </summary>
        public void AddLogEntry(LogEntry entry)
        {
            _eventsTabControl?.AddEntry(entry);
        }

        /// <summary>
        /// Adds a log entry with parameters.
        /// </summary>
        public void AddLogEntry(string category, string level, string eventType, string details)
        {
            _eventsTabControl?.AddEntry(category, level, eventType, details);
        }

        #region Event Handlers

        private void OnExportLog(object sender, EventArgs e)
        {
            // Trigger the log viewer's export
            _tabControl.SelectedTab = _eventsTab;
        }

        private void OnCreateDump(object sender, EventArgs e)
        {
            try
            {
                Cursor = Cursors.WaitCursor;
                var path = _stateDumper.DumpState();
                _dumpsTabControl?.RefreshData();
                Cursor = Cursors.Default;

                var result = MessageBox.Show(
                    $"State dump created:\n{path}\n\nOpen in editor?",
                    "Dump Created",
                    MessageBoxButtons.YesNo,
                    MessageBoxIcon.Information);

                if (result == DialogResult.Yes)
                {
                    Process.Start("notepad.exe", path);
                }
            }
            catch (Exception ex)
            {
                Cursor = Cursors.Default;
                Logger.Error(ex, "Error creating state dump");
                MessageBox.Show($"Error creating dump:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void OnRefreshAll(object sender, EventArgs e)
        {
            _pluginsTabControl?.RefreshData();
            _dumpsTabControl?.RefreshData();
            _overviewTabControl?.RefreshData();
            _metricsTabControl?.RefreshData();
        }

        private void OnClearLog(object sender, EventArgs e)
        {
            // The log viewer handles its own clear
        }

        private void OnAbout(object sender, EventArgs e)
        {
            var aboutForm = new Form
            {
                Text = "About mbClouseau",
                Size = new Size(430, 450),
                FormBorderStyle = FormBorderStyle.FixedDialog,
                StartPosition = FormStartPosition.CenterParent,
                MaximizeBox = false,
                MinimizeBox = false,
                BackColor = Color.White
            };

            var titleLabel = new Label
            {
                Text = "mbClouseau",
                Location = new Point(20, 20),
                AutoSize = true,
                Font = new Font("Segoe UI", 16F, FontStyle.Bold),
                ForeColor = Color.FromArgb(0, 102, 204)
            };
            aboutForm.Controls.Add(titleLabel);

            var version = typeof(DashboardForm).Assembly.GetName().Version;
            var infoLabel = new Label
            {
                Text = $"Version: {version}\n\n" +
                       "Uncovering clues with MusicBee Clouseau\n\n" +
                       "A trace/debug utility for MusicBee plugin developers.\n\n" +
                       "by HALRAD 2026",
                Location = new Point(20, 90),
                Size = new Size(290, 250),
                Font = new Font("Segoe UI", 8F),
                ForeColor = Color.Black
            };
            aboutForm.Controls.Add(infoLabel);

            // Logo in upper right - add after other controls so it's positioned correctly
            var logoBox = new PictureBox
            {
                Size = new Size(80, 80),
                Location = new Point(310, 15),
                SizeMode = PictureBoxSizeMode.Zoom,
                BackColor = Color.Transparent
            };
            try
            {
                var assembly = System.Reflection.Assembly.GetExecutingAssembly();
                using (var stream = assembly.GetManifestResourceStream("HALRAD_mb_Clouseau.png"))
                {
                    if (stream != null)
                        logoBox.Image = Image.FromStream(stream);
                }
            }
            catch { }
            aboutForm.Controls.Add(logoBox);
            logoBox.BringToFront();

            var updatesLink = new LinkLabel
            {
                Text = "Check for updates",
                Location = new Point(20, 335),
                AutoSize = true,
                Font = new Font("Segoe UI", 9F),
                LinkColor = Color.FromArgb(0, 102, 204)
            };
            updatesLink.LinkClicked += (s, args) =>
            {
                try { System.Diagnostics.Process.Start("https://halrad.com/mbClouseau/download.html"); }
                catch { }
            };
            aboutForm.Controls.Add(updatesLink);

            var okButton = new Button
            {
                Text = "OK",
                Size = new Size(80, 28),
                Location = new Point(310, 335),
                DialogResult = DialogResult.OK
            };
            aboutForm.Controls.Add(okButton);
            aboutForm.AcceptButton = okButton;

            aboutForm.ShowDialog(this);
        }

        private void OnOpenUIGallery(object sender, EventArgs e)
        {
            try
            {
                // Check if gallery is already open
                foreach (Form form in Application.OpenForms)
                {
                    if (form is UIGalleryForm existingGallery && !form.IsDisposed)
                    {
                        existingGallery.BringToFront();
                        existingGallery.Focus();
                        return;
                    }
                }

                // Open new UI Gallery
                var gallery = new UIGalleryForm(
                    _mbApi,
                    _dataFolder,
                    _pluginsPath,
                    _settings,
                    _stateManager,
                    _metricsCollector,
                    _eventLogger);
                gallery.Show();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error opening UI Gallery");
                MessageBox.Show($"Error opening UI Gallery:\n{ex.Message}", "Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        #endregion

        /// <summary>
        /// Cleanup on form close.
        /// </summary>
        protected override void OnFormClosing(FormClosingEventArgs e)
        {
            // Save window position/size
            SaveWindowSettings();

            _statusTimer?.Stop();
            _statusTimer?.Dispose();

            if (_eventLogger != null)
            {
                _eventLogger.LogEntryAdded -= OnEventLoggerEntryAdded;
            }

            base.OnFormClosing(e);
        }

        #region Window Settings Persistence

        /// <summary>
        /// Load window position and size from settings.
        /// </summary>
        private void LoadWindowSettings()
        {
            // Ensure Dashboard settings exist (may be null if loaded from old settings.json)
            if (_settings?.UI != null && _settings.UI.Dashboard == null)
            {
                _settings.UI.Dashboard = new DashboardWindowSettings();
            }

            var dashSettings = _settings?.UI?.Dashboard;
            if (dashSettings == null)
            {
                // Use defaults
                Size = new Size(1920, 900);
                StartPosition = FormStartPosition.CenterScreen;
                return;
            }

            // Restore maximized state
            if (dashSettings.Maximized)
            {
                // Set size first for when user un-maximizes
                Size = new Size(dashSettings.Width, dashSettings.Height);
                StartPosition = FormStartPosition.CenterScreen;
                WindowState = FormWindowState.Maximized;
                return;
            }

            // Restore size
            int width = dashSettings.Width > 0 ? dashSettings.Width : 1920;
            int height = dashSettings.Height > 0 ? dashSettings.Height : 900;
            Size = new Size(width, height);

            // Restore position if valid (X=-1 AND Y=-1 means "not set", use center)
            // Note: negative coordinates are valid on multi-monitor setups
            bool hasPosition = !(dashSettings.X == -1 && dashSettings.Y == -1);

            if (hasPosition)
            {
                // Validate position is on a visible screen
                var savedBounds = new Rectangle(dashSettings.X, dashSettings.Y, width, height);
                bool onScreen = false;
                foreach (var screen in Screen.AllScreens)
                {
                    if (screen.WorkingArea.IntersectsWith(savedBounds))
                    {
                        onScreen = true;
                        break;
                    }
                }

                if (onScreen)
                {
                    StartPosition = FormStartPosition.Manual;
                    Location = new Point(dashSettings.X, dashSettings.Y);
                    Logger.Debug($"Restored dashboard position: {dashSettings.X},{dashSettings.Y}");
                }
                else
                {
                    StartPosition = FormStartPosition.CenterScreen;
                    Logger.Debug($"Saved position {dashSettings.X},{dashSettings.Y} is off-screen, centering");
                }
            }
            else
            {
                StartPosition = FormStartPosition.CenterScreen;
                Logger.Debug("No saved position, centering dashboard");
            }
        }

        /// <summary>
        /// Save window position and size to settings.
        /// </summary>
        private void SaveWindowSettings()
        {
            if (_settings == null)
            {
                Logger.Warn("Cannot save dashboard settings: _settings is null");
                return;
            }

            try
            {
                // Ensure UI and Dashboard settings exist
                if (_settings.UI == null)
                {
                    _settings.UI = new UISettings();
                }
                if (_settings.UI.Dashboard == null)
                {
                    _settings.UI.Dashboard = new DashboardWindowSettings();
                }

                var dashSettings = _settings.UI.Dashboard;

                // Save maximized state
                dashSettings.Maximized = WindowState == FormWindowState.Maximized;

                // Save normal bounds (not maximized bounds)
                if (WindowState == FormWindowState.Normal)
                {
                    dashSettings.X = Location.X;
                    dashSettings.Y = Location.Y;
                    dashSettings.Width = Size.Width;
                    dashSettings.Height = Size.Height;
                }
                // If maximized, keep the previous normal size

                // Persist to file
                _settings.Save(_dataFolder);
                Logger.Info($"Dashboard window settings saved: X={dashSettings.X}, Y={dashSettings.Y}, {dashSettings.Width}x{dashSettings.Height}, Maximized={dashSettings.Maximized}");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to save dashboard window settings");
            }
        }

        #endregion
    }
}
