using System;

namespace MusicBeePlugin.Clouseau.Introspection
{
    /// <summary>
    /// Data model representing a discovered MusicBee plugin.
    /// Contains assembly information, file metadata, and validation status.
    /// </summary>
    public class DiscoveredPlugin
    {
        /// <summary>
        /// Full path to the plugin DLL file.
        /// </summary>
        public string FilePath { get; set; }

        /// <summary>
        /// File name of the plugin DLL (without path).
        /// </summary>
        public string FileName { get; set; }

        /// <summary>
        /// Assembly name from the loaded assembly metadata.
        /// </summary>
        public string AssemblyName { get; set; }

        /// <summary>
        /// Assembly version (e.g., "1.0.0.0").
        /// </summary>
        public string Version { get; set; }

        /// <summary>
        /// File version from assembly attributes.
        /// </summary>
        public string FileVersion { get; set; }

        /// <summary>
        /// Last modification date of the plugin file.
        /// </summary>
        public DateTime LastModified { get; set; }

        /// <summary>
        /// Size of the plugin file in bytes.
        /// </summary>
        public long FileSizeBytes { get; set; }

        /// <summary>
        /// Whether this DLL appears to be a valid MusicBee plugin
        /// (has Initialise method matching MusicBee pattern).
        /// </summary>
        public bool IsValidPlugin { get; set; }

        /// <summary>
        /// Full type name of the plugin class (e.g., "MusicBeePlugin.Plugin").
        /// </summary>
        public string PluginTypeName { get; set; }

        /// <summary>
        /// Error message if the plugin failed to load or validate.
        /// Null if no errors occurred.
        /// </summary>
        public string LoadError { get; set; }

        /// <summary>
        /// Plugin name extracted from PluginInfo if available.
        /// </summary>
        public string PluginName { get; set; }

        /// <summary>
        /// Plugin description extracted from PluginInfo if available.
        /// </summary>
        public string PluginDescription { get; set; }

        /// <summary>
        /// Plugin author extracted from PluginInfo if available.
        /// </summary>
        public string PluginAuthor { get; set; }

        /// <summary>
        /// Plugin type (General, LyricsRetrieval, ArtworkRetrieval, etc.).
        /// </summary>
        public string PluginType { get; set; }

        /// <summary>
        /// Major version number from PluginInfo.
        /// </summary>
        public short VersionMajor { get; set; }

        /// <summary>
        /// Minor version number from PluginInfo.
        /// </summary>
        public short VersionMinor { get; set; }

        /// <summary>
        /// Revision number from PluginInfo.
        /// </summary>
        public short Revision { get; set; }

        /// <summary>
        /// Returns a string representation of the plugin information.
        /// </summary>
        public override string ToString()
        {
            if (IsValidPlugin)
            {
                return $"{PluginName ?? FileName} v{VersionMajor}.{VersionMinor}.{Revision} ({PluginType})";
            }
            return $"{FileName} [Invalid: {LoadError ?? "Unknown error"}]";
        }

        /// <summary>
        /// Creates a formatted display string suitable for UI display.
        /// </summary>
        public string ToDisplayString()
        {
            if (!IsValidPlugin)
            {
                return $"[INVALID] {FileName}\n  Error: {LoadError ?? "Not a valid MusicBee plugin"}";
            }

            var display = $"{PluginName ?? FileName}";
            if (!string.IsNullOrEmpty(PluginAuthor))
            {
                display += $" by {PluginAuthor}";
            }
            display += $"\n  Version: {VersionMajor}.{VersionMinor}.{Revision}";
            display += $"\n  Type: {PluginType}";
            display += $"\n  File: {FileName} ({FormatFileSize(FileSizeBytes)})";
            display += $"\n  Modified: {LastModified:yyyy-MM-dd HH:mm:ss}";

            if (!string.IsNullOrEmpty(PluginDescription))
            {
                display += $"\n  Description: {PluginDescription}";
            }

            return display;
        }

        /// <summary>
        /// Formats file size in human-readable format.
        /// </summary>
        private static string FormatFileSize(long bytes)
        {
            string[] sizes = { "B", "KB", "MB", "GB" };
            double len = bytes;
            int order = 0;
            while (len >= 1024 && order < sizes.Length - 1)
            {
                order++;
                len = len / 1024;
            }
            return $"{len:0.##} {sizes[order]}";
        }
    }
}
