using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.Serialization.Json;
using System.Runtime.Serialization;
using System.Text;

namespace MusicBeePlugin
{
    [DataContract]
    public class ShuffleState
    {
        [DataMember(Name = "playedTracks")]
        public List<string> PlayedTracksList { get; set; } = new List<string>();

        [IgnoreDataMember]
        public HashSet<string> PlayedTracks { get; private set; } = new HashSet<string>();

        [DataMember(Name = "libraryHash")]
        public string LibraryHash { get; set; }

        [DataMember(Name = "totalTracks")]
        public int TotalTracks { get; set; }

        [DataMember(Name = "lastReset")]
        public DateTime LastReset { get; set; }

        [DataMember(Name = "lastPlayed")]
        public DateTime LastPlayed { get; set; }

        public int PlayedCount => PlayedTracks.Count;
        public int RemainingCount => Math.Max(0, TotalTracks - PlayedCount);
        public double Progress => TotalTracks > 0 ? (double)PlayedCount / TotalTracks : 0;

        public ShuffleState()
        {
            LastReset = DateTime.UtcNow;
        }

        public void MarkPlayed(string trackUrl)
        {
            if (string.IsNullOrEmpty(trackUrl)) return;

            PlayedTracks.Add(trackUrl);
            LastPlayed = DateTime.UtcNow;
        }

        public bool IsPlayed(string trackUrl)
        {
            return PlayedTracks.Contains(trackUrl);
        }

        public void Reset()
        {
            PlayedTracks.Clear();
            PlayedTracksList.Clear();
            LastReset = DateTime.UtcNow;
        }

        public void Save(string filePath)
        {
            try
            {
                // Sync HashSet to List for serialization
                PlayedTracksList = new List<string>(PlayedTracks);

                var directory = Path.GetDirectoryName(filePath);
                if (!string.IsNullOrEmpty(directory) && !Directory.Exists(directory))
                {
                    Directory.CreateDirectory(directory);
                }

                var serializer = new DataContractJsonSerializer(typeof(ShuffleState));
                using (var stream = new FileStream(filePath, FileMode.Create))
                {
                    serializer.WriteObject(stream, this);
                }
            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine($"Failed to save shuffle state: {ex.Message}");
            }
        }

        public static ShuffleState Load(string filePath)
        {
            try
            {
                if (!File.Exists(filePath))
                {
                    return new ShuffleState();
                }

                var serializer = new DataContractJsonSerializer(typeof(ShuffleState));
                using (var stream = new FileStream(filePath, FileMode.Open))
                {
                    var state = (ShuffleState)serializer.ReadObject(stream);

                    // Sync List to HashSet after deserialization
                    state.PlayedTracks = new HashSet<string>(state.PlayedTracksList ?? new List<string>());

                    return state;
                }
            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine($"Failed to load shuffle state: {ex.Message}");
                return new ShuffleState();
            }
        }
    }
}
