using System;
using System.Collections.Generic;
using System.Linq;

namespace MusicBeePlugin
{
    public class ShuffleEngine
    {
        private readonly ShuffleState _state;
        private readonly Random _random;
        private string[] _libraryCache;

        public ShuffleEngine(ShuffleState state)
        {
            _state = state ?? throw new ArgumentNullException(nameof(state));
            _random = new Random();
        }

        public ShuffleState State => _state;

        public void UpdateLibrary(string[] library)
        {
            _libraryCache = library;
            _state.TotalTracks = library?.Length ?? 0;

            // Compute simple hash for change detection
            _state.LibraryHash = ComputeLibraryHash(library);
        }

        public void MarkPlayed(string trackUrl)
        {
            _state.MarkPlayed(trackUrl);
        }

        public bool IsPlayed(string trackUrl)
        {
            return _state.IsPlayed(trackUrl);
        }

        public string GetNextUnplayedTrack()
        {
            if (_libraryCache == null || _libraryCache.Length == 0)
                return null;

            var unplayed = _libraryCache.Where(t => !_state.IsPlayed(t)).ToList();

            if (unplayed.Count == 0)
            {
                // All tracks played - full cycle complete
                return null;
            }

            // Pick a random unplayed track
            int index = _random.Next(unplayed.Count);
            return unplayed[index];
        }

        public string[] GetUnplayedTracks()
        {
            if (_libraryCache == null)
                return Array.Empty<string>();

            return _libraryCache.Where(t => !_state.IsPlayed(t)).ToArray();
        }

        public string[] GetPlayedTracks()
        {
            return _state.PlayedTracks.ToArray();
        }

        public void Reset()
        {
            _state.Reset();
        }

        public double GetProgress()
        {
            return _state.Progress;
        }

        public int GetPlayedCount()
        {
            return _state.PlayedCount;
        }

        public int GetTotalCount()
        {
            return _state.TotalTracks;
        }

        public int GetRemainingCount()
        {
            return _state.RemainingCount;
        }

        public bool IsComplete()
        {
            return _state.TotalTracks > 0 && _state.PlayedCount >= _state.TotalTracks;
        }

        private string ComputeLibraryHash(string[] library)
        {
            if (library == null || library.Length == 0)
                return string.Empty;

            // Simple hash based on count and first/last items
            int hash = library.Length;
            if (library.Length > 0)
                hash ^= library[0].GetHashCode();
            if (library.Length > 1)
                hash ^= library[library.Length - 1].GetHashCode();

            return hash.ToString("X8");
        }
    }
}
