# Master Shuffle - Technical Specification

![MusicBeeRADomizer Logo](mb_radomizer_512.png)

**Part of the HALRAD MusicBee Suite:**
- [MBXRemote](https://halrad.com/MBXRemote/) - Windows desktop remote client
- [MusicBeeChromecast2026](https://halrad.com/MBChromecast-2026/) - Chromecast streaming
- [MusicBee TrueShuffle](https://halrad.com/MBTrueShuffle/) - True shuffle plugin

---

## Background

The matter of people experiencing unexpected results when playing their music 'randomly' is a continuing and recurring topic of confusion - not only regarding MusicBee, but in general.  

### Core Issues

- Random ≠ "plays everything once before repeating"
- Users expect shuffle = deck of cards (no repeats until reshuffled)
- Most players reset state on restart
- True randomness feels wrong to humans (we notice patterns)

## Architecture

```
MusicBeeMasterShuffle/
├── Plugin.cs              # Main entry point, MusicBee integration
├── ShuffleState.cs        # Persisted state management
├── ShuffleEngine.cs       # Shuffle logic and track selection
├── ConfigPanel.cs         # Settings UI panel
└── MusicBeeInterface.cs   # MusicBee API definitions (from SDK)
```

## State Model

```
┌─────────────────────────────────────────┐
│         SHUFFLE STATE (persisted)       │
├─────────────────────────────────────────┤
│  Library Hash: a3f8b2c1                 │
│  Total Tracks: 12,847                   │
│  Played: 3,421                          │
│  Remaining: 9,426                       │
│                                         │
│  Played Set: [track_id, track_id, ...]  │
│  Last Played: 2025-12-28T21:45:00       │
│  Seed: 847291 (for reproducible order)  │
└─────────────────────────────────────────┘
```

## MusicBee API Methods Used

| Method                                   | Purpose                               |
| ---------------------------------------- | ------------------------------------- |
| `Library_QueryFilesEx(query, out files)` | Get all library tracks                |
| `NowPlaying_GetFileUrl()`                | Current playing track URL (unique ID) |
| `NowPlaying_GetDuration()`               | Track duration in ms                  |
| `Player_GetPosition()`                   | Current playback position in ms       |
| `NotificationType.TrackChanged`          | Event when track changes              |
| `FilePropertyType.PlayCount`             | Built-in play count                   |
| `FilePropertyType.LastPlayed`            | Last played timestamp                 |
| `NowPlayingList_PlayNow(fileUrl)`        | Force play a specific track           |
| `NowPlayingList_QueueNext(fileUrl)`      | Queue next track                      |
| `NowPlayingList_Clear()`                 | Clear now playing list                |
| `Player_SetShuffle(bool)`                | Control shuffle state                 |
| `Setting_GetPersistentStoragePath()`     | Plugin data folder                    |
| `MB_AddMenuItem()`                       | Add menu items                        |

## Implementation Options

| Approach         | Pros                      | Cons                     |
| ---------------- | ------------------------- | ------------------------ |
| Plugin           | Non-invasive, user choice | Limited API access       |
| SQLite DB        | Fast lookups, portable    | Extra dependency         |
| Simple JSON file | Easy, human-readable      | Slow for large libraries |
| Bloom filter     | Memory efficient          | False positives possible |

**Chosen approach:** JSON file with HashSet in memory

## Design Decisions

| Decision                 | Options                      | Chosen                           |
| ------------------------ | ---------------------------- | -------------------------------- |
| What counts as "played"? | >30sec, >50%, full track     | User configurable, default 50%   |
| Scope                    | Whole library, per-playlist  | Global first, per-playlist later |
| Library changes          | Re-scan on startup           | Detect adds/removes on startup   |
| UI                       | Config panel, menu, dockable | Dockable panel + config          |
| Reset                    | Manual only, auto-reset      | Manual with optional auto        |

## Core Classes

### ShuffleState

```csharp
public class ShuffleState
{
    public HashSet<string> PlayedTracks { get; set; } = new();
    public string LibraryHash { get; set; }
    public int TotalTracks { get; set; }
    public DateTime LastReset { get; set; }
    public DateTime LastPlayed { get; set; }

    public void Save(string path);
    public static ShuffleState Load(string path);
}
```

### ShuffleEngine

```csharp
public class ShuffleEngine
{
    private ShuffleState _state;
    private Random _random;

    public void MarkPlayed(string trackUrl);
    public string GetNextUnplayedTrack(string[] library);
    public void Reset();
    public double GetProgress();
}
```

### Play Threshold Logic

```csharp
// In ReceiveNotification TrackChanged handler:
// Check if PREVIOUS track was played long enough
if (_lastTrackUrl != null && _lastTrackDuration > 0)
{
    int playedMs = _lastTrackPosition;
    double playedPercent = (double)playedMs / _lastTrackDuration;

    if (playedPercent >= _playThreshold)
    {
        _engine.MarkPlayed(_lastTrackUrl);
    }
}

// Store current track info for next change
_lastTrackUrl = mbApiInterface.NowPlaying_GetFileUrl();
_lastTrackDuration = mbApiInterface.NowPlaying_GetDuration();
```

## UI Panel

```
┌─────────────────────────────────────────┐
│  Master Shuffle                         │
│  ════════════════════════════           │
│  Progress: ████████░░░░░░ 3,421/12,847  │
│  26.6% complete                         │
│                                         │
│  [Reset] [View Played] [Settings]       │
└─────────────────────────────────────────┘
```

## Configuration Options

```csharp
public class ShuffleSettings
{
    public bool Enabled { get; set; } = true;
    public double PlayThreshold { get; set; } = 0.5;  // 50%
    public bool AutoReset { get; set; } = false;
    public bool ShowPanel { get; set; } = true;
}
```

## Storage

State file location: `{MusicBee Data}\mb_MasterShuffle\state.json`

```json
{
  "playedTracks": ["file://path/to/song1.mp3", "file://path/to/song2.mp3"],
  "libraryHash": "a3f8b2c1",
  "totalTracks": 12847,
  "lastReset": "2025-12-01T00:00:00Z",
  "lastPlayed": "2025-12-28T21:45:00Z"
}
```

## Complicating Factors

1. **Library changes** - New songs added, songs removed
   
   - Solution: Recalculate on startup, new tracks are automatically unplayed

2. **Multiple playlists** - Per-playlist tracking or global?
   
   - Solution: Start global, add per-playlist as enhancement

3. **Partial plays** - Does skipping count as "played"?
   
   - Solution: Configurable threshold

4. **Storage size** - 100k track IDs = ~2MB
   
   - Solution: Acceptable for modern systems

## Building

```bash
# Debug build
dotnet build -c Debug

# Release build
dotnet build -c Release
```

Output: `bin/{Configuration}/net48/mb_MasterShuffle.dll`

## Testing

1. Copy DLL to MusicBee Plugins folder
2. Restart MusicBee
3. Enable plugin in Preferences
4. Play tracks and verify state persistence
5. Restart MusicBee and verify state restored
