using System;
using System.Drawing;
using System.Windows.Forms;
using System.IO;
using System.Reflection;

namespace MusicBeePlugin
{
    public partial class Plugin
    {
        private MusicBeeApiInterface _mbApi;
        private PluginInfo _about = new PluginInfo();

        private ShuffleEngine _engine;
        private ShuffleSettings _settings;
        private string _dataPath;
        private string _stateFilePath;
        private string _settingsFilePath;

        // Track state for detecting when a track was actually played
        private string _lastTrackUrl;
        private int _lastTrackDuration;
        private int _lastTrackPosition;
        private System.Windows.Forms.Timer _positionTimer;

        public PluginInfo Initialise(IntPtr apiInterfacePtr)
        {
            _mbApi = new MusicBeeApiInterface();
            _mbApi.Initialise(apiInterfacePtr);

            _about.PluginInfoVersion = PluginInfoVersion;
            _about.Name = "TrueShuffle";
            _about.Description = "Plays tracks exactly once before any repeats";
            _about.Author = "Halrad";
            _about.TargetApplication = "TrueShuffle";
            _about.Type = PluginType.PanelView;

            var version = Assembly.GetExecutingAssembly().GetName().Version;
            _about.VersionMajor = (short)version.Major;
            _about.VersionMinor = (short)version.Minor;
            _about.Revision = (short)version.Build;  // 1.0.25.1229 format
            _about.MinInterfaceVersion = MinInterfaceVersion;
            _about.MinApiRevision = MinApiRevision;
            _about.ReceiveNotifications = ReceiveNotificationFlags.PlayerEvents;
            _about.ConfigurationPanelHeight = 100;

            return _about;
        }

        private void InitializePlugin()
        {
            // Set up data paths
            _dataPath = Path.Combine(_mbApi.Setting_GetPersistentStoragePath(), "mb_trueshuffle");
            _stateFilePath = Path.Combine(_dataPath, "state.json");
            _settingsFilePath = Path.Combine(_dataPath, "settings.json");

            // Create directory if needed
            if (!Directory.Exists(_dataPath))
                Directory.CreateDirectory(_dataPath);

            // Load settings and state
            _settings = ShuffleSettings.Load(_settingsFilePath);
            var state = ShuffleState.Load(_stateFilePath);
            _engine = new ShuffleEngine(state);

            // Load library
            RefreshLibrary();

            // Add menu item
            _mbApi.MB_AddMenuItem("mnuTools/TrueShuffle: Reset", "Reset TrueShuffle", OnResetClicked);

            // Timer to track playback position
            _positionTimer = new System.Windows.Forms.Timer { Interval = 1000 };
            _positionTimer.Tick += (s, e) =>
            {
                if (_mbApi.Player_GetPlayState() == PlayState.Playing)
                    _lastTrackPosition = _mbApi.Player_GetPosition();
            };
            _positionTimer.Start();
        }

        private void RefreshLibrary()
        {
            try
            {
                string[] files;
                if (_mbApi.Library_QueryFilesEx("", out files) && files != null)
                {
                    _engine.UpdateLibrary(files);
                }
            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine($"Failed to refresh library: {ex.Message}");
            }
        }

        public bool Configure(IntPtr panelHandle)
        {
            if (panelHandle != IntPtr.Zero)
            {
                Panel configPanel = (Panel)Panel.FromHandle(panelHandle);

                // Play threshold
                var lblThreshold = new Label
                {
                    Text = "Count as played after:",
                    Location = new Point(10, 12),
                    AutoSize = true
                };

                var cmbThreshold = new ComboBox
                {
                    Location = new Point(245, 8),
                    Width = 140,
                    DropDownStyle = ComboBoxStyle.DropDownList
                };
                cmbThreshold.Items.AddRange(new object[] { "30 seconds", "50%", "75%", "Full track" });
                cmbThreshold.SelectedIndex = GetThresholdIndex(_settings?.PlayThreshold ?? 0.5);
                cmbThreshold.SelectedIndexChanged += (s, e) =>
                {
                    if (_settings != null)
                        _settings.PlayThreshold = GetThresholdValue(cmbThreshold.SelectedIndex);
                };

                // Auto-reset checkbox
                var chkAutoReset = new CheckBox
                {
                    Text = "Auto-reset after complete cycle",
                    Checked = _settings?.AutoReset ?? false,
                    Location = new Point(10, 45),
                    AutoSize = true
                };
                chkAutoReset.CheckedChanged += (s, e) =>
                {
                    if (_settings != null) _settings.AutoReset = chkAutoReset.Checked;
                };

                // Reset button - positioned at right side, flat style
                var btnReset = new Button
                {
                    Text = "Reset",
                    Location = new Point(660, 8),
                    Size = new Size(110, 35),
                    FlatStyle = FlatStyle.Flat
                };
                btnReset.FlatAppearance.BorderSize = 0;
                btnReset.Click += (s, e) => OnResetClicked(s, e);

                configPanel.Controls.AddRange(new Control[] { lblThreshold, cmbThreshold, chkAutoReset, btnReset });
            }

            return false;
        }

        private int GetThresholdIndex(double threshold)
        {
            if (threshold < 0.1) return 0;      // 30 seconds (will use time-based)
            if (threshold < 0.6) return 1;      // 50%
            if (threshold < 0.9) return 2;      // 75%
            return 3;                            // Full track
        }

        private double GetThresholdValue(int index)
        {
            switch (index)
            {
                case 0: return 0.0;   // 30 seconds (special case)
                case 1: return 0.5;   // 50%
                case 2: return 0.75;  // 75%
                case 3: return 1.0;   // Full track
                default: return 0.5;
            }
        }

        public void SaveSettings()
        {
            _settings?.Save(_settingsFilePath);
        }

        public void Close(PluginCloseReason reason)
        {
            // Stop timer
            _positionTimer?.Stop();
            _positionTimer?.Dispose();

            // Save state on close
            _engine?.State?.Save(_stateFilePath);
            _settings?.Save(_settingsFilePath);
        }

        public void Uninstall()
        {
            // Clean up persisted files
            try
            {
                if (Directory.Exists(_dataPath))
                {
                    Directory.Delete(_dataPath, true);
                }
            }
            catch { }
        }

        public void ReceiveNotification(string sourceFileUrl, NotificationType type)
        {
            switch (type)
            {
                case NotificationType.PluginStartup:
                    InitializePlugin();
                    break;

                case NotificationType.TrackChanged:
                    OnTrackChanged();
                    break;

                case NotificationType.PlayStateChanged:
                    // Update position when play state changes (for tracking partial plays)
                    _lastTrackPosition = _mbApi.Player_GetPosition();
                    break;
            }
        }

        private void OnTrackChanged()
        {
            if (_engine == null || _settings == null)
                return;

            // Check if PREVIOUS track was played long enough
            if (!string.IsNullOrEmpty(_lastTrackUrl) && _lastTrackDuration > 0)
            {
                bool shouldCount = false;

                if (_settings.PlayThreshold < 0.1)
                {
                    // Time-based: 30 seconds
                    shouldCount = _lastTrackPosition >= 30000;
                }
                else
                {
                    // Percentage-based
                    double playedPercent = (double)_lastTrackPosition / _lastTrackDuration;
                    shouldCount = playedPercent >= _settings.PlayThreshold;
                }

                if (shouldCount && !_engine.IsPlayed(_lastTrackUrl))
                {
                    _engine.MarkPlayed(_lastTrackUrl);
                    _engine.State.Save(_stateFilePath);

                    // Check for cycle complete
                    if (_engine.IsComplete())
                    {
                        if (_settings.AutoReset)
                        {
                            _engine.Reset();
                            _engine.State.Save(_stateFilePath);
                        }
                    }
                }
            }

            // Store current track info for next change
            _lastTrackUrl = _mbApi.NowPlaying_GetFileUrl();
            _lastTrackDuration = _mbApi.NowPlaying_GetDuration();
            _lastTrackPosition = 0;
        }

        private void OnResetClicked(object sender, EventArgs e)
        {
            if (_engine == null) return;

            var result = MessageBox.Show(
                $"Reset TrueShuffle?\n\nThis will clear {_engine.GetPlayedCount()} played tracks and start fresh.",
                "TrueShuffle",
                MessageBoxButtons.YesNo,
                MessageBoxIcon.Question);

            if (result == DialogResult.Yes)
            {
                _engine.Reset();
                _engine.State.Save(_stateFilePath);
                MessageBox.Show(
                    "TrueShuffle has been reset.\nAll tracks are now unplayed.",
                    "TrueShuffle",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Information);
            }
        }

        // Dockable panel support
        public int OnDockablePanelCreated(Control panel)
        {
            // Ensure plugin is initialized before creating panel
            if (_engine == null)
            {
                InitializePlugin();
            }

            float dpiScaling;
            using (Graphics g = panel.CreateGraphics())
            {
                dpiScaling = g.DpiY / 96f;
            }

            var progressPanel = new ProgressPanel(_engine);
            progressPanel.Dock = DockStyle.Fill;
            panel.Controls.Add(progressPanel);

            return Convert.ToInt32(95 * dpiScaling);
        }
    }

    // Simple progress display panel
    public class ProgressPanel : UserControl
    {
        private readonly ShuffleEngine _engine;
        private ProgressBar _progressBar;
        private Label _lblStats;
        private Timer _refreshTimer;

        public ProgressPanel(ShuffleEngine engine)
        {
            _engine = engine;
            InitializeComponents();
        }

        private void InitializeComponents()
        {
            this.BackColor = Color.Transparent;

            _progressBar = new ProgressBar
            {
                Location = new Point(0, 10),
                Size = new Size(300, 54),
                Minimum = 0,
                Maximum = 100,
                Anchor = AnchorStyles.Top | AnchorStyles.Left | AnchorStyles.Right
            };

            _lblStats = new Label
            {
                Text = "0 / 0 (0%)",
                Location = new Point(5, 68),
                AutoSize = true,
                ForeColor = SystemColors.ControlText
            };

            this.Controls.AddRange(new Control[] { _progressBar, _lblStats });

            // Refresh timer
            _refreshTimer = new Timer { Interval = 5000 };
            _refreshTimer.Tick += (s, e) => UpdateDisplay();
            _refreshTimer.Start();

            UpdateDisplay();
        }

        private void UpdateDisplay()
        {
            if (_engine == null) return;

            int played = _engine.GetPlayedCount();
            int total = _engine.GetTotalCount();
            double progress = _engine.GetProgress();

            _progressBar.Value = Math.Min(100, (int)(progress * 100));
            _lblStats.Text = $"{played:N0} / {total:N0} ({progress:P1})";
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                _refreshTimer?.Stop();
                _refreshTimer?.Dispose();
            }
            base.Dispose(disposing);
        }
    }
}
